#Include This Once
#Include Once "C:\HLib3\Array\LnArr.inc"
#Include Once "C:\HLib3\Array\SsArr.inc"

Macro OjArrTag = 1734344665
Type OjArr
    tag As Long
    arr As LnArr Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Object Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = OjArrNew() 'get handle for new container
    '       h = OjArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

'++
Declare Function OjArrCompareCB(a As IUnknown, b As IUnknown) As Long
    'compare callback function
    '   a < b : return < 0
    '   a = b : return = 0
    '   a > b : return > 0

Declare Function OjArrStoreCB(o As IUnknown) As String
    'callback function to pack object into string, used by OjArrStore() and OjArrFileStore()

Declare Function OjArrRestoreCB(ByVal s As String) As IUnknown
    'callback function to restore object from string, used by OjArrRestore() and OjArrFileStore()
'--

Function OjArrNew() As Long
    'allocate new container - return handle
    Local p As OjArr Ptr
    Err = 0
    p = MemAlloc(SizeOf(OjArr))
    ExitF(p=0, LibErrM)
    @p.tag = OjArrTag
    @p.arr = LnArrNew() : If Err Then Exit Function
    Function = p
End Function

Function OjArrFinal(ByVal p As OjArr Ptr) As Long
    'free allocated container - return null
    Local i As Long
    If p Then
        ExitF(@p.tag<>OjArrTag, LibErrH)
        OjArrClear p
        @p.arr = LnArrFinal(@p.arr)
        MemFree(p)
    End If
End Function

Function OjArrValidate(ByVal p As OjArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = OjArrTag Then Function = @p.tag
End Function

Sub OjArrClear(ByVal p As OjArr Ptr)
    'delete all data
    Local i As Long
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    For i = 1 To @p.@arr.count
        @p.@arr.@arr[i] = MemFreeObj(@p.@arr.@arr[i])
    Next i
    LnArrClear @p.arr
End Sub

Function OjArrCount(ByVal p As OjArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    Function = @p.@arr.count
End Function

Sub OjArrReDim(ByVal p As OjArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local i, items As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        OjArrClear p
    ElseIf Count <> @p.@arr.count Then
        items = @p.@arr.count
        For i = Count + 1 To items
            @p.@arr.@arr[i] = MemFreeObj(@p.@arr.@arr[i])
        Next i
        LnArrReDim @p.arr, Count
    End If
End Sub

Function OjArrGet(ByVal p As OjArr Ptr, ByVal index As Long) As IUnknown
    'get value at index - one-based index
    Register i As Long : i = index
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitF(i<1 Or i>@p.@arr.count, LibErrB)
    ExitF(@p.@arr.@arr[i]=0, LibErrO)
    Poke Long, VarPtr(o), @p.@arr.@arr[i]
    Function = o
    Poke Long, VarPtr(o), 0
End Function

Sub OjArrSet(ByVal p As OjArr Ptr, ByVal index As Long, value As IUnknown)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    ExitS(IsNothing(value), LibErrO)
    If @p.@arr.@arr[i] Then @p.@arr.@arr[i] = MemFreeObj(@p.@arr.@arr[i])
    value.AddRef()
    @p.@arr.@arr[i] = Peek(Long, VarPtr(value))
End Sub

Sub OjArrAdd(ByVal p As OjArr Ptr, value As IUnknown)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(IsNothing(value), LibErrO)
    value.AddRef()
    LnArrAdd @p.arr, Peek(Long, VarPtr(value))
End Sub

Sub OjArrIns(ByVal p As OjArr Ptr, ByVal index As Long, value As IUnknown)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    ExitS(IsNothing(value), LibErrO)
    value.AddRef()
    LnArrIns @p.arr, i, Peek(Long, VarPtr(value))
End Sub

Sub OjArrDel(ByVal p As OjArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    @p.@arr.@arr[i] = MemFreeObj(@p.@arr.@arr[i])
    LnArrDel @p.arr, i
End Sub

Sub OjArrSort(ByVal p As OjArr Ptr, ByVal compareCallback As Long)
    'sort array - fast non-recursive Quick Sort
    'compareCallback = CodePtr( callback function to compare two objects stored in container )
    '   OjArrCompareCB(a As IUnknown, b As IUnknown) As Long
    '       a < b : return < 0
    '       a = b : return = 0
    '       a > b : return > 0
    Local i, j, k, leftIndex, rightIndex, counter, compare As Long
    Local arr As Long Ptr
    Local oK, oI, oJ As IUnknown
    Local pK, pI, pJ As Long Ptr
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(compareCallback=0, LibErrC)
    pK = VarPtr(oK) : pI = VarPtr(oI) : pJ = VarPtr(oJ)
    If @p.@arr.count > 1 Then
        arr = @p.@arr.arr
        leftIndex = 1
        rightIndex = @p.@arr.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            pK = @arr[k] : If @pK = 0 Then GoTo HERE
            While i <= j
                pI = @arr[i] : If @pI = 0 Then GoTo HERE
                Call Dword compareCallback Using OjArrCompareCB(oI, oK) To compare
                While compare < 0
                    Incr i
                    pI = @arr[i] : If @pI = 0 Then GoTo HERE
                    Call Dword compareCallback Using OjArrCompareCB(oI, oK) To compare
                Wend
                pJ = @arr[j] : If @pJ = 0 Then GoTo HERE
                Call Dword compareCallback Using OjArrCompareCB(oJ, oK) To compare
                While compare > 0
                    pJ = @arr[j] : If @pJ = 0 Then GoTo HERE
                    Call Dword compareCallback Using OjArrCompareCB(oJ, oK) To compare
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            @pK = 0 : @pI = 0 : @pJ = 0
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
    Exit Sub
    HERE:
        @pK = 0 : @pI = 0 : @pJ = 0
        ExitS(1, LibErrO)
End Sub

Sub OjArrUniqueSort(ByVal p As OjArr Ptr, ByVal compareCallback As Long)
    'sort array and delete all duplicates
    'compareCallback = CodePtr( callback function to compare two objects stored in container )
    '   OjArrCompareCB(a As IUnknown, b As IUnknown) As Long
    '       a < b : return < 0
    '       a = b : return = 0
    '       a > b : return > 0
    Local i, compare As Long
    Local arr As Long Ptr
    Local a, b As IUnknown
    Local pA, pB As Long Ptr
    Err = 0
    OjArrSort p, compareCallback : If Err Then Exit Sub
    arr = @p.@arr.arr
    pA = VarPtr(a) : pB = VarPtr(b)
    For i = @p.@arr.count - 1 To 1 Step -1
        @pA = @arr[i + 1] : @pB = @arr[i]
        Call Dword compareCallback Using OjArrCompareCB(a, b) To compare
        @pA = 0 : @pB = 0
        If compare = 0 Then OjArrDel p, i + 1
    Next i
End Sub

Sub OjArrReverse(ByVal p As OjArr Ptr)
    'reverse array
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    LnArrReverse @p.arr
End Sub

Function OjArrBinSearch(ByVal p As OjArr Ptr, value As IUnknown, ByVal compareCallback As Long) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    'compareCallback = CodePtr( callback function to compare two objects stored in container )
    '   OjArrCompareCB(a As IUnknown, b As IUnknown) As Long
    '       a < b : return < 0
    '       a = b : return = 0
    '       a > b : return > 0
    Local i, top, bot, compare As Long
    Local arr As Long Ptr
    Local o As IUnknown
    Local pO As Long Ptr : pO = VarPtr(o)
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitF(IsNothing(value), LibErrA)
    bot = 1
    top = @p.@arr.count
    arr = @p.@arr.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If @arr[i] = 0 Then GoTo HERE
        @pO = @arr[i]
        Call Dword compareCallback Using OjArrCompareCB(o, value) To compare
        If compare > 0 Then
            bot = i + 1
        ElseIf compare < 0 Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
    @pO = 0
    Exit Function
    HERE:
        @pO = 0
        ExitF(1, LibErrN)
End Function

'++
    '----------------------------------------------------------------------------------------
    '   String Stack, Queue, Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub OjArrPushFirst(ByVal p As OjArr Ptr, value As IUnknown)
    'add value to front of container
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then OjArrIns p, 1, value Else OjArrAdd p, value
End Sub

Sub OjArrPushLast(ByVal p As OjArr Ptr, value As IUnknown)
    'add value to and of container
    OjArrAdd p, value
End Sub

Function OjArrPeekFirst(ByVal p As OjArr Ptr) As IUnknown
    'get first value in container
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ExitF(@p.@arr.@arr[1]=0, LibErrN)
        Poke Long, VarPtr(o), @p.@arr.@arr[1]
        o.AddRef()
        Function = o
    End If
End Function

Function OjArrPeekLast(ByVal p As OjArr Ptr) As IUnknown
    'get last value in container
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ExitF(@p.@arr.@arr[@p.@arr.count]=0, LibErrN)
        Poke Long, VarPtr(o), @p.@arr.@arr[@p.@arr.count]
        o.AddRef()
        Function = o
    End If
End Function

Function OjArrPopFirst(ByVal p As OjArr Ptr) As IUnknown
    'get and remove first value in container
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ExitF(@p.@arr.@arr[1]=0, LibErrN)
        Poke Long, VarPtr(o), @p.@arr.@arr[1]
        o.AddRef()
        Function = o
        OjArrDel p, 1
    End If
End Function

Function OjArrPopLast(ByVal p As OjArr Ptr) As IUnknown
    'get and remove last value in container
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ExitF(@p.@arr.@arr[@p.@arr.count]=0, LibErrN)
        Poke Long, VarPtr(o), @p.@arr.@arr[@p.@arr.count]
        o.AddRef()
        Function = o
        OjArrDel p, @p.@arr.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function OjArrClone(ByVal p As OjArr Ptr) As Long
    'duplicate container - return clone's handle
    Local i As Long
    Local clone As OjArr Ptr
    Local o As IUnknown
    Err = 0
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    clone = OjArrNew() : If Err Then Exit Function
    If @p.@arr.count Then
        OjArrReDim clone, @p.@arr.count : If Err Then Exit Function
        For i = 1 To @p.@arr.count
            o = OjArrGet(p, i)
            OjArrSet clone, i, o
        Next i
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function OjArrStore(ByVal p As OjArr Ptr, ByVal storeCallback As Long) As String
    'store container to string
    'storeCallback = CodePtr( callback function to pack each stored object into a string )
    '   Function OjArrStoreCB(o As IUnknown) As String
    Local i, h As Long
    Local s As String
    Local o As IUnknown
    Err = 0
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitF(storeCallback=0, LibErrC)
    h = SsArrNew() : If Err Then Exit Function
    SsArrReDim h, @p.@arr.count : If Err Then Exit Function
    For i = 1 To @p.@arr.count
        ExitF(@p.@arr.@arr[i]=0, LibErrN)
        o = OjArrGet(p, i)
        Call Dword storeCallback Using OjArrStoreCB(o) To s
        SsArrSet h, i, s
    Next i
    Function = SsArrStore(h)
    h = SsArrFinal(h)
End Function

Sub OjArrRestore(ByVal p As OjArr Ptr, ByVal s As String, ByVal restoreCallback As Long)
    'restore container from string
    'restoreCallback = CodePtr( callback function to convert string back into object )
    '   Function OjArrRestoreCB(ByVal s As String) As IUnknown
    Local i, h As Long
    Local o As IUnknown
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    OjArrClear p
    h = SsArrNew()
    If Len(s) Then
        SsArrRestore h, s : If Err Then Exit Sub
        OjArrReDim p, SsArrCount(h) : If Err Then Exit Sub
        For i = 1 To SsArrCount(h)
            Call Dword restoreCallback Using OjArrRestoreCB(SsArrGet(h, i)) To o
            OjArrSet p, i, o
        Next i
    End If
    h = SsArrFinal(h)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub OjArrFileStore(ByVal p As OjArr Ptr, ByVal file As String, ByVal storeCallback As Long)
    'store container to file
    'storeCallback = CodePtr( callback function to pack each stored object into a string )
    '   Function OjArrStoreCB(o As IUnknown) As String
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    s = OjArrStore(p, storeCallback) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub OjArrFileRestore(ByVal p As OjArr Ptr, ByVal file As String, ByVal restoreCallback As Long)
    'restore container from file
    'restoreCallback = CodePtr( callback function to convert string back into object )
    '   Function OjArrRestoreCB(ByVal s As String) As IUnknown
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(restoreCallback=0, LibErrC)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        OjArrRestore p, s, restoreCallback
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub
