#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\Hash.inc"
#Include Once "C:\HLib3\String\WsStr.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\WsLst.inc"
#Include Once "C:\HLib3\List\LnLst.inc"

'++
    '----------------------------------------------------------------------------------------
    '   WString/Long ~ Hash Table Container
    '       http://en.wikipedia.org/wiki/Hash_table
    '       Key must be unique WString
    '       Key is case sensitive - no nulls
    '       Value stored/retrieved/removed using unique lookup Key
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = WsLnHshNew() 'get handle for new container
    '       h = WsLnHshFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Macro WsLnHshTag = 1328741271
Type WsLnHshNode
    next As WsLnHshNode Ptr
    K As WsStr Ptr
    V As Long
End Type
Type WsLnHsh
    tag As Long
    count As Long
    cap As Long
    arr As Long Ptr
End Type

Function WsLnHshNew(ByVal capacity As Long) As Long
    'allocate new container - return handle
    'capacity = number of expected Keys (minium 100)
    Local i As Long
    Local p As WsLnHsh Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = WsLnHshTag
    @p.cap = Max&(capacity, 10)
    @p.arr = MemAlloc(@p.cap * 4)
    ExitF(@p.arr=0, LibErrM)
    Function = p
End Function

Function WsLnHshFinal(ByVal p As WsLnHsh Ptr) As Long
    'free allocated container - return null
    Local i As Long
    Local n, x As WsLnHshNode Ptr
    If p Then
        ExitF(@p.tag<>WsLnHshTag, LibErrH)
        For i = 0 To @p.cap - 1
            n = @p.@arr[i] : @p.@arr[i] = 0
            While n
                x = n : n = @n.next
                WsFinal(@x.K)
                MemFree(x)
            Wend
        Next i
        @p.arr = MemFree(@p.arr)
    End If
End Function

Function WsLnHshValidate(ByVal p As WsLnHsh Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = WsLnHshTag Then Function = @p.tag
End Function

Sub WsLnHshClear(ByVal p As WsLnHsh Ptr)
    'delete all data
    Local i As Long
    Local n, x As WsLnHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    For i = 0 To  @p.cap - 1
        n = @p.@arr[i] : @p.@arr[i] = 0
        While n
            x = n : n = @n.next
            WsFinal(@x.K)
            MemFree(x)
        Wend
    Next i
    @p.count = 0
End Sub

Function WsLnHshCount(ByVal p As WsLnHsh Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    Function = @p.count
End Function

Function WsLnHshCapGet(ByVal p As WsLnHsh Ptr) As Long
    'get Hash Table capacity
    ExitF(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    Function = @p.cap
End Function

Sub WsLnHshCapSet(ByVal p As WsLnHsh Ptr, ByVal capacity As Long)
    'set Hash Table capacity
    'rebuild Hash Table with new capacity - data preserved
    'capacity should be about the same as number of stored Keys
    Local i, oldCap, newCap As Long
    Local arr As Long Ptr
    Local n, x As WsLnHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    oldCap = @p.cap
    newCap = Max&(capacity, 10)
    arr = MemAlloc(newCap * 4)
    ExitS(arr=0, LibErrM)
    Swap @p.arr, arr
    @p.cap = newCap
    @p.count = 0
    For i = 0 To oldCap - 1
        n = @arr[i] : @arr[i] = 0
        While n
            x = n : n = @n.next
            WsLnHshSet p, WsGet(@x.K), @x.V
            WsFinal(@x.K)
            MemFree(x)
        Wend
    Next i
    MemFree(arr)
End Sub

Sub WsLnHshSet(ByVal p As WsLnHsh Ptr, ByRef key As WString, ByVal value As Long, Opt ByVal DontReplace As Byte)
    'add Key/Value to Tash Table - Value replaced if Key exist unless DontReplace = True
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local n, nn As WsLnHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    If n Then
        Do
            equal = WsEqual(@n.@K.mem, ps)
            If equal Then
                If IsFalse DontReplace Then @n.V = value
                Exit Sub
            ElseIf @n.next Then
                n = @n.next
            Else
                Exit Loop
            End If
        Loop
        nn = MemAlloc(SizeOf(WsLnHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = WsSetNew(key)
        @nn.V = value
        @n.next = nn
        Incr @p.count
    Else
        nn = MemAlloc(SizeOf(WsLnHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = WsSetNew(key)
        @nn.V = value
        @p.@arr[i] = nn
        Incr @p.count
    End If
End Sub

Function WsLnHshGet(ByVal p As WsLnHsh Ptr, ByRef key As WString) As Long
    'get Key's associated Value
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local n As WsLnHshNode Ptr
    ExitF(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        equal = WsEqual(@n.@K.mem, ps)
        If equal Then
            Function = @n.V
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Function WsLnHshGot(ByVal p As WsLnHsh Ptr, ByRef key As WString) As Byte
    'True/False if Key in Hash Table
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local n As WsLnHshNode Ptr
    ExitF(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        equal = WsEqual(@n.@K.mem, ps)
        If equal Then
            Function = 1
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Sub WsLnHshDel(ByVal p As WsLnHsh Ptr, ByRef key As WString)
    'remove Key and associated Value from Hash Table
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local prev, n As WsLnHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    prev = 0
    n = @p.@arr[i]
    While n
        equal = WsEqual(@n.@K.mem, ps)
        If equal Then
            If prev Then
                @prev.next = @n.next
            Else
                @p.@arr[i] = @n.next
            End If
            @n.K = WsFinal(@n.K)
            MemFree(n)
            ExitS(@p.count=0, LibErrU)
            Decr @p.count
            Exit Loop
        End If
        prev = n
        n = @n.next
    Wend
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function WsLnHshClone(ByVal p As WsLnHsh Ptr) As Long
    'create duplicate container - return handle to cloned container
    Local i As Long
    Local n As WsLnHshNode Ptr
    Local clone As Long
    Err = 0
    ExitF(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    clone = WsLnHshNew(@p.cap) : If Err Then Exit Function
    For i = 0 To @p.cap - 1
        n = @p.@arr[i]
        While n
            WsLnHshSet clone, WsGet(@n.K), @n.V
            n = @n.next
        Wend
    Next i
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function WsLnHshStore(ByVal p As WsLnHsh Ptr) As String
    'store container to string
    Local i As Long
    Local keys, vals, store As Long
    Local n As WsLnHshNode Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    keys = WsLstNew() : If Err Then Exit Function
    vals = LnLstNew() : If Err Then Exit Function
    store = SsLstNew() : If Err Then Exit Function
    If @p.count Then
        For i = 0 To @p.cap - 1
            n = @p.@arr[i]
            While n
                WsLstAdd keys, WsGet(@n.K)
                LnLstAdd vals, @n.V
                n = @n.next
            Wend
        Next i
        SsLstAdd store, Mkl$(@p.cap)
        SsLstAdd store, WsLstStore(keys)
        SsLstAdd store, LnLstStore(vals)
        Function = SsLstStore(store)
    End If
    keys = WsLstFinal(keys)
    vals = LnLstFinal(vals)
    store = SsLstFinal(store)
End Function

Sub WsLnHshRestore(ByVal p As WsLnHsh Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, store As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    WsLnHshClear p
    keys = WsLstNew() : If Err Then Exit Sub
    vals = LnLstNew() : If Err Then Exit Sub
    store = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore store, s : If Err Then Exit Sub
        ExitS(SsLstCount(store)<>3, LibErrU)
        WsLnHshCapSet p, Cvl(SsLstPopFirst(store))
        WsLstRestore keys, SsLstPopFirst(store)
        LnLstRestore vals, SsLstPopFirst(store)
        ExitS(WsLstCount(keys)<>LnLstCount(vals), LibErrU)
        While WsLstCount(keys)
            WsLnHshSet p, WsLstPopFirst(keys), LnLstPopFirst(vals)
        Wend
    End If
    keys = WsLstFinal(keys)
    vals = LnLstFinal(vals)
    store = SsLstFinal(store)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub WsLnHshFileStore(ByVal p As WsLnHsh Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    s = WsLnHshStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub WsLnHshFileRestore(ByVal p As WsLnHsh Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>WsLnHshTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        WsLnHshRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub
