#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro WdLstNodeTag = 1540788735
Macro WdLstTag = 1868553327
Type WdLstNode
    tag As Long
    next As WdLstNode Ptr
    prev As WdLstNode Ptr
    value As Word
End Type
Type WdLst
    tag As Long
    count As Long
    first As WdLstNode Ptr
    last As WdLstNode Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Word List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = WdLstNew() 'get handle for new container
    '       h = WdLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function WdLstNew() As Long
    'allocate new container - return handle
    Local pLst As WdLst Ptr
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = WdLstTag
    Function = pLst
End Function

Function WdLstFinal(ByVal pLst As WdLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>WdLstTag, LibErrH)
        WdLstClear pLst
        MemFree(pLst)
    End If
End Function

Function WdLstValidate(ByVal pLst As WdLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = WdLstTag Then Function = @pLst.tag
End Function

Sub WdLstClear(ByVal pLst As WdLst Ptr)
    'delete all data
    Local node As WdLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    While @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    Wend
    @pLst.last = 0
    @pLst.count = 0
End Sub

Function WdLstCount(ByVal pLst As WdLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    Function = @pLst.count
End Function

Sub WdLstAdd(ByVal pLst As WdLst Ptr, ByVal value As Word)
    'append Value to end of List
    Local node As WdLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.count Then
        node = WdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    Else
        node = WdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Sub WdLstIns(ByVal pLst As WdLst Ptr, ByVal value As Word)
    'insert Value at front of List
    Local node As WdLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.count Then
        node = WdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    Else
        node = WdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Function WdLstFirst(ByVal pLst As WdLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    Function = @pLst.first
End Function

Function WdLstLast(ByVal pLst As WdLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    Function = @pLst.last
End Function

Function WdLstNext(ByVal pNode As WdLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>WdLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function WdLstPrev(ByVal pNode As WdLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>WdLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function WdLstGet(ByVal pNode As WdLstNode Ptr) As Word
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>WdLstNodeTag, LibErrH)
    Function = @pNode.value
End Function

Sub WdLstSet(ByVal pNode As WdLstNode Ptr, ByVal value As Word)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>WdLstNodeTag, LibErrH)
    @pNode.value = value
End Sub

Sub WdLstInsPrev(ByVal pLst As WdLst Ptr, ByVal pNode As WdLstNode Ptr, ByVal value As Word)
    'insert Value before node
    Local node As WdLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>WdLstTag Or @pNode.tag<>WdLstNodeTag, LibErrH)
    If pNode = @pLst.first Then
        WdLstIns pLst, value
    Else
        ExitS(@pNode.prev=0, LibErrU)
        node = WdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    End If
End Sub

Sub WdLstInsNext(ByVal pLst As WdLst Ptr, ByVal pNode As WdLstNode Ptr, ByVal value As Word)
    'insert Value after Cursor
    Local node As WdLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>WdLstTag Or @pNode.tag<>WdLstNodeTag, LibErrH)
    If pNode = @pLst.last Then
        WdLstAdd pLst, value
    Else
        ExitS(@pNode.next=0, LibErrU)
        node = WdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    End If
End Sub

Sub WdLstDel(ByVal pLst As WdLst Ptr, ByVal pNode As WdLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>WdLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>WdLstNodeTag, LibErrH)
        If @pLst.first = pNode Then @pLst.first = @pNode.next
        If @pLst.last = pNode Then @pLst.last = @pNode.prev
        If @pNode.prev Then @pNode.@prev.next = @pNode.next
        If @pNode.next Then @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        Decr @pLst.count
        MemFree(pNode)
    End If
End Sub

Sub WdLstDelPrev(ByVal pLst As WdLst Ptr, ByVal pNode As WdLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>WdLstTag Or @pNode.tag<>WdLstNodeTag, LibErrH)
    WdLstDel pLst, @pNode.prev
End Sub

Sub WdLstDelNext(ByVal pLst As WdLst Ptr, ByVal pNode As WdLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>WdLstTag Or @pNode.tag<>WdLstNodeTag, LibErrH)
    WdLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub WdLstStkPush(ByVal pLst As WdLst Ptr, ByVal value As Word)
    'Push Value on Stack
    WdLstAdd pLst, value
End Sub

Function WdLstStkPeek(ByVal pLst As WdLst Ptr) As Word
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function WdLstStkPop(ByVal pLst As WdLst Ptr) As Word
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        WdLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub WdLstQuePush(ByVal pLst As WdLst Ptr, ByVal value As Word)
    'Add Value to end of Queue
    WdLstAdd pLst, value
End Sub

Function WdLstQuePeek(ByVal pLst As WdLst Ptr) As Word
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function WdLstQuePop(ByVal pLst As WdLst Ptr) As Word
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        WdLstDel pLst, @pLst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub WdLstPushFirst(ByVal pLst As WdLst Ptr, ByVal value As Word)
    'Add Value at front of container
    WdLstIns pLst, value
End Sub

Sub WdLstPushLast(ByVal pLst As WdLst Ptr, ByVal value As Word)
    'Add Value at end of container
    WdLstAdd pLst, value
End Sub

Function WdLstPeekFirst(ByVal pLst As WdLst Ptr) As Word
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function WdLstPeekLast(ByVal pLst As WdLst Ptr) As Word
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function WdLstPopFirst(ByVal pLst As WdLst Ptr) As Word
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        WdLstDel pLst, @pLst.first
    End If
End Function

Function WdLstPopLast(ByVal pLst As WdLst Ptr) As Word
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        WdLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function WdLstClone(ByVal pLst As WdLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As WdLst Ptr
    Local node As WdLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    Err = 0
    pClone = WdLstNew() : If Err Then Exit Function
    node = @pLst.first
    While node
        WdLstAdd pClone, @node.value
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function WdLstStore(ByVal pLst As WdLst Ptr) As String
    'store container to String
    Local s As String
    Local node As WdLstNode Ptr
    Local p As Word Ptr
    ExitF(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    If @pLst.count Then
        s = Nul$(@pLst.count * %SizeWd)
        p = StrPtr(s)
        node = @pLst.first
        While node
            @p = @node.value : Incr p
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub WdLstRestore(ByVal pLst As WdLst Ptr, ByRef s As String)
    'restore container from string
    Local items As Long
    Local p As Word Ptr
    ExitS(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    WdLstClear pLst
    If Len(s) Then
        p = StrPtr(s)
        items = Len(s) / %SizeWd
        While items
            WdLstAdd pLst, @p
            Incr p
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub WdLstFileStore(ByVal pLst As WdLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    s = WdLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub WdLstFileRestore(ByVal pLst As WdLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>WdLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        WdLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function WdLstNodeAlloc(ByVal pLst As WdLst Ptr) Private As Long
    Local node As WdLstNode Ptr
    node = MemAlloc(SizeOf(WdLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = WdLstNodeTag
    Incr @pLst.count
    Function = node
End Function
