#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\LnLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'String/Long ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - no limit on Key length
    '   - use SsLnTreComparison() to change how Keys compared
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = SsLnTreNew() 'get handle for new container
    '   h = SsLnTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro SsLnTreNodeTag = 513674861
Macro SsLnTreTag = 719713421
Type SsLnTreNode
    tag As Long
    P As SsLnTreNode Ptr
    L As SsLnTreNode Ptr
    R As SsLnTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As Long
End Type
Type SsLnTre
    tag As Long
    count As Long
    root As SsLnTreNode Ptr
    compareCB As Long
    collation As SsStr Ptr
End Type

Function SsLnTreNew() As Long
    'allocate new container - return handle
    Local p As SsLnTre Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = SsLnTreTag
    @p.compareCB = CodePtr(SsCompare)
    @p.collation = SsNew() : If Err Then Exit Function
    Function = p
End Function

Function SsLnTreFinal(ByVal pTree As SsLnTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>SsLnTreTag, LibErrH)
        @pTree.collation = SsFinal(@pTree.collation)
        SsLnTreClear pTree
        MemFree(pTree)
    End If
End Function

Function SsLnTreValidate(ByVal pTree As SsLnTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = SsLnTreTag Then Function = @pTree.tag
End Function

Sub SsLnTreComparison(ByVal pTree As SsLnTre Ptr, ByVal compareUCase As Long, ByVal collationSequence As String)
    'set how Strings compared
    'default = case ignored
    'if collationSequence String provided then
    '   Strings are compared using the order of the collation sequence String
    '   collation String must be 256 characters
    'else if compareUCase = True then
    '   Strings compared UCase
    ExitS(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    @pTree.compareCB = CodePtr(SsCompare)
    SsClear @pTree.collation
    If Len(collationSequence) Then
        ExitS(Len(collationSequence)<>256, LibErrS)
        SsSet @pTree.collation, collationSequence : If Err Then Exit Sub
        @pTree.compareCB = CodePtr(SsCompareCollate)
    ElseIf compareUCase Then
        @pTree.compareCB = CodePtr(SsCompareUCase)
    End If
End Sub

Sub SsLnTreClear(ByVal pTree As SsLnTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As SsLnTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = SsLnTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = SsLnTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            SsLnTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function SsLnTreCount(ByVal pTree As SsLnTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub SsLnTreSet(ByVal pTree As SsLnTre Ptr, ByRef key As String, ByVal value As Long, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local compare, temp As Long
    Local n As SsLnTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Sub
    If @pTree.root Then
        n = @pTree.root
        While 1
            Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
            If compare > 0 Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = SsLnTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    SsSet @n.@R.K, key
                    @n.@R.V = value
                    SsLnTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf compare < 0 Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = SsLnTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    SsSet @n.@L.K, key
                    @n.@L.V = value
                    SsLnTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then @n.V = value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = SsLnTreAllocNode(pTree) : If Err Then Exit Sub
        SsSet @pTree.@root.K, key
        @pTree.@root.V = value
        @pTree.count = 1
    End If
    temp = SsFinal(temp)
End Sub

Function SsLnTreGet(ByVal pTree As SsLnTre Ptr, ByRef key As String) As Long
    'get Key's associated Value
    Local compare, temp As Long
    Local n As SsLnTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = @n.V
            Exit Loop
        End If
    Wend
    temp = SsFinal(temp)
End Function

Function SsLnTreGot(ByVal pTree As SsLnTre Ptr, ByRef key As String) As Long
    'True/False if Key exist
    Local compare, temp As Long
    Local n As SsLnTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
    temp = SsFinal(temp)
End Function

Sub SsLnTreDel(ByVal pTree As SsLnTre Ptr, ByRef key As String)
    'remove Key and associated Value
    Local pNode As SsLnTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    pNode = SsLnTreGot(pTree, key)
    If pNode Then
        SsLnTreRemoveNode(pTree, pNode)
    End If
End Sub

Function SsLnTreFirst(ByVal pTree As SsLnTre Ptr) As Long
    'get handle to first node in tree
    Local n As SsLnTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function SsLnTreLast(ByVal pTree As SsLnTre Ptr) As Long
    'get handle to last node in tree
    Local n As SsLnTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function SsLnTreNext(ByVal pNode As SsLnTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As SsLnTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>SsLnTreNodeTag, LibErrH)
        minR = SsLnTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = SsLnTreParentGreater(pNode)
        End If
    End If
End Function

Function SsLnTrePrev(ByVal pNode As SsLnTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As SsLnTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>SsLnTreNodeTag, LibErrH)
        maxL = SsLnTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = SsLnTreParentLesser(pNode)
        End If
    End If
End Function

Function SsLnTreGetKey(ByVal pNode As SsLnTreNode Ptr) As String
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>SsLnTreNodeTag, LibErrH)
    Function = SsGet(@pNode.K)
End Function

Function SsLnTreGetVal(ByVal pNode As SsLnTreNode Ptr) As Long
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>SsLnTreNodeTag, LibErrH)
    Function = @pNode.V
End Function

Sub SsLnTreSetVal(ByVal pNode As SsLnTreNode Ptr, ByVal value As Long)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>SsLnTreNodeTag, LibErrH)
    @pNode.V = value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SsLnTreClone(ByVal pTree As SsLnTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    clone = SsLnTreNew() : If Err Then Exit Function
    h = SsLnTreFirst(pTree)
    While h
        SsLnTreSet clone, SsLnTreGetKey(h), SsLnTreGetVal(h)
        h = SsLnTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SsLnTreStore(ByVal pTree As SsLnTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    keys = SsLstNew() : If Err Then Exit Function
    vals = LnLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = SsLnTreFirst(pTree)
        While h
            SsLstAdd keys, SsLnTreGetKey(h)
            LnLstAdd vals, SsLnTreGetVal(h)
            h = SsLnTreNext(h)
        Wend
        SsLstAdd stor, SsLstStore(keys)
        SsLstAdd stor, LnLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = SsLstFinal(keys)
    vals = LnLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub SsLnTreRestore(ByVal pTree As SsLnTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    SsLnTreClear pTree
    keys = SsLstNew() : If Err Then Exit Sub
    vals = LnLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        SsLstRestore keys, SsLstPopFirst(stor)
        LnLstRestore vals, SsLstPopFirst(stor)
        ExitS(SsLstCount(keys)<>LnLstCount(vals), LibErrU)
        While SsLstCount(keys)
            SsLnTreSet pTree, SsLstPopFirst(keys), LnLstPopFirst(vals)
        Wend
    End If
    keys = SsLstFinal(keys)
    vals = LnLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SsLnTreFileStore(ByVal pTree As SsLnTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    s = SsLnTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsLnTreFileRestore(ByVal pTree As SsLnTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsLnTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SsLnTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub SsLnTreRemoveNode(ByVal p As SsLnTre Ptr, ByVal n As SsLnTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As SsLnTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, SsLnTreMaxLeft(n), SsLnTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        SsLnTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = SsLnTreFreeNode(p, n)
        SsLnTreBalanceBranch p, nP
    End If
End Sub

Function SsLnTreAllocNode(ByVal p As SsLnTre Ptr) Private As Long
    Local n As SsLnTreNode Ptr
    n = MemAlloc(SizeOf(SsLnTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = SsLnTreNodeTag
    @n.K = SsNew() : If Err Then Exit Function
    Incr @p.count
    Function = n
End Function

Function SsLnTreFreeNode(ByVal p As SsLnTre Ptr, ByVal n As SsLnTreNode Ptr) As Long
    If n Then
        @n.K = SsFinal(@n.K)
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub SsLnTreBalanceBranch(ByVal p As SsLnTre Ptr, ByVal n As SsLnTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = SsLnTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = SsLnTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function SsLnTreMaxLeft(ByVal n As SsLnTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function SsLnTreMinRight(ByVal n As SsLnTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function SsLnTreParentGreater(ByVal n As SsLnTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function SsLnTreParentLesser(ByVal n As SsLnTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function SsLnTreRotateLeft(ByVal p As SsLnTre Ptr, ByVal n As SsLnTreNode Ptr) Private As Long
    Local nR, nRL As SsLnTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function SsLnTreRotateRight(ByVal p As SsLnTre Ptr, ByVal n As SsLnTreNode Ptr) Private As Long
    Local nL, nLR As SsLnTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
