#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro CuArrTag = -1948894953
Type CuArr
    tag As Long
    count As Long
    arr As Currency Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Currency Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = CuArrNew() 'get handle for new container
    '       h = CuArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function CuArrNew() As Long
    'allocate new container - return handle
    Local p As CuArr Ptr
    p = MemAlloc(SizeOf(CuArr))
    ExitF(p=0, LibErrM)
    @p.tag = CuArrTag
    Function = p
End Function

Function CuArrFinal(ByVal p As CuArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>CuArrTag, LibErrH)
        CuArrClear p
        MemFree(p)
    End If
End Function

Function CuArrValidate(ByVal p As CuArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = CuArrTag Then Function = @p.tag
End Function

Sub CuArrClear(ByVal p As CuArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeCu
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function CuArrCount(ByVal p As CuArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    Function = @p.count
End Function

Sub CuArrReDim(ByVal p As CuArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        CuArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeCu
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeCu)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeCu
    End If
End Sub

Function CuArrGet(ByVal p As CuArr Ptr, ByVal index As Long) As Currency
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub CuArrSet(ByVal p As CuArr Ptr, ByVal index As Long, ByVal value As Currency)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub CuArrAdd(ByVal p As CuArr Ptr, ByVal value As Currency)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    CuArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub CuArrIns(ByVal p As CuArr Ptr, ByVal index As Long, ByVal value As Currency)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    CuArrReDim p, @p.count + 1 : If Err Then Exit Sub
    CuArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub CuArrDel(ByVal p As CuArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then CuArrMove @p.arr, i + 1, i, @p.count - i
    CuArrReDim p, @p.count - 1
End Sub

Sub CuArrSort(ByVal p As CuArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As Currency
    Local arr As Currency Ptr
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub CuArrUniqueSort(ByVal p As CuArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As Currency Ptr
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    CuArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then CuArrDel p, i + 1
    Next i
End Sub

Sub CuArrReverse(ByVal p As CuArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As Currency Ptr
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function CuArrBinSearch(ByVal p As CuArr Ptr, ByVal value As Currency) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As Currency Ptr
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub CuArrBinInsert(ByVal p As CuArr Ptr, ByVal value As Currency)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    i = CuArrBinPos(p, value, compare)
    If i = 0 Then
        CuArrAdd p, value
    ElseIf compare <= 0 Then
        CuArrIns p, i, value
    ElseIf i < @p.count Then
        CuArrIns p, i + 1, value
    Else
        CuArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Currency Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub CuArrStkPush(ByVal p As CuArr Ptr, ByVal value As Currency)
    'push value on top of Stack
    CuArrAdd p, value
End Sub

Function CuArrStkPeek(ByVal p As CuArr Ptr) As Currency
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function CuArrStkPop(ByVal p As CuArr Ptr) As Currency
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        CuArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Currency Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub CuArrQuePush(ByVal p As CuArr Ptr, ByVal value As Currency)
    'add value to end of Queue
    CuArrAdd p, value
End Sub

Function CuArrQuePeek(ByVal p As CuArr Ptr) As Currency
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function CuArrQuePop(ByVal p As CuArr Ptr) As Currency
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        CuArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Currency Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub CuArrPushFirst(ByVal p As CuArr Ptr, ByVal value As Currency)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then CuArrIns p, 1, value Else CuArrAdd p, value
End Sub

Sub CuArrPushLast(ByVal p As CuArr Ptr, ByVal value As Currency)
    'add value to and of Deque (double-ended Queue)
    CuArrAdd p, value
End Sub

Function CuArrPeekFirst(ByVal p As CuArr Ptr) As Currency
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function CuArrPeekLast(ByVal p As CuArr Ptr) As Currency
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function CuArrPopFirst(ByVal p As CuArr Ptr) As Currency
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        CuArrDel p, 1
    End If
End Function

Function CuArrPopLast(ByVal p As CuArr Ptr) As Currency
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        CuArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function CuArrClone(ByVal p As CuArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As CuArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    clone = CuArrNew() : If Err Then Exit Function
    If @p.count Then
        CuArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeCu, @clone.arr + %SizeCu, @p.count * %SizeCu
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function CuArrStore(ByVal p As CuArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>CuArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeCu, @p.count * %SizeCu)
End Function

Sub CuArrRestore(ByVal p As CuArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    CuArrClear p
    If strLen Then
        CuArrReDim p, strLen / %SizeCu : If Err Then Exit Sub
        Poke$ @p.arr + %SizeCu, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub CuArrFileStore(ByVal p As CuArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    s = CuArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub CuArrFileRestore(ByVal p As CuArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>CuArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        CuArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub CuArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeCu), arr + (toIndex * %SizeCu),  Count * %SizeCu
End Sub

Function CuArrBinPos(ByVal p As CuArr Ptr, ByVal value As Currency, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Currency Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function
