#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro DbArrTag = -1821650520
Type DbArr
    tag As Long
    count As Long
    arr As Double Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Double Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = DbArrNew() 'get handle for new container
    '       h = DbArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function DbArrNew() As Long
    'allocate new container - return handle
    Local p As DbArr Ptr
    p = MemAlloc(SizeOf(DbArr))
    ExitF(p=0, LibErrM)
    @p.tag = DbArrTag
    Function = p
End Function

Function DbArrFinal(ByVal p As DbArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>DbArrTag, LibErrH)
        DbArrClear p
        MemFree(p)
    End If
End Function

Function DbArrValidate(ByVal p As DbArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = DbArrTag Then Function = @p.tag
End Function

Sub DbArrClear(ByVal p As DbArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeDb
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function DbArrCount(ByVal p As DbArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    Function = @p.count
End Function

Sub DbArrReDim(ByVal p As DbArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        DbArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeDb
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeDb)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeDb
    End If
End Sub

Function DbArrGet(ByVal p As DbArr Ptr, ByVal index As Long) As Double
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub DbArrSet(ByVal p As DbArr Ptr, ByVal index As Long, ByVal value As Double)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub DbArrAdd(ByVal p As DbArr Ptr, ByVal value As Double)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    DbArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub DbArrIns(ByVal p As DbArr Ptr, ByVal index As Long, ByVal value As Double)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    DbArrReDim p, @p.count + 1 : If Err Then Exit Sub
    DbArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub DbArrDel(ByVal p As DbArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then DbArrMove @p.arr, i + 1, i, @p.count - i
    DbArrReDim p, @p.count - 1
End Sub

Sub DbArrSort(ByVal p As DbArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As Double
    Local arr As Double Ptr
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub DbArrUniqueSort(ByVal p As DbArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As Double Ptr
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    DbArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then DbArrDel p, i + 1
    Next i
End Sub

Sub DbArrReverse(ByVal p As DbArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As Double Ptr
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function DbArrBinSearch(ByVal p As DbArr Ptr, ByVal value As Double) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As Double Ptr
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub DbArrBinInsert(ByVal p As DbArr Ptr, ByVal value As Double)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    i = DbArrBinPos(p, value, compare)
    If i = 0 Then
        DbArrAdd p, value
    ElseIf compare <= 0 Then
        DbArrIns p, i, value
    ElseIf i < @p.count Then
        DbArrIns p, i + 1, value
    Else
        DbArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Double Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub DbArrStkPush(ByVal p As DbArr Ptr, ByVal value As Double)
    'push value on top of Stack
    DbArrAdd p, value
End Sub

Function DbArrStkPeek(ByVal p As DbArr Ptr) As Double
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function DbArrStkPop(ByVal p As DbArr Ptr) As Double
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        DbArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Double Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub DbArrQuePush(ByVal p As DbArr Ptr, ByVal value As Double)
    'add value to end of Queue
    DbArrAdd p, value
End Sub

Function DbArrQuePeek(ByVal p As DbArr Ptr) As Double
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function DbArrQuePop(ByVal p As DbArr Ptr) As Double
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        DbArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Double Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub DbArrPushFirst(ByVal p As DbArr Ptr, ByVal value As Double)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then DbArrIns p, 1, value Else DbArrAdd p, value
End Sub

Sub DbArrPushLast(ByVal p As DbArr Ptr, ByVal value As Double)
    'add value to and of Deque (double-ended Queue)
    DbArrAdd p, value
End Sub

Function DbArrPeekFirst(ByVal p As DbArr Ptr) As Double
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function DbArrPeekLast(ByVal p As DbArr Ptr) As Double
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function DbArrPopFirst(ByVal p As DbArr Ptr) As Double
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        DbArrDel p, 1
    End If
End Function

Function DbArrPopLast(ByVal p As DbArr Ptr) As Double
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        DbArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function DbArrClone(ByVal p As DbArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As DbArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    clone = DbArrNew() : If Err Then Exit Function
    If @p.count Then
        DbArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeDb, @clone.arr + %SizeDb, @p.count * %SizeDb
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function DbArrStore(ByVal p As DbArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>DbArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeDb, @p.count * %SizeDb)
End Function

Sub DbArrRestore(ByVal p As DbArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    DbArrClear p
    If strLen Then
        DbArrReDim p, strLen / %SizeDb : If Err Then Exit Sub
        Poke$ @p.arr + %SizeDb, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub DbArrFileStore(ByVal p As DbArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    s = DbArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub DbArrFileRestore(ByVal p As DbArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>DbArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        DbArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub DbArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeDb), arr + (toIndex * %SizeDb),  Count * %SizeDb
End Sub

Function DbArrBinPos(ByVal p As DbArr Ptr, ByVal value As Double, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Double Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function
