#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro QdArrTag = 1406048983
Type QdArr
    tag As Long
    count As Long
    arr As Quad Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Quad Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = QdArrNew() 'get handle for new container
    '       h = QdArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function QdArrNew() As Long
    'allocate new container - return handle
    Local p As QdArr Ptr
    p = MemAlloc(SizeOf(QdArr))
    ExitF(p=0, LibErrM)
    @p.tag = QdArrTag
    Function = p
End Function

Function QdArrFinal(ByVal p As QdArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>QdArrTag, LibErrH)
        QdArrClear p
        MemFree(p)
    End If
End Function

Function QdArrValidate(ByVal p As QdArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = QdArrTag Then Function = @p.tag
End Function

Sub QdArrClear(ByVal p As QdArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeQd
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function QdArrCount(ByVal p As QdArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    Function = @p.count
End Function

Sub QdArrReDim(ByVal p As QdArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        QdArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeQd
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeQd)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeQd
    End If
End Sub

Function QdArrGet(ByVal p As QdArr Ptr, ByVal index As Long) As Quad
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub QdArrSet(ByVal p As QdArr Ptr, ByVal index As Long, ByVal value As Quad)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub QdArrAdd(ByVal p As QdArr Ptr, ByVal value As Quad)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    QdArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub QdArrIns(ByVal p As QdArr Ptr, ByVal index As Long, ByVal value As Quad)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    QdArrReDim p, @p.count + 1 : If Err Then Exit Sub
    QdArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub QdArrDel(ByVal p As QdArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then QdArrMove @p.arr, i + 1, i, @p.count - i
    QdArrReDim p, @p.count - 1
End Sub

Sub QdArrSort(ByVal p As QdArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As Quad
    Local arr As Quad Ptr
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub QdArrUniqueSort(ByVal p As QdArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As Quad Ptr
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    QdArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then QdArrDel p, i + 1
    Next i
End Sub

Sub QdArrReverse(ByVal p As QdArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As Quad Ptr
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function QdArrBinSearch(ByVal p As QdArr Ptr, ByVal value As Quad) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As Quad Ptr
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub QdArrBinInsert(ByVal p As QdArr Ptr, ByVal value As Quad)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    i = QdArrBinPos(p, value, compare)
    If i = 0 Then
        QdArrAdd p, value
    ElseIf compare <= 0 Then
        QdArrIns p, i, value
    ElseIf i < @p.count Then
        QdArrIns p, i + 1, value
    Else
        QdArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Quad Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub QdArrStkPush(ByVal p As QdArr Ptr, ByVal value As Quad)
    'push value on top of Stack
    QdArrAdd p, value
End Sub

Function QdArrStkPeek(ByVal p As QdArr Ptr) As Quad
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function QdArrStkPop(ByVal p As QdArr Ptr) As Quad
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        QdArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Quad Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub QdArrQuePush(ByVal p As QdArr Ptr, ByVal value As Quad)
    'add value to end of Queue
    QdArrAdd p, value
End Sub

Function QdArrQuePeek(ByVal p As QdArr Ptr) As Quad
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function QdArrQuePop(ByVal p As QdArr Ptr) As Quad
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        QdArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Quad Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub QdArrPushFirst(ByVal p As QdArr Ptr, ByVal value As Quad)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then QdArrIns p, 1, value Else QdArrAdd p, value
End Sub

Sub QdArrPushLast(ByVal p As QdArr Ptr, ByVal value As Quad)
    'add value to and of Deque (double-ended Queue)
    QdArrAdd p, value
End Sub

Function QdArrPeekFirst(ByVal p As QdArr Ptr) As Quad
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function QdArrPeekLast(ByVal p As QdArr Ptr) As Quad
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function QdArrPopFirst(ByVal p As QdArr Ptr) As Quad
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        QdArrDel p, 1
    End If
End Function

Function QdArrPopLast(ByVal p As QdArr Ptr) As Quad
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        QdArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function QdArrClone(ByVal p As QdArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As QdArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    clone = QdArrNew() : If Err Then Exit Function
    If @p.count Then
        QdArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeQd, @clone.arr + %SizeQd, @p.count * %SizeQd
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function QdArrStore(ByVal p As QdArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>QdArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeQd, @p.count * %SizeQd)
End Function

Sub QdArrRestore(ByVal p As QdArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    QdArrClear p
    If strLen Then
        QdArrReDim p, strLen / %SizeQd : If Err Then Exit Sub
        Poke$ @p.arr + %SizeQd, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub QdArrFileStore(ByVal p As QdArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    s = QdArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub QdArrFileRestore(ByVal p As QdArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>QdArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        QdArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub QdArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeQd), arr + (toIndex * %SizeQd),  Count * %SizeQd
End Sub

Function QdArrBinPos(ByVal p As QdArr Ptr, ByVal value As Quad, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Quad Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function
