#Include This Once
#Include Once "C:\HLib3\String\SsStr.inc"
#Include Once "C:\HLib3\Array\LnArr.inc"

Macro SsArrTag = 264591370
Type SsArr
    tag As Long
    arr As LnArr Ptr
    compareCB As Long
    collation As SsStr Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   String Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = SsArrNew() 'get handle for new container
    '       h = SsArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function SsArrNew() As Long
    'allocate new container - return handle
    Local p As SsArr Ptr
    Err = 0
    p = MemAlloc(SizeOf(SsArr))
    ExitF(p=0, LibErrM)
    @p.tag = SsArrTag
    @p.compareCB = CodePtr(SsCompare)
    @p.collation = SsNew() : If Err Then Exit Function
    @p.arr = LnArrNew() : If Err Then Exit Function
    Function = p
End Function

Function SsArrFinal(ByVal p As SsArr Ptr) As Long
    'free allocated container - return null
    Local i As Long
    If p Then
        ExitF(@p.tag<>SsArrTag, LibErrH)
        SsArrClear p
        @p.arr = LnArrFinal(@p.arr)
        @p.collation = SsFinal(@p.collation)
        MemFree(p)
    End If
End Function

Function SsArrValidate(ByVal p As SsArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = SsArrTag Then Function = @p.tag
End Function

Sub SsArrComparison(ByVal p As SsArr Ptr, ByVal compareUCase As Long, ByVal collationSequence As String)
    'set how Strings compared
    'default = case ignored
    'if collationSequence String provided then
    '   Strings are compared using the order of the collation sequence String
    '   collation String must be 256 characters
    'else if compareUCase = True then
    '   Strings compared UCase
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    @p.compareCB = CodePtr(SsCompare)
    SsClear @p.collation
    If Len(collationSequence) Then
        ExitS(Len(collationSequence)<>256, LibErrS)
        SsSet @p.collation, collationSequence : If Err Then Exit Sub
        @p.compareCB = CodePtr(SsCompareCollate)
    ElseIf compareUCase Then
        @p.compareCB = CodePtr(SsCompareUCase)
    End If
End Sub

Sub SsArrClear(ByVal p As SsArr Ptr)
    'delete all data
    Local i As Long
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    For i = 1 To @p.@arr.count
        @p.@arr.@arr[i] = SsFinal(@p.@arr.@arr[i])
    Next i
    LnArrClear @p.arr
End Sub

Function SsArrCount(ByVal p As SsArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    Function = @p.@arr.count
End Function

Sub SsArrReDim(ByVal p As SsArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local i, items As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        SsArrClear p
    ElseIf Count <> @p.@arr.count Then
        items = @p.@arr.count
        For i = Count + 1 To items
            @p.@arr.@arr[i] = SsFinal(@p.@arr.@arr[i])
        Next i
        LnArrReDim @p.arr, Count : If Err Then Exit Sub
        For i = items + 1 To Count
            @p.@arr.@arr[i] = SsNew() : If Err Then Exit Sub
        Next i
    End If
End Sub

Function SsArrGet(ByVal p As SsArr Ptr, ByVal index As Long) As String
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    ExitF(i<1 Or i>@p.@arr.count, LibErrB)
    Function = SsGet(@p.@arr.@arr[i])
End Function

Sub SsArrSet(ByVal p As SsArr Ptr, ByVal index As Long, ByRef value As String)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    SsSet @p.@arr.@arr[i], value
End Sub

Sub SsArrAdd(ByVal p As SsArr Ptr, ByRef value As String)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    LnArrAdd @p.arr, SsSetNew(value)
End Sub

Sub SsArrIns(ByVal p As SsArr Ptr, ByVal index As Long, ByRef value As String)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    LnArrIns @p.arr, i, SsSetNew(value)
End Sub

Sub SsArrDel(ByVal p As SsArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    @p.@arr.@arr[i] = SsFinal(@p.@arr.@arr[i])
    LnArrDel @p.arr, i
End Sub

Sub SsArrSort(ByVal p As SsArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter, compare As Long
    Local value As Long
    Local arr As Long Ptr
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count > 1 Then
        arr = @p.@arr.arr
        leftIndex = 1
        rightIndex = @p.@arr.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                Call Dword @p.compareCB Using SsCompareCB(@arr[i], value, @p.collation) To compare
                While compare < 0
                    Incr i
                    Call Dword @p.compareCB Using SsCompareCB(@arr[i], value, @p.collation) To compare
                Wend
                Call Dword @p.compareCB Using SsCompareCB(@arr[j], value, @p.collation) To compare
                While compare > 0
                    Decr j
                    Call Dword @p.compareCB Using SsCompareCB(@arr[j], value, @p.collation) To compare
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub SsArrUniqueSort(ByVal p As SsArr Ptr)
    'sort array and delete all duplicates
    Local i, compare As Long
    Local arr As Long Ptr
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    SsArrSort p
    arr = @p.@arr.arr
    For i = @p.@arr.count - 1 To 1 Step -1
        Call Dword @p.compareCB Using SsCompareCB(@arr[i + 1], @arr[i], @p.collation) To compare
        If compare = 0 Then SsArrDel p, i + 1
    Next i
End Sub

Sub SsArrReverse(ByVal p As SsArr Ptr)
    'reverse array
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    LnArrReverse @p.arr
End Sub

Function SsArrBinSearch(ByVal p As SsArr Ptr, ByRef value As String) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot, compare, temp As Long
    Local arr As Long Ptr
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    bot = 1
    top = @p.@arr.count
    arr = @p.@arr.arr
    temp = SsSetNew(value)
    While top >= bot
        i = bot + top
        Shift Right i, 1
        Call Dword @p.compareCB Using SsCompareCB(temp, @arr[i], @p.collation) To compare
        If compare > 0 Then
            bot = i + 1
        ElseIf compare < 0 Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
    SsFinal temp
End Function

Sub SsArrBinInsert(ByVal p As SsArr Ptr, ByRef value As String)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare, temp As Long
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    temp = SsSetNew(value)
    i = SsArrBinPos(p, temp, compare)
    If i = 0 Then
        SsArrAdd p, value
    ElseIf compare <= 0 Then
        SsArrIns p, i, value
    ElseIf i < @p.@arr.count Then
        SsArrIns p, i + 1, value
    Else
        SsArrAdd p, value
    End If
    temp = SsFinal(temp)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   String Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SsArrStkPush(ByVal p As SsArr Ptr, ByRef value As String)
    'push value on top of Stack
    SsArrAdd p, value
End Sub

Function SsArrStkPeek(ByVal p As SsArr Ptr) As String
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count Then Function = SsGet(@p.@arr.@arr[@p.@arr.count])
End Function

Function SsArrStkPop(ByVal p As SsArr Ptr) As String
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count Then
        Function = SsGet(@p.@arr.@arr[@p.@arr.count])
        SsArrDel p, @p.@arr.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   String Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SsArrQuePush(ByVal p As SsArr Ptr, ByRef value As String)
    'add value to end of Queue
    SsArrAdd p, value
End Sub

Function SsArrQuePeek(ByVal p As SsArr Ptr) As String
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count Then Function = SsGet(@p.@arr.@arr[1])
End Function

Function SsArrQuePop(ByVal p As SsArr Ptr) As String
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count Then
        Function = SsGet(@p.@arr.@arr[1])
        SsArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   String Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SsArrPushFirst(ByVal p As SsArr Ptr, ByRef value As String)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count Then SsArrIns p, 1, value Else SsArrAdd p, value
End Sub

Sub SsArrPushLast(ByVal p As SsArr Ptr, ByRef value As String)
    'add value to and of Deque (double-ended Queue)
    SsArrAdd p, value
End Sub

Function SsArrPeekFirst(ByVal p As SsArr Ptr) As String
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count Then Function = SsGet(@p.@arr.@arr[1])
End Function

Function SsArrPeekLast(ByVal p As SsArr Ptr) As String
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count Then Function = SsGet(@p.@arr.@arr[@p.@arr.count])
End Function

Function SsArrPopFirst(ByVal p As SsArr Ptr) As String
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count Then
        Function = SsGet(@p.@arr.@arr[1])
        SsArrDel p, 1
    End If
End Function

Function SsArrPopLast(ByVal p As SsArr Ptr) As String
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count Then
        Function = SsGet(@p.@arr.@arr[@p.@arr.count])
        SsArrDel p, @p.@arr.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SsArrClone(ByVal p As SsArr Ptr) As Long
    'duplicate container - return clone's handle
    Local i As Long
    Local clone As SsArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    clone = SsArrNew()  : If Err Then Exit Function
    If @p.@arr.count Then
        SsArrReDim clone, @p.@arr.count : If Err Then Exit Function
        For i = 1 To @p.@arr.count
            SsArrSet clone, i, SsArrGet(p, i)
        Next i
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SsArrStore(ByVal p As SsArr Ptr) As String
    'store container to string
    Local i, bytes, tot As Long
    Local s As String
    Local ps As SsStr Ptr
    Local pl As Long Ptr
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    If @p.@arr.count Then
        tot = 4
        For i = 1 To @p.@arr.count
            ps = @p.@arr.@arr[i]
            ExitF(ps=0, LibErrU)
            tot += 4 + @ps.count
        Next i
        s = Nul$(tot)
        pl = StrPtr(s)
        @pl = @p.@arr.count : Incr pl
        For i = 1 To @p.@arr.count
            ps = @p.@arr.@arr[i]
            bytes = @ps.count
            @pl = bytes : Incr pl
            If bytes Then
                Memory Copy @ps.mem, pl, bytes : pl += bytes
            End If
        Next i
    End If
    Function = s
End Function

Sub SsArrRestore(ByVal p As SsArr Ptr, ByVal s As String)
    'restore container from string
    Local i, items, bytes As Long
    Local pl As Long Ptr
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    SsArrClear p
    If strLen Then
        pl = StrPtr(s)
        items = @pl : Incr pl
        SsArrReDim p, items : If Err Then Exit Sub
        For i = 1 To items
            bytes = @pl : Incr pl
            If bytes Then
                SsArrSet p, i, Peek$(pl, bytes) : pl += bytes
            End If
        Next i
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SsArrFileStore(ByVal p As SsArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    s = SsArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsArrFileRestore(ByVal p As SsArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SsArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   String Builder Procedures
    '       use container's other procedures to Add/Insert/Modify/Remove string segments
    '----------------------------------------------------------------------------------------
'--

Function SsArrBuildStr(ByVal p As SsArr Ptr) As String
    'get complete string containing all string segments in container
    Local i, tot, h As Long
    Local s As String
    Local pt As SsStr Ptr
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    For i = 1 To @p.@arr.count
        pt = @p.@arr.@arr[i]
        tot += @pt.count
    Next i
    If tot Then
        s = Nul$(tot)
        h = StrPtr(s)
        For i = 1 To @p.@arr.count
            pt = @p.@arr.@arr[i]
            If @pt.count Then
                Memory Copy @pt.mem, h, @pt.count : h += @pt.count
            End If
        Next i
    End If
    Function = s
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Text Document Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SsArrTextLoad(ByVal p As SsArr Ptr, ByVal file As String)
    'load text file into container clearing current contents - Modifies Container Data
    Local f  As Long
    Local s As String
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    ExitS(IsFalse IsFile(file), LibErrF)
    SsArrClear p
    Try
        f = FreeFile
        Open file For Input As f
        While IsFalse Eof(f)
            Line Input# f, s
            SsArrAdd p, s
        Wend
    Catch
         ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsArrTextSave(ByVal p As SsArr Ptr, ByVal file As String)
    'save container to text file overwriting file's current contents
    Local i, f  As Long
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Output As f
        For i = 1 To @p.@arr.count
            Print# f, SsArrGet(p, i)
        Next i
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Function SsArrTextGet(ByVal p As SsArr Ptr) As String
    'get container's contents as a Text Document, appending CrLf to each item
    Local i, tot As Long
    Local ps As SsStr Ptr
    Local pb As Byte Ptr
    Local s As String
    ExitF(p=0 Or @p.tag<>SsArrTag, LibErrH)
    For i = 1 To @p.@arr.count
        ps = @p.@arr.@arr[i]
        tot += @ps.count + 2
    Next i
    If tot Then
        s = Nul$(tot)
        pb = StrPtr(s)
        For i = 1 To @p.@arr.count
            ps = @p.@arr.@arr[i]
            If @ps.count Then
                Memory Copy @ps.mem, pb, @ps.count : pb += @ps.count
            End If
            @pb = 13 : Incr pb
            @pb = 10 : Incr pb
        Next i
    End If
    Function = s
End Function

Sub SsArrTextSet(ByVal p As SsArr Ptr, ByVal s As String)
    'replace container's contents with a Text Document - each line = item in container (removing CrLf) - Modifies Container Data
    Local start, x, bytes As Long
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    SsArrClear p
    If Len(s) Then
        start = 1
        x = InStr(start, s, $CrLf)
        While x
            bytes = x - start
            SsArrAdd p, Mid$(s, start, bytes)
            start = x + 2
            x = InStr(start, s, $CrLf)
        Wend
        If start < Len(s) Then SsArrAdd p, Mid$(s, start)
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   File List Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SsArrGetFiles(ByVal p As SsArr Ptr, ByVal folder As String, ByVal mask As String)
    'load container with all files in folder matching the mask - Modifies Container Data
    'loads file names without path
    Local file, s As String
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    SsArrClear p
    ExitS(IsFalse IsFolder(folder), LibErrA)
    s = RTrim$(folder, "\") + "\" + mask
    file = Dir$(s)
    While Len(file)
        SsArrAdd p, file
        file = Dir$(Next)
    Wend
End Sub

Sub SsArrGetPaths(ByVal p As SsArr Ptr, ByVal folder As String, ByVal mask As String)
    'load container with all files in folder matching the mask - Modifies Container Data
    'loads full path
    Local file, s As String
    ExitS(p=0 Or @p.tag<>SsArrTag, LibErrH)
    SsArrClear p
    ExitS(IsFalse IsFolder(folder), LibErrA)
    folder = RTrim$(folder, "\") + "\"
    s = folder + mask
    file = Dir$(s)
    While Len(file)
        SsArrAdd p, folder + file
        file = Dir$(Next)
    Wend
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function SsArrBinPos(ByVal p As SsArr Ptr, ByVal hValue As Long, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Long Ptr
    compare = -1
    bot = 1
    top = @p.@arr.count
    arr = @p.@arr.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        Call Dword @p.compareCB Using SsCompareCB(hValue, @arr[i], @p.collation) To compare
        If compare > 0 Then
            bot = i + 1
        ElseIf compare < 0 Then
            top = i - 1
        Else
            Exit Loop
        End If
    Wend
    Function = i
End Function
