#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\Hash.inc"
#Include Once "C:\HLib3\String\SsStr.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\ExLst.inc"

'++
    '----------------------------------------------------------------------------------------
    '   String/Extended ~ Hash Table Container
    '       http://en.wikipedia.org/wiki/Hash_table
    '       Key must be unique String
    '       Key is case sensitive - no nulls
    '       Value stored/retrieved/removed using unique lookup Key
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = SsExHshNew() 'get handle for new container
    '       h = SsExHshFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Macro SsExHshTag = 1600944301
Type SsExHshNode
    next As SsExHshNode Ptr
    K As SsStr Ptr
    V As Extended
End Type
Type SsExHsh
    tag As Long
    count As Long
    cap As Long
    arr As Long Ptr
End Type

Function SsExHshNew(ByVal capacity As Long) As Long
    'allocate new container - return handle
    'capacity = number of expected Keys (minium 100)
    Local i As Long
    Local p As SsExHsh Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = SsExHshTag
    @p.cap = Max&(capacity, 10)
    @p.arr = MemAlloc(@p.cap * 4)
    ExitF(@p.arr=0, LibErrM)
    Function = p
End Function

Function SsExHshFinal(ByVal p As SsExHsh Ptr) As Long
    'free allocated container - return null
    Local i As Long
    Local n, x As SsExHshNode Ptr
    If p Then
        ExitF(@p.tag<>SsExHshTag, LibErrH)
        For i = 0 To @p.cap - 1
            n = @p.@arr[i] : @p.@arr[i] = 0
            While n
                x = n : n = @n.next
                SsFinal(@x.K)
                MemFree(x)
            Wend
        Next i
        @p.arr = MemFree(@p.arr)
    End If
End Function

Function SsExHshValidate(ByVal p As SsExHsh Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = SsExHshTag Then Function = @p.tag
End Function

Sub SsExHshClear(ByVal p As SsExHsh Ptr)
    'delete all data
    Local i As Long
    Local n, x As SsExHshNode Ptr
    ExitS(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    For i = 0 To  @p.cap - 1
        n = @p.@arr[i] : @p.@arr[i] = 0
        While n
            x = n : n = @n.next
            SsFinal(@x.K)
            MemFree(x)
        Wend
    Next i
    @p.count = 0
End Sub

Function SsExHshCount(ByVal p As SsExHsh Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    Function = @p.count
End Function

Function SsExHshCapGet(ByVal p As SsExHsh Ptr) As Long
    'get Hash Table capacity
    ExitF(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    Function = @p.cap
End Function

Sub SsExHshCapSet(ByVal p As SsExHsh Ptr, ByVal capacity As Long)
    'set Hash Table capacity
    'rebuild Hash Table with new capacity - data preserved
    'capacity should be about the same as number of stored Keys
    Local i, oldCap, newCap As Long
    Local arr As Long Ptr
    Local n, x As SsExHshNode Ptr
    ExitS(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    oldCap = @p.cap
    newCap = Max&(capacity, 10)
    arr = MemAlloc(newCap * 4)
    ExitS(arr=0, LibErrM)
    Swap @p.arr, arr
    @p.cap = newCap
    @p.count = 0
    For i = 0 To oldCap - 1
        n = @arr[i] : @arr[i] = 0
        While n
            x = n : n = @n.next
            SsExHshSet p, SsGet(@x.K), @x.V
            SsFinal(@x.K)
            MemFree(x)
        Wend
    Next i
    MemFree(arr)
End Sub

Sub SsExHshSet(ByVal p As SsExHsh Ptr, ByRef key As String, ByVal value As Extended, Opt ByVal DontReplace As Byte)
    'add Key/Value to Tash Table - Value replaced if Key exist unless DontReplace = True
    Local i, equal As Long
    Local ps As String Ptr : ps = StrPtr(key)
    Local n, nn As SsExHshNode Ptr
    ExitS(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    i = StrHash(ps, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    If n Then
        Do
            equal = SsEqual(@n.@K.mem, ps)
            If equal Then
                If IsFalse DontReplace Then @n.V = value
                Exit Sub
            ElseIf @n.next Then
                n = @n.next
            Else
                Exit Loop
            End If
        Loop
        nn = MemAlloc(SizeOf(SsExHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = SsSetNew(key)
        @nn.V = value
        @n.next = nn
        Incr @p.count
    Else
        nn = MemAlloc(SizeOf(SsExHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = SsSetNew(key)
        @nn.V = value
        @p.@arr[i] = nn
        Incr @p.count
    End If
End Sub

Function SsExHshGet(ByVal p As SsExHsh Ptr, ByRef key As String) As Extended
    'get Key's associated Value
    Local i, equal As Long
    Local ps As String Ptr : ps = StrPtr(key)
    Local n As SsExHshNode Ptr
    ExitF(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    i = StrHash(ps, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        equal = SsEqual(@n.@K.mem, ps)
        If equal Then
            Function = @n.V
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Function SsExHshGot(ByVal p As SsExHsh Ptr, ByRef key As String) As Byte
    'True/False if Key in Hash Table
    Local i, equal As Long
    Local ps As String Ptr : ps = StrPtr(key)
    Local n As SsExHshNode Ptr
    ExitF(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    i = StrHash(ps, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        equal = SsEqual(@n.@K.mem, ps)
        If equal Then
            Function = 1
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Sub SsExHshDel(ByVal p As SsExHsh Ptr, ByRef key As String)
    'remove Key and associated Value from Hash Table
    Local i, equal As Long
    Local ps As String Ptr : ps = StrPtr(key)
    Local prev, n As SsExHshNode Ptr
    ExitS(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    i = StrHash(ps, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    prev = 0
    n = @p.@arr[i]
    While n
        equal = SsEqual(@n.@K.mem, ps)
        If equal Then
            If prev Then
                @prev.next = @n.next
            Else
                @p.@arr[i] = @n.next
            End If
            @n.K = SsFinal(@n.K)
            MemFree(n)
            ExitS(@p.count=0, LibErrU)
            Decr @p.count
            Exit Loop
        End If
        prev = n
        n = @n.next
    Wend
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SsExHshClone(ByVal p As SsExHsh Ptr) As Long
    'create duplicate container - return handle to cloned container
    Local i As Long
    Local n As SsExHshNode Ptr
    Local clone As Long
    Err = 0
    ExitF(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    clone = SsExHshNew(@p.cap) : If Err Then Exit Function
    For i = 0 To @p.cap - 1
        n = @p.@arr[i]
        While n
            SsExHshSet clone, SsGet(@n.K), @n.V
            n = @n.next
        Wend
    Next i
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SsExHshStore(ByVal p As SsExHsh Ptr) As String
    'store container to string
    Local i As Long
    Local keys, vals, store As Long
    Local n As SsExHshNode Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    keys = SsLstNew() : If Err Then Exit Function
    vals = ExLstNew() : If Err Then Exit Function
    store = SsLstNew() : If Err Then Exit Function
    If @p.count Then
        For i = 0 To @p.cap - 1
            n = @p.@arr[i]
            While n
                SsLstAdd keys, SsGet(@n.K)
                ExLstAdd vals, @n.V
                n = @n.next
            Wend
        Next i
        SsLstAdd store, Mkl$(@p.cap)
        SsLstAdd store, SsLstStore(keys)
        SsLstAdd store, ExLstStore(vals)
        Function = SsLstStore(store)
    End If
    keys = SsLstFinal(keys)
    vals = ExLstFinal(vals)
    store = SsLstFinal(store)
End Function

Sub SsExHshRestore(ByVal p As SsExHsh Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, store As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    SsExHshClear p
    keys = SsLstNew() : If Err Then Exit Sub
    vals = ExLstNew() : If Err Then Exit Sub
    store = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore store, s : If Err Then Exit Sub
        ExitS(SsLstCount(store)<>3, LibErrU)
        SsExHshCapSet p, Cvl(SsLstPopFirst(store))
        SsLstRestore keys, SsLstPopFirst(store)
        ExLstRestore vals, SsLstPopFirst(store)
        ExitS(SsLstCount(keys)<>ExLstCount(vals), LibErrU)
        While SsLstCount(keys)
            SsExHshSet p, SsLstPopFirst(keys), ExLstPopFirst(vals)
        Wend
    End If
    keys = SsLstFinal(keys)
    vals = ExLstFinal(vals)
    store = SsLstFinal(store)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SsExHshFileStore(ByVal p As SsExHsh Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    s = SsExHshStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsExHshFileRestore(ByVal p As SsExHsh Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>SsExHshTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SsExHshRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub
