#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro DbLstNodeTag = 667850440
Macro DbLstTag = 1012397857
Type DbLstNode
    tag As Long
    next As DbLstNode Ptr
    prev As DbLstNode Ptr
    value As Double
End Type
Type DbLst
    tag As Long
    count As Long
    first As DbLstNode Ptr
    last As DbLstNode Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Double List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = DbLstNew() 'get handle for new container
    '       h = DbLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function DbLstNew() As Long
    'allocate new container - return handle
    Local pLst As DbLst Ptr
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = DbLstTag
    Function = pLst
End Function

Function DbLstFinal(ByVal pLst As DbLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>DbLstTag, LibErrH)
        DbLstClear pLst
        MemFree(pLst)
    End If
End Function

Function DbLstValidate(ByVal pLst As DbLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = DbLstTag Then Function = @pLst.tag
End Function

Sub DbLstClear(ByVal pLst As DbLst Ptr)
    'delete all data
    Local node As DbLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    While @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    Wend
    @pLst.last = 0
    @pLst.count = 0
End Sub

Function DbLstCount(ByVal pLst As DbLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    Function = @pLst.count
End Function

Sub DbLstAdd(ByVal pLst As DbLst Ptr, ByVal value As Double)
    'append Value to end of List
    Local node As DbLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.count Then
        node = DbLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    Else
        node = DbLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Sub DbLstIns(ByVal pLst As DbLst Ptr, ByVal value As Double)
    'insert Value at front of List
    Local node As DbLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.count Then
        node = DbLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    Else
        node = DbLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Function DbLstFirst(ByVal pLst As DbLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    Function = @pLst.first
End Function

Function DbLstLast(ByVal pLst As DbLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    Function = @pLst.last
End Function

Function DbLstNext(ByVal pNode As DbLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>DbLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function DbLstPrev(ByVal pNode As DbLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>DbLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function DbLstGet(ByVal pNode As DbLstNode Ptr) As Double
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>DbLstNodeTag, LibErrH)
    Function = @pNode.value
End Function

Sub DbLstSet(ByVal pNode As DbLstNode Ptr, ByVal value As Double)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>DbLstNodeTag, LibErrH)
    @pNode.value = value
End Sub

Sub DbLstInsPrev(ByVal pLst As DbLst Ptr, ByVal pNode As DbLstNode Ptr, ByVal value As Double)
    'insert Value before node
    Local node As DbLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>DbLstTag Or @pNode.tag<>DbLstNodeTag, LibErrH)
    If pNode = @pLst.first Then
        DbLstIns pLst, value
    Else
        ExitS(@pNode.prev=0, LibErrU)
        node = DbLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    End If
End Sub

Sub DbLstInsNext(ByVal pLst As DbLst Ptr, ByVal pNode As DbLstNode Ptr, ByVal value As Double)
    'insert Value after Cursor
    Local node As DbLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>DbLstTag Or @pNode.tag<>DbLstNodeTag, LibErrH)
    If pNode = @pLst.last Then
        DbLstAdd pLst, value
    Else
        ExitS(@pNode.next=0, LibErrU)
        node = DbLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    End If
End Sub

Sub DbLstDel(ByVal pLst As DbLst Ptr, ByVal pNode As DbLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>DbLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>DbLstNodeTag, LibErrH)
        If @pLst.first = pNode Then @pLst.first = @pNode.next
        If @pLst.last = pNode Then @pLst.last = @pNode.prev
        If @pNode.prev Then @pNode.@prev.next = @pNode.next
        If @pNode.next Then @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        Decr @pLst.count
        MemFree(pNode)
    End If
End Sub

Sub DbLstDelPrev(ByVal pLst As DbLst Ptr, ByVal pNode As DbLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>DbLstTag Or @pNode.tag<>DbLstNodeTag, LibErrH)
    DbLstDel pLst, @pNode.prev
End Sub

Sub DbLstDelNext(ByVal pLst As DbLst Ptr, ByVal pNode As DbLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>DbLstTag Or @pNode.tag<>DbLstNodeTag, LibErrH)
    DbLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub DbLstStkPush(ByVal pLst As DbLst Ptr, ByVal value As Double)
    'Push Value on Stack
    DbLstAdd pLst, value
End Sub

Function DbLstStkPeek(ByVal pLst As DbLst Ptr) As Double
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function DbLstStkPop(ByVal pLst As DbLst Ptr) As Double
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        DbLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub DbLstQuePush(ByVal pLst As DbLst Ptr, ByVal value As Double)
    'Add Value to end of Queue
    DbLstAdd pLst, value
End Sub

Function DbLstQuePeek(ByVal pLst As DbLst Ptr) As Double
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function DbLstQuePop(ByVal pLst As DbLst Ptr) As Double
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        DbLstDel pLst, @pLst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub DbLstPushFirst(ByVal pLst As DbLst Ptr, ByVal value As Double)
    'Add Value at front of container
    DbLstIns pLst, value
End Sub

Sub DbLstPushLast(ByVal pLst As DbLst Ptr, ByVal value As Double)
    'Add Value at end of container
    DbLstAdd pLst, value
End Sub

Function DbLstPeekFirst(ByVal pLst As DbLst Ptr) As Double
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function DbLstPeekLast(ByVal pLst As DbLst Ptr) As Double
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function DbLstPopFirst(ByVal pLst As DbLst Ptr) As Double
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        DbLstDel pLst, @pLst.first
    End If
End Function

Function DbLstPopLast(ByVal pLst As DbLst Ptr) As Double
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        DbLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function DbLstClone(ByVal pLst As DbLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As DbLst Ptr
    Local node As DbLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    Err = 0
    pClone = DbLstNew() : If Err Then Exit Function
    node = @pLst.first
    While node
        DbLstAdd pClone, @node.value
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function DbLstStore(ByVal pLst As DbLst Ptr) As String
    'store container to String
    Local s As String
    Local node As DbLstNode Ptr
    Local p As Double Ptr
    ExitF(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    If @pLst.count Then
        s = Nul$(@pLst.count * %SizeDb)
        p = StrPtr(s)
        node = @pLst.first
        While node
            @p = @node.value : Incr p
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub DbLstRestore(ByVal pLst As DbLst Ptr, ByRef s As String)
    'restore container from string
    Local items As Long
    Local p As Double Ptr
    ExitS(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    DbLstClear pLst
    If Len(s) Then
        p = StrPtr(s)
        items = Len(s) / %SizeDb
        While items
            DbLstAdd pLst, @p
            Incr p
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub DbLstFileStore(ByVal pLst As DbLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    s = DbLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub DbLstFileRestore(ByVal pLst As DbLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>DbLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        DbLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function DbLstNodeAlloc(ByVal pLst As DbLst Ptr) Private As Long
    Local node As DbLstNode Ptr
    node = MemAlloc(SizeOf(DbLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = DbLstNodeTag
    Incr @pLst.count
    Function = node
End Function
