#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro ExLstNodeTag = 1540788735
Macro ExLstTag = -2145892447
Type ExLstNode
    tag As Long
    next As ExLstNode Ptr
    prev As ExLstNode Ptr
    value As Extended
End Type
Type ExLst
    tag As Long
    count As Long
    first As ExLstNode Ptr
    last As ExLstNode Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Extended List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = ExLstNew() 'get handle for new container
    '       h = ExLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function ExLstNew() As Long
    'allocate new container - return handle
    Local pLst As ExLst Ptr
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = ExLstTag
    Function = pLst
End Function

Function ExLstFinal(ByVal pLst As ExLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>ExLstTag, LibErrH)
        ExLstClear pLst
        MemFree(pLst)
    End If
End Function

Function ExLstValidate(ByVal pLst As ExLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = ExLstTag Then Function = @pLst.tag
End Function

Sub ExLstClear(ByVal pLst As ExLst Ptr)
    'delete all data
    Local node As ExLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    While @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    Wend
    @pLst.last = 0
    @pLst.count = 0
End Sub

Function ExLstCount(ByVal pLst As ExLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    Function = @pLst.count
End Function

Sub ExLstAdd(ByVal pLst As ExLst Ptr, ByVal value As Extended)
    'append Value to end of List
    Local node As ExLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.count Then
        node = ExLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    Else
        node = ExLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Sub ExLstIns(ByVal pLst As ExLst Ptr, ByVal value As Extended)
    'insert Value at front of List
    Local node As ExLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.count Then
        node = ExLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    Else
        node = ExLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Function ExLstFirst(ByVal pLst As ExLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    Function = @pLst.first
End Function

Function ExLstLast(ByVal pLst As ExLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    Function = @pLst.last
End Function

Function ExLstNext(ByVal pNode As ExLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>ExLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function ExLstPrev(ByVal pNode As ExLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>ExLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function ExLstGet(ByVal pNode As ExLstNode Ptr) As Extended
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>ExLstNodeTag, LibErrH)
    Function = @pNode.value
End Function

Sub ExLstSet(ByVal pNode As ExLstNode Ptr, ByVal value As Extended)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>ExLstNodeTag, LibErrH)
    @pNode.value = value
End Sub

Sub ExLstInsPrev(ByVal pLst As ExLst Ptr, ByVal pNode As ExLstNode Ptr, ByVal value As Extended)
    'insert Value before node
    Local node As ExLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>ExLstTag Or @pNode.tag<>ExLstNodeTag, LibErrH)
    If pNode = @pLst.first Then
        ExLstIns pLst, value
    Else
        ExitS(@pNode.prev=0, LibErrU)
        node = ExLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    End If
End Sub

Sub ExLstInsNext(ByVal pLst As ExLst Ptr, ByVal pNode As ExLstNode Ptr, ByVal value As Extended)
    'insert Value after Cursor
    Local node As ExLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>ExLstTag Or @pNode.tag<>ExLstNodeTag, LibErrH)
    If pNode = @pLst.last Then
        ExLstAdd pLst, value
    Else
        ExitS(@pNode.next=0, LibErrU)
        node = ExLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    End If
End Sub

Sub ExLstDel(ByVal pLst As ExLst Ptr, ByVal pNode As ExLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>ExLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>ExLstNodeTag, LibErrH)
        If @pLst.first = pNode Then @pLst.first = @pNode.next
        If @pLst.last = pNode Then @pLst.last = @pNode.prev
        If @pNode.prev Then @pNode.@prev.next = @pNode.next
        If @pNode.next Then @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        Decr @pLst.count
        MemFree(pNode)
    End If
End Sub

Sub ExLstDelPrev(ByVal pLst As ExLst Ptr, ByVal pNode As ExLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>ExLstTag Or @pNode.tag<>ExLstNodeTag, LibErrH)
    ExLstDel pLst, @pNode.prev
End Sub

Sub ExLstDelNext(ByVal pLst As ExLst Ptr, ByVal pNode As ExLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>ExLstTag Or @pNode.tag<>ExLstNodeTag, LibErrH)
    ExLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub ExLstStkPush(ByVal pLst As ExLst Ptr, ByVal value As Extended)
    'Push Value on Stack
    ExLstAdd pLst, value
End Sub

Function ExLstStkPeek(ByVal pLst As ExLst Ptr) As Extended
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function ExLstStkPop(ByVal pLst As ExLst Ptr) As Extended
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        ExLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub ExLstQuePush(ByVal pLst As ExLst Ptr, ByVal value As Extended)
    'Add Value to end of Queue
    ExLstAdd pLst, value
End Sub

Function ExLstQuePeek(ByVal pLst As ExLst Ptr) As Extended
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function ExLstQuePop(ByVal pLst As ExLst Ptr) As Extended
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        ExLstDel pLst, @pLst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub ExLstPushFirst(ByVal pLst As ExLst Ptr, ByVal value As Extended)
    'Add Value at front of container
    ExLstIns pLst, value
End Sub

Sub ExLstPushLast(ByVal pLst As ExLst Ptr, ByVal value As Extended)
    'Add Value at end of container
    ExLstAdd pLst, value
End Sub

Function ExLstPeekFirst(ByVal pLst As ExLst Ptr) As Extended
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function ExLstPeekLast(ByVal pLst As ExLst Ptr) As Extended
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function ExLstPopFirst(ByVal pLst As ExLst Ptr) As Extended
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        ExLstDel pLst, @pLst.first
    End If
End Function

Function ExLstPopLast(ByVal pLst As ExLst Ptr) As Extended
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        ExLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function ExLstClone(ByVal pLst As ExLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As ExLst Ptr
    Local node As ExLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    Err = 0
    pClone = ExLstNew() : If Err Then Exit Function
    node = @pLst.first
    While node
        ExLstAdd pClone, @node.value
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function ExLstStore(ByVal pLst As ExLst Ptr) As String
    'store container to String
    Local s As String
    Local node As ExLstNode Ptr
    Local p As Extended Ptr
    ExitF(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    If @pLst.count Then
        s = Nul$(@pLst.count * %SizeEx)
        p = StrPtr(s)
        node = @pLst.first
        While node
            @p = @node.value : Incr p
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub ExLstRestore(ByVal pLst As ExLst Ptr, ByRef s As String)
    'restore container from string
    Local items As Long
    Local p As Extended Ptr
    ExitS(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    ExLstClear pLst
    If Len(s) Then
        p = StrPtr(s)
        items = Len(s) / %SizeEx
        While items
            ExLstAdd pLst, @p
            Incr p
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub ExLstFileStore(ByVal pLst As ExLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    s = ExLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub ExLstFileRestore(ByVal pLst As ExLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>ExLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        ExLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function ExLstNodeAlloc(ByVal pLst As ExLst Ptr) Private As Long
    Local node As ExLstNode Ptr
    node = MemAlloc(SizeOf(ExLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = ExLstNodeTag
    Incr @pLst.count
    Function = node
End Function
