#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\List\CxLst.inc"
#Include Once "C:\HLib3\List\SsLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'Long/CurrencyX ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = LnCxTreNew() 'get handle for new container
    '   h = LnCxTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro LnCxTreNodeTag = -1900523422
Macro LnCxTreTag = 719713421
Type LnCxTreNode
    tag As Long
    P As LnCxTreNode Ptr
    L As LnCxTreNode Ptr
    R As LnCxTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As CurrencyX
End Type
Type LnCxTre
    tag As Long
    count As Long
    root As LnCxTreNode Ptr
End Type

Function LnCxTreNew() As Long
    'allocate new container - return handle
    Local p As LnCxTre Ptr
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = LnCxTreTag
    Function = p
End Function

Function LnCxTreFinal(ByVal pTree As LnCxTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>LnCxTreTag, LibErrH)
        LnCxTreClear pTree
        MemFree(pTree)
    End If
End Function

Function LnCxTreValidate(ByVal pTree As LnCxTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = LnCxTreTag Then Function = @pTree.tag
End Function

Sub LnCxTreClear(ByVal pTree As LnCxTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As LnCxTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = LnCxTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = LnCxTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            LnCxTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function LnCxTreCount(ByVal pTree As LnCxTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub LnCxTreSet(ByVal pTree As LnCxTre Ptr, ByVal key As Long, ByVal value As CurrencyX, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local n As LnCxTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    If @pTree.root Then
        n = @pTree.root
        While 1
            If key > @n.K Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = LnCxTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    @n.@R.K = key
                    @n.@R.V = value
                    LnCxTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf key < @n.K Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = LnCxTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    @n.@L.K = key
                    @n.@L.V = value
                    LnCxTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then @n.V = value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = LnCxTreAllocNode(pTree) : If Err Then Exit Sub
        @pTree.@root.K = key
        @pTree.@root.V = value
        @pTree.count = 1
    End If
End Sub

Function LnCxTreGet(ByVal pTree As LnCxTre Ptr, ByVal key As Long) As CurrencyX
    'get Key's associated Value
    Register k As Long : k = key
    Local n As LnCxTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = @n.V
            Exit Loop
        End If
    Wend
End Function

Function LnCxTreGot(ByVal pTree As LnCxTre Ptr, ByVal key As Long) As Long
    'True/False if Key exist
    Register k As Long : k = key
    Local n As LnCxTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
End Function

Sub LnCxTreDel(ByVal pTree As LnCxTre Ptr, ByVal key As Long)
    'remove Key and associated Value
    Local pNode As LnCxTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    pNode = LnCxTreGot(pTree, key)
    If pNode Then
        LnCxTreRemoveNode(pTree, pNode)
    End If
End Sub

Function LnCxTreFirst(ByVal pTree As LnCxTre Ptr) As Long
    'get handle to first node in tree
    Local n As LnCxTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function LnCxTreLast(ByVal pTree As LnCxTre Ptr) As Long
    'get handle to last node in tree
    Local n As LnCxTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function LnCxTreNext(ByVal pNode As LnCxTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As LnCxTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnCxTreNodeTag, LibErrH)
        minR = LnCxTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = LnCxTreParentGreater(pNode)
        End If
    End If
End Function

Function LnCxTrePrev(ByVal pNode As LnCxTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As LnCxTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnCxTreNodeTag, LibErrH)
        maxL = LnCxTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = LnCxTreParentLesser(pNode)
        End If
    End If
End Function

Function LnCxTreGetKey(ByVal pNode As LnCxTreNode Ptr) As Long
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>LnCxTreNodeTag, LibErrH)
    Function = @pNode.K
End Function

Function LnCxTreGetVal(ByVal pNode As LnCxTreNode Ptr) As CurrencyX
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>LnCxTreNodeTag, LibErrH)
    Function = @pNode.V
End Function

Sub LnCxTreSetVal(ByVal pNode As LnCxTreNode Ptr, ByVal value As CurrencyX)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>LnCxTreNodeTag, LibErrH)
    @pNode.V = value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnCxTreClone(ByVal pTree As LnCxTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    clone = LnCxTreNew() : If Err Then Exit Function
    h = LnCxTreFirst(pTree)
    While h
        LnCxTreSet clone, LnCxTreGetKey(h), LnCxTreGetVal(h)
        h = LnCxTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnCxTreStore(ByVal pTree As LnCxTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    keys = LnLstNew() : If Err Then Exit Function
    vals = CxLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = LnCxTreFirst(pTree)
        While h
            LnLstAdd keys, LnCxTreGetKey(h)
            CxLstAdd vals, LnCxTreGetVal(h)
            h = LnCxTreNext(h)
        Wend
        SsLstAdd stor, LnLstStore(keys)
        SsLstAdd stor, CxLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = LnLstFinal(keys)
    vals = CxLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub LnCxTreRestore(ByVal pTree As LnCxTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    LnCxTreClear pTree
    keys = LnLstNew() : If Err Then Exit Sub
    vals = CxLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        LnLstRestore keys, SsLstPopFirst(stor)
        CxLstRestore vals, SsLstPopFirst(stor)
        ExitS(LnLstCount(keys)<>CxLstCount(vals), LibErrU)
        While LnLstCount(keys)
            LnCxTreSet pTree, LnLstPopFirst(keys), CxLstPopFirst(vals)
        Wend
    End If
    keys = LnLstFinal(keys)
    vals = CxLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnCxTreFileStore(ByVal pTree As LnCxTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    s = LnCxTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnCxTreFileRestore(ByVal pTree As LnCxTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnCxTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnCxTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub LnCxTreRemoveNode(ByVal p As LnCxTre Ptr, ByVal n As LnCxTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As LnCxTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, LnCxTreMaxLeft(n), LnCxTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        LnCxTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = LnCxTreFreeNode(p, n)
        LnCxTreBalanceBranch p, nP
    End If
End Sub

Function LnCxTreAllocNode(ByVal p As LnCxTre Ptr) Private As Long
    Local n As LnCxTreNode Ptr
    n = MemAlloc(SizeOf(LnCxTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = LnCxTreNodeTag
    Incr @p.count
    Function = n
End Function

Function LnCxTreFreeNode(ByVal p As LnCxTre Ptr, ByVal n As LnCxTreNode Ptr) As Long
    If n Then
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub LnCxTreBalanceBranch(ByVal p As LnCxTre Ptr, ByVal n As LnCxTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = LnCxTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = LnCxTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function LnCxTreMaxLeft(ByVal n As LnCxTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function LnCxTreMinRight(ByVal n As LnCxTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function LnCxTreParentGreater(ByVal n As LnCxTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnCxTreParentLesser(ByVal n As LnCxTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnCxTreRotateLeft(ByVal p As LnCxTre Ptr, ByVal n As LnCxTreNode Ptr) Private As Long
    Local nR, nRL As LnCxTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function LnCxTreRotateRight(ByVal p As LnCxTre Ptr, ByVal n As LnCxTreNode Ptr) Private As Long
    Local nL, nLR As LnCxTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
