#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\List\DbLst.inc"
#Include Once "C:\HLib3\List\SsLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'Long/Double ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = LnDbTreNew() 'get handle for new container
    '   h = LnDbTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro LnDbTreNodeTag = 1255889709
Macro LnDbTreTag = 1012397857
Type LnDbTreNode
    tag As Long
    P As LnDbTreNode Ptr
    L As LnDbTreNode Ptr
    R As LnDbTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As Double
End Type
Type LnDbTre
    tag As Long
    count As Long
    root As LnDbTreNode Ptr
End Type

Function LnDbTreNew() As Long
    'allocate new container - return handle
    Local p As LnDbTre Ptr
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = LnDbTreTag
    Function = p
End Function

Function LnDbTreFinal(ByVal pTree As LnDbTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>LnDbTreTag, LibErrH)
        LnDbTreClear pTree
        MemFree(pTree)
    End If
End Function

Function LnDbTreValidate(ByVal pTree As LnDbTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = LnDbTreTag Then Function = @pTree.tag
End Function

Sub LnDbTreClear(ByVal pTree As LnDbTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As LnDbTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = LnDbTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = LnDbTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            LnDbTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function LnDbTreCount(ByVal pTree As LnDbTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub LnDbTreSet(ByVal pTree As LnDbTre Ptr, ByVal key As Long, ByVal value As Double, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local n As LnDbTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    If @pTree.root Then
        n = @pTree.root
        While 1
            If key > @n.K Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = LnDbTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    @n.@R.K = key
                    @n.@R.V = value
                    LnDbTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf key < @n.K Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = LnDbTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    @n.@L.K = key
                    @n.@L.V = value
                    LnDbTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then @n.V = value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = LnDbTreAllocNode(pTree) : If Err Then Exit Sub
        @pTree.@root.K = key
        @pTree.@root.V = value
        @pTree.count = 1
    End If
End Sub

Function LnDbTreGet(ByVal pTree As LnDbTre Ptr, ByVal key As Long) As Double
    'get Key's associated Value
    Register k As Long : k = key
    Local n As LnDbTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = @n.V
            Exit Loop
        End If
    Wend
End Function

Function LnDbTreGot(ByVal pTree As LnDbTre Ptr, ByVal key As Long) As Long
    'True/False if Key exist
    Register k As Long : k = key
    Local n As LnDbTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
End Function

Sub LnDbTreDel(ByVal pTree As LnDbTre Ptr, ByVal key As Long)
    'remove Key and associated Value
    Local pNode As LnDbTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    pNode = LnDbTreGot(pTree, key)
    If pNode Then
        LnDbTreRemoveNode(pTree, pNode)
    End If
End Sub

Function LnDbTreFirst(ByVal pTree As LnDbTre Ptr) As Long
    'get handle to first node in tree
    Local n As LnDbTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function LnDbTreLast(ByVal pTree As LnDbTre Ptr) As Long
    'get handle to last node in tree
    Local n As LnDbTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function LnDbTreNext(ByVal pNode As LnDbTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As LnDbTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnDbTreNodeTag, LibErrH)
        minR = LnDbTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = LnDbTreParentGreater(pNode)
        End If
    End If
End Function

Function LnDbTrePrev(ByVal pNode As LnDbTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As LnDbTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnDbTreNodeTag, LibErrH)
        maxL = LnDbTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = LnDbTreParentLesser(pNode)
        End If
    End If
End Function

Function LnDbTreGetKey(ByVal pNode As LnDbTreNode Ptr) As Long
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>LnDbTreNodeTag, LibErrH)
    Function = @pNode.K
End Function

Function LnDbTreGetVal(ByVal pNode As LnDbTreNode Ptr) As Double
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>LnDbTreNodeTag, LibErrH)
    Function = @pNode.V
End Function

Sub LnDbTreSetVal(ByVal pNode As LnDbTreNode Ptr, ByVal value As Double)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>LnDbTreNodeTag, LibErrH)
    @pNode.V = value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnDbTreClone(ByVal pTree As LnDbTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    clone = LnDbTreNew() : If Err Then Exit Function
    h = LnDbTreFirst(pTree)
    While h
        LnDbTreSet clone, LnDbTreGetKey(h), LnDbTreGetVal(h)
        h = LnDbTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnDbTreStore(ByVal pTree As LnDbTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    keys = LnLstNew() : If Err Then Exit Function
    vals = DbLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = LnDbTreFirst(pTree)
        While h
            LnLstAdd keys, LnDbTreGetKey(h)
            DbLstAdd vals, LnDbTreGetVal(h)
            h = LnDbTreNext(h)
        Wend
        SsLstAdd stor, LnLstStore(keys)
        SsLstAdd stor, DbLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = LnLstFinal(keys)
    vals = DbLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub LnDbTreRestore(ByVal pTree As LnDbTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    LnDbTreClear pTree
    keys = LnLstNew() : If Err Then Exit Sub
    vals = DbLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        LnLstRestore keys, SsLstPopFirst(stor)
        DbLstRestore vals, SsLstPopFirst(stor)
        ExitS(LnLstCount(keys)<>DbLstCount(vals), LibErrU)
        While LnLstCount(keys)
            LnDbTreSet pTree, LnLstPopFirst(keys), DbLstPopFirst(vals)
        Wend
    End If
    keys = LnLstFinal(keys)
    vals = DbLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnDbTreFileStore(ByVal pTree As LnDbTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    s = LnDbTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnDbTreFileRestore(ByVal pTree As LnDbTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnDbTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnDbTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub LnDbTreRemoveNode(ByVal p As LnDbTre Ptr, ByVal n As LnDbTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As LnDbTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, LnDbTreMaxLeft(n), LnDbTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        LnDbTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = LnDbTreFreeNode(p, n)
        LnDbTreBalanceBranch p, nP
    End If
End Sub

Function LnDbTreAllocNode(ByVal p As LnDbTre Ptr) Private As Long
    Local n As LnDbTreNode Ptr
    n = MemAlloc(SizeOf(LnDbTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = LnDbTreNodeTag
    Incr @p.count
    Function = n
End Function

Function LnDbTreFreeNode(ByVal p As LnDbTre Ptr, ByVal n As LnDbTreNode Ptr) As Long
    If n Then
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub LnDbTreBalanceBranch(ByVal p As LnDbTre Ptr, ByVal n As LnDbTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = LnDbTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = LnDbTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function LnDbTreMaxLeft(ByVal n As LnDbTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function LnDbTreMinRight(ByVal n As LnDbTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function LnDbTreParentGreater(ByVal n As LnDbTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnDbTreParentLesser(ByVal n As LnDbTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnDbTreRotateLeft(ByVal p As LnDbTre Ptr, ByVal n As LnDbTreNode Ptr) Private As Long
    Local nR, nRL As LnDbTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function LnDbTreRotateRight(ByVal p As LnDbTre Ptr, ByVal n As LnDbTreNode Ptr) Private As Long
    Local nL, nLR As LnDbTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
