#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\List\QdLst.inc"
#Include Once "C:\HLib3\List\SsLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'Long/Quad ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = LnQdTreNew() 'get handle for new container
    '   h = LnQdTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro LnQdTreNodeTag = -1900523422
Macro LnQdTreTag = 719713421
Type LnQdTreNode
    tag As Long
    P As LnQdTreNode Ptr
    L As LnQdTreNode Ptr
    R As LnQdTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As Quad
End Type
Type LnQdTre
    tag As Long
    count As Long
    root As LnQdTreNode Ptr
End Type

Function LnQdTreNew() As Long
    'allocate new container - return handle
    Local p As LnQdTre Ptr
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = LnQdTreTag
    Function = p
End Function

Function LnQdTreFinal(ByVal pTree As LnQdTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>LnQdTreTag, LibErrH)
        LnQdTreClear pTree
        MemFree(pTree)
    End If
End Function

Function LnQdTreValidate(ByVal pTree As LnQdTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = LnQdTreTag Then Function = @pTree.tag
End Function

Sub LnQdTreClear(ByVal pTree As LnQdTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As LnQdTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = LnQdTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = LnQdTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            LnQdTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function LnQdTreCount(ByVal pTree As LnQdTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub LnQdTreSet(ByVal pTree As LnQdTre Ptr, ByVal key As Long, ByVal value As Quad, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local n As LnQdTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    If @pTree.root Then
        n = @pTree.root
        While 1
            If key > @n.K Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = LnQdTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    @n.@R.K = key
                    @n.@R.V = value
                    LnQdTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf key < @n.K Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = LnQdTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    @n.@L.K = key
                    @n.@L.V = value
                    LnQdTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then @n.V = value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = LnQdTreAllocNode(pTree) : If Err Then Exit Sub
        @pTree.@root.K = key
        @pTree.@root.V = value
        @pTree.count = 1
    End If
End Sub

Function LnQdTreGet(ByVal pTree As LnQdTre Ptr, ByVal key As Long) As Quad
    'get Key's associated Value
    Register k As Long : k = key
    Local n As LnQdTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = @n.V
            Exit Loop
        End If
    Wend
End Function

Function LnQdTreGot(ByVal pTree As LnQdTre Ptr, ByVal key As Long) As Long
    'True/False if Key exist
    Register k As Long : k = key
    Local n As LnQdTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
End Function

Sub LnQdTreDel(ByVal pTree As LnQdTre Ptr, ByVal key As Long)
    'remove Key and associated Value
    Local pNode As LnQdTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    pNode = LnQdTreGot(pTree, key)
    If pNode Then
        LnQdTreRemoveNode(pTree, pNode)
    End If
End Sub

Function LnQdTreFirst(ByVal pTree As LnQdTre Ptr) As Long
    'get handle to first node in tree
    Local n As LnQdTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function LnQdTreLast(ByVal pTree As LnQdTre Ptr) As Long
    'get handle to last node in tree
    Local n As LnQdTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function LnQdTreNext(ByVal pNode As LnQdTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As LnQdTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnQdTreNodeTag, LibErrH)
        minR = LnQdTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = LnQdTreParentGreater(pNode)
        End If
    End If
End Function

Function LnQdTrePrev(ByVal pNode As LnQdTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As LnQdTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnQdTreNodeTag, LibErrH)
        maxL = LnQdTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = LnQdTreParentLesser(pNode)
        End If
    End If
End Function

Function LnQdTreGetKey(ByVal pNode As LnQdTreNode Ptr) As Long
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>LnQdTreNodeTag, LibErrH)
    Function = @pNode.K
End Function

Function LnQdTreGetVal(ByVal pNode As LnQdTreNode Ptr) As Quad
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>LnQdTreNodeTag, LibErrH)
    Function = @pNode.V
End Function

Sub LnQdTreSetVal(ByVal pNode As LnQdTreNode Ptr, ByVal value As Quad)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>LnQdTreNodeTag, LibErrH)
    @pNode.V = value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnQdTreClone(ByVal pTree As LnQdTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    clone = LnQdTreNew() : If Err Then Exit Function
    h = LnQdTreFirst(pTree)
    While h
        LnQdTreSet clone, LnQdTreGetKey(h), LnQdTreGetVal(h)
        h = LnQdTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnQdTreStore(ByVal pTree As LnQdTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    keys = LnLstNew() : If Err Then Exit Function
    vals = QdLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = LnQdTreFirst(pTree)
        While h
            LnLstAdd keys, LnQdTreGetKey(h)
            QdLstAdd vals, LnQdTreGetVal(h)
            h = LnQdTreNext(h)
        Wend
        SsLstAdd stor, LnLstStore(keys)
        SsLstAdd stor, QdLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = LnLstFinal(keys)
    vals = QdLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub LnQdTreRestore(ByVal pTree As LnQdTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    LnQdTreClear pTree
    keys = LnLstNew() : If Err Then Exit Sub
    vals = QdLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        LnLstRestore keys, SsLstPopFirst(stor)
        QdLstRestore vals, SsLstPopFirst(stor)
        ExitS(LnLstCount(keys)<>QdLstCount(vals), LibErrU)
        While LnLstCount(keys)
            LnQdTreSet pTree, LnLstPopFirst(keys), QdLstPopFirst(vals)
        Wend
    End If
    keys = LnLstFinal(keys)
    vals = QdLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnQdTreFileStore(ByVal pTree As LnQdTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    s = LnQdTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnQdTreFileRestore(ByVal pTree As LnQdTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnQdTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnQdTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub LnQdTreRemoveNode(ByVal p As LnQdTre Ptr, ByVal n As LnQdTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As LnQdTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, LnQdTreMaxLeft(n), LnQdTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        LnQdTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = LnQdTreFreeNode(p, n)
        LnQdTreBalanceBranch p, nP
    End If
End Sub

Function LnQdTreAllocNode(ByVal p As LnQdTre Ptr) Private As Long
    Local n As LnQdTreNode Ptr
    n = MemAlloc(SizeOf(LnQdTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = LnQdTreNodeTag
    Incr @p.count
    Function = n
End Function

Function LnQdTreFreeNode(ByVal p As LnQdTre Ptr, ByVal n As LnQdTreNode Ptr) As Long
    If n Then
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub LnQdTreBalanceBranch(ByVal p As LnQdTre Ptr, ByVal n As LnQdTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = LnQdTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = LnQdTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function LnQdTreMaxLeft(ByVal n As LnQdTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function LnQdTreMinRight(ByVal n As LnQdTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function LnQdTreParentGreater(ByVal n As LnQdTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnQdTreParentLesser(ByVal n As LnQdTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnQdTreRotateLeft(ByVal p As LnQdTre Ptr, ByVal n As LnQdTreNode Ptr) Private As Long
    Local nR, nRL As LnQdTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function LnQdTreRotateRight(ByVal p As LnQdTre Ptr, ByVal n As LnQdTreNode Ptr) Private As Long
    Local nL, nLR As LnQdTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
