#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\String\SsStr.inc"
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\List\BtLst.inc"
#Include Once "C:\HLib3\List\SsLst.inc"

'++
    'String/String Key/Value Range Trie Tree Container
    '   !!! fast as a Hash Table !!!
    '
    '   a "trie" is both an Hash and a Tree container
    '   - as fast as a hash
    '   - always in Key order
    '   - unlike a hash, doesn't have to rebuild for an optimal capacity
    '   - "trie" comes from the word "re(trie)ve";  http://en.wikipedia.org/wiki/Trie
    '
    '   Key/Value data structure
    '       Value stored/retrieved/removed using a unique lookup Key
    '   Key = string - any length - NULL not allowed
    '   Key case ignored (compared lower case)
    '
    '   Value = string - any length - any type of data
    '
    '   Prefix search
    '       a Trie Tree is the king of Prefix search; IntelliSense like lookup
    '
    '   Speed
    '       a Trie is a radix type data structure
    '       essentially, a Trie is an array of arrays; each character being the index to the next character's node
    '
    '   Keys aren't stored in Tree
    '-------------------------------------------------------------------
    '   Keys lay on top of each other until a difference is encountered
    '   Keys = (cat, cats, catastrophe, catastrophes, cattle, cattleman, cattlemen, catch, catches, caught)
    '
    '   c
    '   a
    '   t*--------------u
    '   a--c--s*--t     g
    '   s  h*     l     h
    '   t  e      e*    t*
    '   r  s*     m
    '   o         a--e
    '   p         n* n*
    '   h
    '   e*
    '   s*
    '-------------------------------------------------------------------
    '
    '   Compression Algorithm
    '       a Trie is used in some compression algorithms because words lay on top of each other
    '
    '   "Range" Trie Tree
    '       the "Range" Trie is my innovation
    '       instead of each node holding space for all possible characters;
    '       each node only reserves space for the narrowest "range" possible
    '           in most cases, it will be only one character
    '
    '   Note:
    '       the code is public domain, but the concept of a dynamic range trie node isn't placed in public domain
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = SsSsTriNew() 'get handle for new container
    '   h = SsSsTriFinal(h) 'free handle before it goes out of scope
'--

%SsSsTriDn = 1
%SsSsTriRt = 2
%SsSsTriUp = 3

Macro SsSsTriTag = 513674861
Type SsSsTriNode
    prnt As SsSsTriNode Ptr
    char As Byte
    low As Byte
    high As Byte
    count As Byte
    arr As Long Ptr
    value As Long
End Type
Type SsSsTri
    tag As Long
    count As Long
    root As SsSsTriNode Ptr
    cursor As SsSsTriNode Ptr
    ndx As Long
    way As Long
    prefix As SsSsTriNode Ptr
End Type

Function SsSsTriNew() As Long
    'allocate new container - return handle
    Local p As SsSsTri Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = SsSsTriTag
    Function = p
End Function

Function SsSsTriFinal(ByVal p As SsSsTri Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>SsSsTriTag, LibErrH)
        SsSsTriClear p
        MemFree(p)
    End If
End Function

Function SsSsTriValidate(ByVal p As SsSsTri Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = SsSsTriTag Then Function = @p.tag
End Function

Sub SsSsTriClear(ByVal p As SsSsTri Ptr)
    'delete all data
    ExitS(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    @p.root = SsSsTriNodeFree(@p.root)
    @p.count = 0
    @p.cursor = 0
    @p.ndx = -1
    @p.prefix = 0
End Sub

Function SsSsTriCount(ByVal p As SsSsTri Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    Function = @p.count
End Function

Sub SsSsTriSet(ByVal p As SsSsTri Ptr, ByRef key As String, ByRef value As String, Opt ByVal DontReplace As Long)
    'add Key/Value to container - Value replaced if Key exist unless DontReplace = True
    Local c, x As Long
    Local n As SsSsTriNode Ptr
    Local k As Byte Ptr
    Err = 0
    ExitS(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    ExitS(IsNull(key), LibErrA)
    If @p.root = 0 Then @p.root = SsSsTriNodeAlloc(0, 0) : If Err Then Exit Sub
    n = @p.root
    k = StrPtr(key)
    While @k
        c = @k
        If c >= 97 And c <= 122 Then c -= 32
        x = SsSsTriNodeAddToRange(n, c)
        If Err Then Exit Sub
        If @n.@arr[x] = 0 Then @n.@arr[x] = SsSsTriNodeAlloc(n, c)
        If Err Then Exit Sub
        n = @n.@arr[x]
        Incr k
    Wend
    ExitS(n=0, LibErrU)
    'all of key's characters are now in tree
    If @n.value Then 'already in tree
        If IsFalse DontReplace Then
            SsSet @n.value, value
        End If
    Else
        @n.value = SsSetNew(value)
        Incr @p.count
    End If
End Sub

Function SsSsTriGet(ByVal p As SsSsTri Ptr, ByRef key As String) As String
    'get Key's associated Value
    Local c As Long
    Local n As SsSsTriNode Ptr
    Local k As Byte Ptr
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    k = StrPtr(key)
    If k Then
        c = @k
        n = @p.root
        While n And c
            If c >= 97 And c <= 122 Then c -= 32
            If c < @n.low Or c > @n.high Then Exit Function
            n = @n.@arr[c - @n.low]
            Incr k : c = @k
        Wend
    End If
    If n And @n.value Then Function = SsGet(@n.value)
End Function

Function SsSsTriGot(ByVal p As SsSsTri Ptr, ByRef key As String) As Long
    'True/False if Key exist
    Local c As Long
    Local n As SsSsTriNode Ptr
    Local k As Byte Ptr
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    k = StrPtr(key)
    If k Then
        c = @k
        n = @p.root
        While n And c
            If c >= 97 And c <= 122 Then c -= 32
            If c < @n.low Or c > @n.high Then Exit Function
            n = @n.@arr[c - @n.low]
            Incr k : c = @k
        Wend
    End If
    If n And @n.value Then Function = n
End Function

Sub SsSsTriDel(ByVal p As SsSsTri Ptr, ByRef key As String)
    'remove Key and associated Value
    Local c As Long
    Local n, prnt As SsSsTriNode Ptr
    Local k As Byte Ptr
    Err = 0
    ExitS(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    If Err Then Exit Sub
    k = StrPtr(key)
    If k Then
        n = @p.root
        While n And @k
            c = @k
            If c >= 97 And c <= 122 Then c -= 32
            If c < @n.low Or c > @n.high Then Exit Sub
            n = @n.@arr[c - @n.low]
            Incr k
        Wend
        If n And @n.value Then
            @n.value = SsFinal(@n.value)
            Decr @p.count
            While n
                prnt = @n.prnt
                If @n.count Or @n.value Then Exit Sub
                SsSsTriNodeDisconnect p, n
                SsSsTriNodeFree(n)
                n = prnt
            Wend
            If @p.count = 0 Then SsSsTriClear p
        End If
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Key Order Cursor
    '----------------------------------------------------------------------------------------
'--

Function SsSsTriFirstKey(ByVal p As SsSsTri Ptr) As Long
    'move to first Key in tree - True/False success
    'Keys stored in alphabetical order - case ignored
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    @p.cursor = 0
    @p.prefix = 0
    If @p.root Then
        @p.cursor = @p.root
        @p.ndx = -1 'incr by %SsSsTriRt
        @p.way = %SsSsTriRt
        Function = SsSsTriNextKey(p)
    End If
End Function

Function SsSsTriNextKey(ByVal p As SsSsTri Ptr) As Long
    'move to next Key in tree - True/False success
    'Keys stored in alphabetical order - case ignored
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    If @p.count Then
        While @p.cursor
            Select Case As Const @p.way
            Case %SsSsTriDn
                If @p.@cursor.count = 0 Or @p.ndx >= @p.@cursor.count Then
                    @p.way = %SsSsTriUp 'can't go down or right - go up
                ElseIf @p.@cursor.@arr[@p.ndx] = 0 Then
                    @p.way = %SsSsTriRt 'no child - can't go down - go right
                Else 'go down
                    @p.cursor = @p.@cursor.@arr[@p.ndx]
                    @p.ndx = 0
                    @p.way = %SsSsTriDn
                    If @p.@cursor.value Then
                        Function = 1 : Exit Function
                    End If
                End If
            Case %SsSsTriRt
                Incr @p.ndx 'go right
                If @p.ndx >= @p.@cursor.count Then
                    @p.way = %SsSsTriUp 'can't go right or down
                Else
                    @p.way = %SsSsTriDn
                End If
            Case %SsSsTriUp
                If @p.@cursor.prnt Then @p.ndx = @p.@cursor.char - @p.@cursor.@prnt.low
                @p.cursor = @p.@cursor.prnt 'will exit loop if cursor on root node
                @p.way = %SsSsTriRt
            End Select
        Wend
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Prefix Cursor - find all Keys starting with a "prefix" - IntelliSense like search
    '----------------------------------------------------------------------------------------
'--

Function SsSsTriFirstPrefix(ByVal p As SsSsTri Ptr, ByRef keyPrefix As String) As Long
    'move cursor to first Key starting with keyPrefix - True/False success
    Local c, x As Long
    Local k As Byte Ptr
    Local n As SsSsTriNode Ptr
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    @p.cursor = 0
    @p.prefix = 0
    k = StrPtr(keyPrefix)
    n = @p.root
    If k Then
        While @k And n
            c = @k
            If c >= 97 And c <= 122 Then c -= 32
            If c < @n.low Or c > @n.high Then Exit Function
            x = c - @n.low
            Incr k
            n = @n.@arr[x]
        Wend
        If n Then
            @p.prefix = @n.prnt
            @p.cursor = n
            @p.ndx = -1
            @p.way = %SsSsTriRt
            If @n.value Then
                Function = 1
            Else
                Function = SsSsTriNextPrefix(p)
            End If
        End If
    End If
End Function

Function SsSsTriNextPrefix(ByVal p As SsSsTri Ptr) As Long
    'move to next Key in tree - True/False success
    'Keys stored in alphabetical order - case ignored
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    If @p.count Then
        While @p.cursor
            Select Case As Const @p.way
            Case %SsSsTriDn
                If @p.@cursor.count = 0 Or @p.ndx >= @p.@cursor.count Then
                    @p.way = %SsSsTriUp 'can't go down or right - go up
                ElseIf @p.@cursor.@arr[@p.ndx] = 0 Then
                    @p.way = %SsSsTriRt 'no child - can't go down - go right
                Else 'go down
                    @p.cursor = @p.@cursor.@arr[@p.ndx]
                    @p.ndx = 0
                    @p.way = %SsSsTriDn
                    If @p.@cursor.value Then
                        Function = 1 : Exit Function
                    End If
                End If
            Case %SsSsTriRt
                Incr @p.ndx 'go right
                If @p.ndx >= @p.@cursor.count Then
                    @p.way = %SsSsTriUp 'can't go right or down
                Else
                    @p.way = %SsSsTriDn
                End If
            Case %SsSsTriUp
                If @p.@cursor.prnt = @p.prefix Then Exit Function
                If @p.@cursor.prnt Then @p.ndx = @p.@cursor.char - @p.@cursor.@prnt.low
                @p.cursor = @p.@cursor.prnt 'will exit loop if cursor on root node
                @p.way = %SsSsTriRt
            End Select
        Wend
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Cursor - Get/Set procedures
    '----------------------------------------------------------------------------------------
'--

Function SsSsTriGetKey(ByVal p As SsSsTri Ptr) As String
    'get Key at current cursor position
    Local s As String
    Local n As SsSsTriNode Ptr
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    ExitF(@p.cursor=0 Or @p.@cursor.value=0, LibErrB)
    n = @p.cursor
    While n And @n.char
        s = Chr$(@n.char) + s
        n = @n.prnt
    Wend
    Function = s
End Function

Function SsSsTriGetVal(ByVal p As SsSsTri Ptr) As String
    'get Value at current cursor position
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    ExitF(@p.cursor=0 Or @p.@cursor.value=0, LibErrB)
    Function = SsGet(@p.@cursor.value)
End Function

Sub SsSsTriSetVal(ByVal p As SsSsTri Ptr, ByRef value As String)
    'set Value at current cursor position
    ExitS(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    ExitS(@p.cursor=0 Or @p.@cursor.value=0, LibErrB)
    SsSet @p.@cursor.value, value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SsSsTriClone(ByVal p As SsSsTri Ptr) As Long
    'create duplicate container - return handle
    Local h, ok As Long
    Err = 0
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    h = SsSsTriNew() : If Err Then Exit Function
    ok = SsSsTriFirstKey(p)
    While ok
        SsSsTriSet h, SsSsTriGetKey(p), SsSsTriGetVal(p)
        ok = SsSsTriNextKey(p)
    Wend
    Function = h
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SsSsTriStore(ByVal p As SsSsTri Ptr) As String
    'store container to string
    Local ok, keys, vals, stor As Long
    Err = 0
    ExitF(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    keys = SsLstNew() : If Err Then Exit Function
    vals = SsLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @p.count Then
        ok = SsSsTriFirstKey(p)
        While ok
            SsLstAdd keys, SsSsTriGetKey(p)
            SsLstAdd vals, SsSsTriGetVal(p)
            ok = SsSsTriNextKey(p)
        Wend
        SsLstQuePush stor, SsLstStore(keys)
        SsLstQuePush stor, SsLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = SsLstFinal(keys)
    vals = SsLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub SsSsTriRestore(ByVal p As SsSsTri Ptr, ByVal s As String)
    'restore container from string
    Local ok, keys, vals, stor As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    keys = SsLstNew() : If Err Then Exit Sub
    vals = SsLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    SsSsTriClear p
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        SsLstRestore keys, SsLstQuePop(stor)
        SsLstRestore vals, SsLstQuePop(stor)
        ExitS(SsLstCount(keys)<>SsLstCount(vals), LibErrU)
        While SsLstCount(keys)
            SsSsTriSet p, SsLstPopFirst(keys), SsLstPopFirst(vals)
        Wend
    End If
    keys = SsLstFinal(keys)
    vals = SsLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SsSsTriFileStore(ByVal p As SsSsTri Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    s = SsSsTriStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsSsTriFileRestore(ByVal p As SsSsTri Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>SsSsTriTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SsSsTriRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function SsSsTriNodeAlloc(ByVal prnt As SsSsTriNode Ptr, ByVal c As Byte) Private As Long
    Local n As SsSsTriNode Ptr
    n = MemAlloc(SizeOf(SsSsTriNode))
    ExitF(n=0, LibErrM)
    @n.prnt = prnt
    @n.char = c
    Function = n
End Function

Function SsSsTriNodeFree(ByVal n As SsSsTriNode Ptr) Private As Long
    Local i As Long
    If n Then
        For i = 0 To @n.count - 1
            If @n.@arr[i] Then @n.@arr[i] = SsSsTriNodeFree(@n.@arr[i])
        Next i
        SsSsTriArrClear n
        If @n.value Then @n.value = SsFinal(@n.value)
        MemFree(n)
    End If
End Function

Function SsSsTriNodeAddToRange(ByVal n As SsSsTriNode Ptr, ByVal c As Byte) Private As Long
    Local i, items As Long
    ExitF(n=0, LibErrM)
    If @n.count = 0 Then
        SsSsTriArrAdd n, 0
        @n.low = c
        @n.high = c
        Function = 0
    ElseIf c < @n.low Then
        items = @n.low - c
        For i = 1 To items
            SsSsTriArrIns n, 0, 0
        Next i
        @n.low = c
        Function = 0
    ElseIf c > @n.high Then
        items = c - @n.high
        For i = 1 To items
            SsSsTriArrAdd n, 0
        Next i
        @n.high = c
        Function = @n.count - 1
    Else
        Function = c - @n.low
    End If
End Function

Sub SsSsTriNodeDisconnect(ByVal p As SsSsTri Ptr, ByVal n As SsSsTriNode Ptr) Private
    Local x As Long
    Local prnt As SsSsTriNode Ptr
    ExitS(n=0, LibErrP)
    prnt = @n.prnt
    If prnt Then
        x = @n.char - @prnt.low
        ExitS(x<0 Or x>=@prnt.count, LibErrB)
        @prnt.@arr[x] = 0
        'may need to collapse range
        While @prnt.count And @prnt.@arr[0] = 0
            SsSsTriArrDel prnt, 0
            Incr @prnt.low
        Wend
        While @prnt.count And @prnt.@arr[@prnt.count - 1] = 0
            SsSsTriArrDel prnt, @prnt.count - 1
            Decr @prnt.high
        Wend
    End If
End Sub

Sub SsSsTriArrClear(ByVal n As SsSsTriNode Ptr) Private
    ExitS(n=0, LibErrP)
    @n.arr = MemFree(@n.arr)
    @n.count = 0
End Sub

Sub SsSsTriArrReDim(ByVal n As SsSsTriNode Ptr, ByVal Count As Long) Private
    ExitS(n=0, LibErrP)
    If Count = 0 Then
        SsSsTriArrClear n
    ElseIf Count <> @n.count Then
        @n.count = 0
        @n.arr = MemReAlloc(@n.arr, Count * 4)
        ExitS(@n.arr=0, LibErrM)
        @n.count = Count
    End If
End Sub

Sub SsSsTriArrAdd(ByVal n As SsSsTriNode Ptr, ByVal value As Long) Private
    Err = 0
    ExitS(n=0, LibErrP)
    SsSsTriArrReDim n, @n.count + 1 : If Err Then Exit Sub
    @n.@arr[@n.count - 1] = value
End Sub

Sub SsSsTriArrIns(ByVal n As SsSsTriNode Ptr, ByVal index As Byte, ByVal value As Long) Private
    Err = 0
    ExitS(n=0, LibErrP)
    ExitS(index>=@n.count, LibErrB)
    SsSsTriArrReDim n, @n.count + 1 : If Err Then Exit Sub
    SsSsTriArrMove n, index, index + 1, @n.count - index - 1
    @n.@arr[index] = value
End Sub

Sub SsSsTriArrDel(ByVal n As SsSsTriNode Ptr, ByVal index As Byte) Private
    ExitS(n=0, LibErrP)
    ExitS(index>=@n.count, LibErrB)
    If index < @n.count - 1 Then SsSsTriArrMove n, index + 1, index , @n.count - index - 1
    SsSsTriArrReDim n, @n.count - 1
End Sub

Sub SsSsTriArrMove(ByVal n As SsSsTriNode Ptr, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy @n.arr + (fromIndex * 4), @n.arr + (toIndex * 4), Count * 4
End Sub
