#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro LnArrTag = 1320870834
Type LnArr
    tag As Long
    count As Long
    arr As Long Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Long Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = LnArrNew() 'get handle for new container
    '       h = LnArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function LnArrNew() As Long
    'allocate new container - return handle
    Local p As LnArr Ptr
    p = MemAlloc(SizeOf(LnArr))
    ExitF(p=0, LibErrM)
    @p.tag = LnArrTag
    Function = p
End Function

Function LnArrFinal(ByVal p As LnArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>LnArrTag, LibErrH)
        LnArrClear p
        MemFree(p)
    End If
End Function

Function LnArrValidate(ByVal p As LnArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = LnArrTag Then Function = @p.tag
End Function

Sub LnArrClear(ByVal p As LnArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeLn
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function LnArrCount(ByVal p As LnArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    Function = @p.count
End Function

Sub LnArrReDim(ByVal p As LnArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        LnArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeLn
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeLn)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeLn
    End If
End Sub

Function LnArrGet(ByVal p As LnArr Ptr, ByVal index As Long) As Long
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub LnArrSet(ByVal p As LnArr Ptr, ByVal index As Long, ByVal value As Long)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub LnArrAdd(ByVal p As LnArr Ptr, ByVal value As Long)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    LnArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub LnArrIns(ByVal p As LnArr Ptr, ByVal index As Long, ByVal value As Long)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    LnArrReDim p, @p.count + 1 : If Err Then Exit Sub
    LnArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub LnArrDel(ByVal p As LnArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then LnArrMove @p.arr, i + 1, i, @p.count - i
    LnArrReDim p, @p.count - 1
End Sub

Sub LnArrSort(ByVal p As LnArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As Long
    Local arr As Long Ptr
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub


Sub LnArrSortUsing(ByVal p As LnArr Ptr, ByVal callbackFunc As Long, Opt ByVal hCollation As Long)
    'sort array using callback function to handle comparisons
    'Function LnCompare(ByVal a As Long, ByVal b As Long, ByVal collation As Long) As Long
    '   a < b : return < 0
    '   a = b : return = 0
    '   a > b : return > 0
    '   hCollation may be used to pass collation sequence string handle
    Local i, j, k, leftIndex, rightIndex, counter, compare As Long
    Local arr As Long Ptr
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    ExitS(callbackFunc=0, LibErrC)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            While i <= j
                Call Dword callbackFunc Using LnCompare(@arr[i], @arr[k], hCollation) To compare
                While compare < 0
                    Incr i
                    Call Dword callbackFunc Using LnCompare(@arr[i], @arr[k], hCollation) To compare
                Wend
                Call Dword callbackFunc Using LnCompare(@arr[j], @arr[k], hCollation) To compare
                While compare > 0
                    Decr j
                    Call Dword callbackFunc Using LnCompare(@arr[j], @arr[k], hCollation) To compare
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub LnArrUniqueSort(ByVal p As LnArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As Long Ptr
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    LnArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then LnArrDel p, i + 1
    Next i
End Sub

Sub LnArrUniqueSortUsing(ByVal p As LnArr Ptr, ByVal callbackFunc As Long, Opt ByVal hCollation As Long)
    'sort array and delete all duplicates using callback function to handle comparisons
    'Function LnCompare(ByVal a As Long, ByVal b As Long, ByVal collation As Long) As Long
    '   a < b : return < 0
    '   a = b : return = 0
    '   a > b : return > 0
    '   hCollation may be used to pass collation sequence string handle
    Local i, compare As Long
    Local arr As Long Ptr
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    ExitS(callbackFunc=0, LibErrC)
    LnArrSortUsing p, callbackFunc, hCollation
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        Call Dword callbackFunc Using LnCompare(@arr[i + 1], @arr[i], hCollation) To compare
        If compare = 0 Then LnArrDel p, i + 1
    Next i
End Sub

Sub LnArrReverse(ByVal p As LnArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As Long Ptr
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function LnArrBinSearch(ByVal p As LnArr Ptr, ByVal value As Long) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As Long Ptr
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function


Function LnArrBinSearchUsing(ByVal p As LnArr Ptr, ByVal value As Long, ByVal callbackFunc As Long, Opt ByVal hCollation As Long) As Long
    'binary search for item using callback function to handle comparisons
    'return index - zero if not found
    'array must be sorted for valid results
    'Function LnCompare(ByVal a As Long, ByVal b As Long, ByVal collation As Long) As Long
    '   a < b : return < 0
    '   a = b : return = 0
    '   a > b : return > 0
    '   hCollation may be used to pass collation sequence string handle
    Local i, top, bot, compare As Long
    Local arr As Long Ptr
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    ExitF(callbackFunc=0, LibErrC)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        Call Dword callbackFunc Using LnCompare(value, @arr[i], hCollation) To compare
        If compare > 0 Then
            bot = i + 1
        ElseIf compare < 0 Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub LnArrBinInsert(ByVal p As LnArr Ptr, ByVal value As Long)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    i = LnArrBinPos(p, value, compare)
    If i = 0 Then
        LnArrAdd p, value
    ElseIf compare <= 0 Then
        LnArrIns p, i, value
    ElseIf i < @p.count Then
        LnArrIns p, i + 1, value
    Else
        LnArrAdd p, value
    End If
End Sub

Sub LnArrBinInsertUsing(ByVal p As LnArr Ptr, ByVal value As Long, ByVal callbackFunc As Long, Opt ByVal hCollation As Long)
    'binary insert value into array using callback function to handle comparisons
    'ok if array empty
    'array must be sorted for valid results
    'Function LnCompare(ByVal a As Long, ByVal b As Long, ByVal collation As Long) As Long
    '   a < b : return < 0
    '   a = b : return = 0
    '   a > b : return > 0
    '   hCollation may be used to pass collation sequence string handle
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    ExitS(callbackFunc=0, LibErrC)
    i = LnArrBinPosUsing(p, value, compare, callbackFunc, hCollation)
    If i = 0 Then
        LnArrAdd p, value
    ElseIf compare <= 0 Then
        LnArrIns p, i, value
    ElseIf i < @p.count Then
        LnArrIns p, i + 1, value
    Else
        LnArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Long Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub LnArrStkPush(ByVal p As LnArr Ptr, ByVal value As Long)
    'push value on top of Stack
    LnArrAdd p, value
End Sub

Function LnArrStkPeek(ByVal p As LnArr Ptr) As Long
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function LnArrStkPop(ByVal p As LnArr Ptr) As Long
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        LnArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Long Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub LnArrQuePush(ByVal p As LnArr Ptr, ByVal value As Long)
    'add value to end of Queue
    LnArrAdd p, value
End Sub

Function LnArrQuePeek(ByVal p As LnArr Ptr) As Long
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function LnArrQuePop(ByVal p As LnArr Ptr) As Long
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        LnArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Long Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub LnArrPushFirst(ByVal p As LnArr Ptr, ByVal value As Long)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then LnArrIns p, 1, value Else LnArrAdd p, value
End Sub

Sub LnArrPushLast(ByVal p As LnArr Ptr, ByVal value As Long)
    'add value to and of Deque (double-ended Queue)
    LnArrAdd p, value
End Sub

Function LnArrPeekFirst(ByVal p As LnArr Ptr) As Long
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function LnArrPeekLast(ByVal p As LnArr Ptr) As Long
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function LnArrPopFirst(ByVal p As LnArr Ptr) As Long
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        LnArrDel p, 1
    End If
End Function

Function LnArrPopLast(ByVal p As LnArr Ptr) As Long
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        LnArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnArrClone(ByVal p As LnArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As LnArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    clone = LnArrNew() : If Err Then Exit Function
    If @p.count Then
        LnArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeLn, @clone.arr + %SizeLn, @p.count * %SizeLn
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnArrStore(ByVal p As LnArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>LnArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeLn, @p.count * %SizeLn)
End Function

Sub LnArrRestore(ByVal p As LnArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    LnArrClear p
    If strLen Then
        LnArrReDim p, strLen / %SizeLn : If Err Then Exit Sub
        Poke$ @p.arr + %SizeLn, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnArrFileStore(ByVal p As LnArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    s = LnArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnArrFileRestore(ByVal p As LnArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>LnArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub LnArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeLn), arr + (toIndex * %SizeLn),  Count * %SizeLn
End Sub

Function LnArrBinPos(ByVal p As LnArr Ptr, ByVal value As Long, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Long Ptr
    compare = 0
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function


Function LnArrBinPosUsing(ByVal p As LnArr Ptr, ByVal value As Long, ByRef compare As Long, ByVal callbackFunc As Long, Opt ByVal hCollation As Long) Private As Long
    Local i, top, bot As Long
    Local arr As Long Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        Call Dword callbackFunc Using LnCompare(value, @arr[i], hCollation) To compare
        If compare > 0 Then
            bot = i + 1
        ElseIf compare < 0 Then
            top = i - 1
        Else
            Exit Loop
        End If
    Wend
    Function = i
End Function
