#Include This Once
#Include Once "C:\HLib3\String\WsStr.inc"
#Include Once "C:\HLib3\Array\LnArr.inc"

Macro WsArrTag = 359211045
Type WsArr
    tag As Long
    arr As LnArr Ptr
    compareCB As Long
    collation As WsStr Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   WString Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = WsArrNew() 'get handle for new container
    '       h = WsArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function WsArrNew() As Long
    'allocate new container - return handle
    Local p As WsArr Ptr
    Err = 0
    p = MemAlloc(SizeOf(WsArr))
    ExitF(p=0, LibErrM)
    @p.tag = WsArrTag
    @p.compareCB = CodePtr(WsCompare)
    @p.collation = WsNew() : If Err Then Exit Function
    @p.arr = LnArrNew() : If Err Then Exit Function
    Function = p
End Function

Function WsArrFinal(ByVal p As WsArr Ptr) As Long
    'free allocated container - return null
    Local i As Long
    If p Then
        ExitF(@p.tag<>WsArrTag, LibErrH)
        WsArrClear p
        @p.arr = LnArrFinal(@p.arr)
        @p.collation = WsFinal(@p.collation)
        MemFree(p)
    End If
End Function

Function WsArrValidate(ByVal p As WsArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = WsArrTag Then Function = @p.tag
End Function

Sub WsArrComparison(ByVal p As WsArr Ptr, ByVal compareUCase As Long, ByVal collationSequence As WString)
    'set how WStrings compared
    'default = case ignored
    'if collationSequence WString provided then
    '   WStrings are compared using the order of the collation sequence WString
    '   collation WString must be 65536 characters
    'else if compareUCase = True then
    '   WStrings compared UCase
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    @p.compareCB = CodePtr(WsCompare)
    WsClear @p.collation
    If Len(collationSequence) Then
        ExitS(Len(collationSequence)<>65536, LibErrS)
        WsSet @p.collation, collationSequence : If Err Then Exit Sub
        @p.compareCB = CodePtr(WsCompareCollate)
    ElseIf compareUCase Then
        @p.compareCB = CodePtr(WsCompareUCase)
    End If
End Sub

Sub WsArrClear(ByVal p As WsArr Ptr)
    'delete all data
    Local i As Long
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    For i = 1 To @p.@arr.count
        @p.@arr.@arr[i] = WsFinal(@p.@arr.@arr[i])
    Next i
    LnArrClear @p.arr
End Sub

Function WsArrCount(ByVal p As WsArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    Function = @p.@arr.count
End Function

Sub WsArrReDim(ByVal p As WsArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local i, items As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        WsArrClear p
    ElseIf Count <> @p.@arr.count Then
        items = @p.@arr.count
        For i = Count + 1 To items
            @p.@arr.@arr[i] = WsFinal(@p.@arr.@arr[i])
        Next i
        LnArrReDim @p.arr, Count : If Err Then Exit Sub
        For i = items + 1 To Count
            @p.@arr.@arr[i] = WsNew() : If Err Then Exit Sub
        Next i
    End If
End Sub

Function WsArrGet(ByVal p As WsArr Ptr, ByVal index As Long) As WString
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    ExitF(i<1 Or i>@p.@arr.count, LibErrB)
    Function = WsGet(@p.@arr.@arr[i])
End Function

Sub WsArrSet(ByVal p As WsArr Ptr, ByVal index As Long, ByRef value As WString)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    WsSet @p.@arr.@arr[i], value
End Sub

Sub WsArrAdd(ByVal p As WsArr Ptr, ByRef value As WString)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    LnArrAdd @p.arr, WsSetNew(value)
End Sub

Sub WsArrIns(ByVal p As WsArr Ptr, ByVal index As Long, ByRef value As WString)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    LnArrIns @p.arr, i, WsSetNew(value)
End Sub

Sub WsArrDel(ByVal p As WsArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    @p.@arr.@arr[i] = WsFinal(@p.@arr.@arr[i])
    LnArrDel @p.arr, i
End Sub

Sub WsArrSort(ByVal p As WsArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter, compare As Long
    Local value As Long
    Local arr As Long Ptr
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count > 1 Then
        arr = @p.@arr.arr
        leftIndex = 1
        rightIndex = @p.@arr.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                Call Dword @p.compareCB Using WsCompareCB(@arr[i], value, @p.collation) To compare
                While compare < 0
                    Incr i
                    Call Dword @p.compareCB Using WsCompareCB(@arr[i], value, @p.collation) To compare
                Wend
                Call Dword @p.compareCB Using WsCompareCB(@arr[j], value, @p.collation) To compare
                While compare > 0
                    Decr j
                    Call Dword @p.compareCB Using WsCompareCB(@arr[j], value, @p.collation) To compare
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub WsArrUniqueSort(ByVal p As WsArr Ptr)
    'sort array and delete all duplicates
    Local i, compare As Long
    Local arr As Long Ptr
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    WsArrSort p
    arr = @p.@arr.arr
    For i = @p.@arr.count - 1 To 1 Step -1
        Call Dword @p.compareCB Using WsCompareCB(@arr[i + 1], @arr[i], @p.collation) To compare
        If compare = 0 Then WsArrDel p, i + 1
    Next i
End Sub

Sub WsArrReverse(ByVal p As WsArr Ptr)
    'reverse array
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    LnArrReverse @p.arr
End Sub

Function WsArrBinSearch(ByVal p As WsArr Ptr, ByRef value As WString) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot, compare, temp As Long
    Local arr As Long Ptr
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    bot = 1
    top = @p.@arr.count
    arr = @p.@arr.arr
    temp = WsSetNew(value)
    While top >= bot
        i = bot + top
        Shift Right i, 1
        Call Dword @p.compareCB Using WsCompareCB(temp, @arr[i], @p.collation) To compare
        If compare > 0 Then
            bot = i + 1
        ElseIf compare < 0 Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
    WsFinal temp
End Function

Sub WsArrBinInsert(ByVal p As WsArr Ptr, ByRef value As WString)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare, temp As Long
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    temp = WsSetNew(value)
    i = WsArrBinPos(p, temp, compare)
    If i = 0 Then
        WsArrAdd p, value
    ElseIf compare <= 0 Then
        WsArrIns p, i, value
    ElseIf i < @p.@arr.count Then
        WsArrIns p, i + 1, value
    Else
        WsArrAdd p, value
    End If
    temp = WsFinal(temp)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   WString Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub WsArrStkPush(ByVal p As WsArr Ptr, ByRef value As WString)
    'push value on top of Stack
    WsArrAdd p, value
End Sub

Function WsArrStkPeek(ByVal p As WsArr Ptr) As WString
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count Then Function = WsGet(@p.@arr.@arr[@p.@arr.count])
End Function

Function WsArrStkPop(ByVal p As WsArr Ptr) As WString
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count Then
        Function = WsGet(@p.@arr.@arr[@p.@arr.count])
        WsArrDel p, @p.@arr.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   WString Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub WsArrQuePush(ByVal p As WsArr Ptr, ByRef value As WString)
    'add value to end of Queue
    WsArrAdd p, value
End Sub

Function WsArrQuePeek(ByVal p As WsArr Ptr) As WString
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count Then Function = WsGet(@p.@arr.@arr[1])
End Function

Function WsArrQuePop(ByVal p As WsArr Ptr) As WString
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count Then
        Function = WsGet(@p.@arr.@arr[1])
        WsArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   WString Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub WsArrPushFirst(ByVal p As WsArr Ptr, ByRef value As WString)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count Then WsArrIns p, 1, value Else WsArrAdd p, value
End Sub

Sub WsArrPushLast(ByVal p As WsArr Ptr, ByRef value As WString)
    'add value to and of Deque (double-ended Queue)
    WsArrAdd p, value
End Sub

Function WsArrPeekFirst(ByVal p As WsArr Ptr) As WString
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count Then Function = WsGet(@p.@arr.@arr[1])
End Function

Function WsArrPeekLast(ByVal p As WsArr Ptr) As WString
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count Then Function = WsGet(@p.@arr.@arr[@p.@arr.count])
End Function

Function WsArrPopFirst(ByVal p As WsArr Ptr) As WString
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count Then
        Function = WsGet(@p.@arr.@arr[1])
        WsArrDel p, 1
    End If
End Function

Function WsArrPopLast(ByVal p As WsArr Ptr) As WString
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count Then
        Function = WsGet(@p.@arr.@arr[@p.@arr.count])
        WsArrDel p, @p.@arr.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function WsArrClone(ByVal p As WsArr Ptr) As Long
    'duplicate container - return clone's handle
    Local i As Long
    Local clone As WsArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    clone = WsArrNew() : If Err Then Exit Function
    If @p.@arr.count Then
        WsArrReDim clone, @p.@arr.count : If Err Then Exit Function
        For i = 1 To @p.@arr.count
            WsArrSet clone, i, WsArrGet(p, i)
        Next i
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function WsArrStore(ByVal p As WsArr Ptr) As String
    'store container to string
    Local i, bytes, tot As Long
    Local s As String
    Local ps As WsStr Ptr
    Local pl As Long Ptr
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    If @p.@arr.count Then
        tot = 4
        For i = 1 To @p.@arr.count
            ps = @p.@arr.@arr[i]
            ExitF(ps=0, LibErrU)
            tot += 4 + (@ps.count * 2)
        Next i
        s = Nul$(tot)
        pl = StrPtr(s)
        @pl = @p.@arr.count : Incr pl
        For i = 1 To @p.@arr.count
            ps = @p.@arr.@arr[i]
            bytes = @ps.count * 2
            @pl = bytes : Incr pl
            If bytes Then
                Memory Copy @ps.mem, pl, bytes : pl += bytes
            End If
        Next i
    End If
    Function = s
End Function

Sub WsArrRestore(ByVal p As WsArr Ptr, ByVal s As String)
    'restore container from string
    Local i, items, bytes As Long
    Local pl As Long Ptr
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    WsArrClear p
    If strLen Then
        pl = StrPtr(s)
        items = @pl : Incr pl
        WsArrReDim p, items : If Err Then Exit Sub
        For i = 1 To items
            bytes = @pl : Incr pl
            If bytes Then
                WsArrSet p, i, Peek$$(pl, bytes / 2) : pl += bytes
            End If
        Next i
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub WsArrFileStore(ByVal p As WsArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    s = WsArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub WsArrFileRestore(ByVal p As WsArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>WsArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        WsArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   WString Builder Procedures
    '       use container's other procedures to Add/Insert/Modify/Remove string segments
    '----------------------------------------------------------------------------------------
'--

Function WsArrBuildWStr(ByVal p As WsArr Ptr) As WString
    'get complete string containing all string segments in container
    Local i, tot, h As Long
    Local s As WString
    Local pt As WsStr Ptr
    ExitF(p=0 Or @p.tag<>WsArrTag, LibErrH)
    For i = 1 To @p.@arr.count
        pt = @p.@arr.@arr[i]
        tot += @pt.count * 2
    Next i
    If tot Then
        s = Nul$(tot / 2)
        h = StrPtr(s)
        For i = 1 To @p.@arr.count
            pt = @p.@arr.@arr[i]
            If @pt.count Then
                Memory Copy @pt.mem, h, @pt.count * 2 : h += (@pt.count * 2)
            End If
        Next i
    End If
    Function = s
End Function

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function WsArrBinPos(ByVal p As WsArr Ptr, ByVal hValue As Long, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Long Ptr
    compare = -1
    bot = 1
    top = @p.@arr.count
    arr = @p.@arr.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        Call Dword @p.compareCB Using WsCompareCB(hValue, @arr[i], @p.collation) To compare
        If compare > 0 Then
            bot = i + 1
        ElseIf compare < 0 Then
            top = i - 1
        Else
            Exit Loop
        End If
    Wend
    Function = i
End Function
