#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\Hash.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\List\ExLst.inc"

'++
    '----------------------------------------------------------------------------------------
    '   Long/Extended ~ Hash Table Container
    '       http://en.wikipedia.org/wiki/Hash_table
    '       Key must be unique Long
    '       Value stored/retrieved/removed using unique lookup Key
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = LnExHshNew() 'get handle for new container
    '       h = LnExHshFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Macro LnExHshTag = -1900523422
Type LnExHshNode
    next As LnExHshNode Ptr
    K As Long
    V As Extended
End Type
Type LnExHsh
    tag As Long
    count As Long
    cap As Long
    arr As Long Ptr
End Type

Function LnExHshNew(ByVal capacity As Long) As Long
    'allocate new container - return handle
    'capacity = number of expected Keys (minium 100)
    Local i As Long
    Local p As LnExHsh Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = LnExHshTag
    @p.cap = Max&(capacity, 10)
    @p.arr = MemAlloc(@p.cap * 4)
    ExitF(@p.arr=0, LibErrM)
    Function = p
End Function

Function LnExHshFinal(ByVal p As LnExHsh Ptr) As Long
    'free allocated container - return null
    Local i As Long
    Local n, x As LnExHshNode Ptr
    If p Then
        ExitF(@p.tag<>LnExHshTag, LibErrH)
        For i = 0 To @p.cap - 1
            n = @p.@arr[i] : @p.@arr[i] = 0
            While n
                x = n : n = @n.next
                MemFree(x)
            Wend
        Next i
        @p.arr = MemFree(@p.arr)
    End If
End Function

Function LnExHshValidate(ByVal p As LnExHsh Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = LnExHshTag Then Function = @p.tag
End Function

Sub LnExHshClear(ByVal p As LnExHsh Ptr)
    'delete all data
    Local i As Long
    Local n, x As LnExHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    For i = 0 To  @p.cap - 1
        n = @p.@arr[i] : @p.@arr[i] = 0
        While n
            x = n : n = @n.next
            MemFree(x)
        Wend
    Next i
    @p.count = 0
End Sub

Function LnExHshCount(ByVal p As LnExHsh Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    Function = @p.count
End Function

Function LnExHshCapGet(ByVal p As LnExHsh Ptr) As Long
    'get Hash Table capacity
    ExitF(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    Function = @p.cap
End Function

Sub LnExHshCapSet(ByVal p As LnExHsh Ptr, ByVal capacity As Long)
    'set Hash Table capacity
    'rebuild Hash Table with new capacity - data preserved
    'capacity should be about the same as number of stored Keys
    Local i, oldCap, newCap As Long
    Local arr As Long Ptr
    Local n, x As LnExHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    oldCap = @p.cap
    newCap = Max&(capacity, 10)
    arr = MemAlloc(newCap * 4)
    ExitS(arr=0, LibErrM)
    Swap @p.arr, arr
    @p.cap = newCap
    @p.count = 0
    For i = 0 To oldCap - 1
        n = @arr[i] : @arr[i] = 0
        While n
            x = n : n = @n.next
            LnExHshSet p, @x.K, @x.V
            MemFree(x)
        Wend
    Next i
    MemFree(arr)
End Sub

Sub LnExHshSet(ByVal p As LnExHsh Ptr, ByVal key As Long, ByVal value As Extended, Opt ByVal DontReplace As Byte)
    'add Key/Value to Tash Table - Value replaced if Key exist unless DontReplace = True
    Local i As Long
    Local n, nn As LnExHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    If n Then
        Do
            If @n.K = key Then
                If IsFalse DontReplace Then @n.V = value
                Exit Sub
            ElseIf @n.next Then
                n = @n.next
            Else
                Exit Loop
            End If
        Loop
        nn = MemAlloc(SizeOf(LnExHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = key
        @nn.V = value
        @n.next = nn
        Incr @p.count
    Else
        nn = MemAlloc(SizeOf(LnExHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = key
        @nn.V = value
        @p.@arr[i] = nn
        Incr @p.count
    End If
End Sub

Function LnExHshGet(ByVal p As LnExHsh Ptr, ByVal key As Long) As Extended
    'get Key's associated Value
    Local i As Long
    Local n As LnExHshNode Ptr
    ExitF(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        If @n.K = key Then
            Function = @n.V
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Function LnExHshGot(ByVal p As LnExHsh Ptr, ByVal key As Long) As Byte
    'True/False if Key in Hash Table
    Local i As Long
    Local n As LnExHshNode Ptr
    ExitF(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        If @n.K = key Then
            Function = 1
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Sub LnExHshDel(ByVal p As LnExHsh Ptr, ByVal key As Long)
    'remove Key and associated Value from Hash Table
    Local i As Long
    Local prev, n As LnExHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    prev = 0
    n = @p.@arr[i]
    While n
        If @n.K = key Then
            If prev Then
                @prev.next = @n.next
            Else
                @p.@arr[i] = @n.next
            End If
            MemFree(n)
            ExitS(@p.count=0, LibErrU)
            Decr @p.count
            Exit Loop
        End If
        prev = n
        n = @n.next
    Wend
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnExHshClone(ByVal p As LnExHsh Ptr) As Long
    'create duplicate container - return handle to cloned container
    Local i As Long
    Local n As LnExHshNode Ptr
    Local clone As Long
    Err = 0
    ExitF(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    clone = LnExHshNew(@p.cap) : If Err Then Exit Function
    For i = 0 To @p.cap - 1
        n = @p.@arr[i]
        While n
            LnExHshSet clone, @n.K, @n.V
            n = @n.next
        Wend
    Next i
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnExHshStore(ByVal p As LnExHsh Ptr) As String
    'store container to string
    Local i As Long
    Local keys, vals, store As Long
    Local n As LnExHshNode Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    keys = LnLstNew() : If Err Then Exit Function
    vals = ExLstNew() : If Err Then Exit Function
    store = SsLstNew() : If Err Then Exit Function
    If @p.count Then
        For i = 0 To @p.cap - 1
            n = @p.@arr[i]
            While n
                LnLstAdd keys, @n.K
                ExLstAdd vals, @n.V
                n = @n.next
            Wend
        Next i
        SsLstAdd store, Mkl$(@p.cap)
        SsLstAdd store, LnLstStore(keys)
        SsLstAdd store, ExLstStore(vals)
        Function = SsLstStore(store)
    End If
    keys = LnLstFinal(keys)
    vals = ExLstFinal(vals)
    store = SsLstFinal(store)
End Function

Sub LnExHshRestore(ByVal p As LnExHsh Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, store As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    LnExHshClear p
    keys = LnLstNew() : If Err Then Exit Sub
    vals = ExLstNew() : If Err Then Exit Sub
    store = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore store, s : If Err Then Exit Sub
        ExitS(SsLstCount(store)<>3, LibErrU)
        LnExHshCapSet p, Cvl(SsLstPopFirst(store))
        LnLstRestore keys, SsLstPopFirst(store)
        ExLstRestore vals, SsLstPopFirst(store)
        ExitS(LnLstCount(keys)<>ExLstCount(vals), LibErrU)
        While LnLstCount(keys)
            LnExHshSet p, LnLstPopFirst(keys), ExLstPopFirst(vals)
        Wend
    End If
    keys = LnLstFinal(keys)
    vals = ExLstFinal(vals)
    store = SsLstFinal(store)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnExHshFileStore(ByVal p As LnExHsh Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    s = LnExHshStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnExHshFileRestore(ByVal p As LnExHsh Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>LnExHshTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnExHshRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub
