#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\Hash.inc"
#Include Once "C:\HLib3\String\WsStr.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\WsLst.inc"

'++
    '----------------------------------------------------------------------------------------
    '   WString/WString ~ Hash Table Container
    '       http://en.wikipedia.org/wiki/Hash_table
    '       Key must be unique WString
    '       Key is case sensitive - no nulls
    '       Value stored/retrieved/removed using unique lookup Key
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = WsWsHshNew() 'get handle for new container
    '       h = WsWsHshFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Macro WsWsHshTag = -1900523422
Type WsWsHshNode
    next As WsWsHshNode Ptr
    K As WsStr Ptr
    V As WsStr Ptr
End Type
Type WsWsHsh
    tag As Long
    count As Long
    cap As Long
    arr As Long Ptr
End Type

Function WsWsHshNew(ByVal capacity As Long) As Long
    'allocate new container - return handle
    'capacity = number of expected Keys (minium 100)
    Local i As Long
    Local p As WsWsHsh Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = WsWsHshTag
    @p.cap = Max&(capacity, 10)
    @p.arr = MemAlloc(@p.cap * 4)
    ExitF(@p.arr=0, LibErrM)
    Function = p
End Function

Function WsWsHshFinal(ByVal p As WsWsHsh Ptr) As Long
    'free allocated container - return null
    Local i As Long
    Local n, x As WsWsHshNode Ptr
    If p Then
        ExitF(@p.tag<>WsWsHshTag, LibErrH)
        For i = 0 To @p.cap - 1
            n = @p.@arr[i] : @p.@arr[i] = 0
            While n
                x = n : n = @n.next
                WsFinal(@x.K)
                WsFinal(@x.V)
                MemFree(x)
            Wend
        Next i
        @p.arr = MemFree(@p.arr)
    End If
End Function

Function WsWsHshValidate(ByVal p As WsWsHsh Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = WsWsHshTag Then Function = @p.tag
End Function

Sub WsWsHshClear(ByVal p As WsWsHsh Ptr)
    'delete all data
    Local i As Long
    Local n, x As WsWsHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    For i = 0 To  @p.cap - 1
        n = @p.@arr[i] : @p.@arr[i] = 0
        While n
            x = n : n = @n.next
            WsFinal(@x.K)
            WsFinal(@x.V)
            MemFree(x)
        Wend
    Next i
    @p.count = 0
End Sub

Function WsWsHshCount(ByVal p As WsWsHsh Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    Function = @p.count
End Function

Function WsWsHshCapGet(ByVal p As WsWsHsh Ptr) As Long
    'get Hash Table capacity
    ExitF(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    Function = @p.cap
End Function

Sub WsWsHshCapSet(ByVal p As WsWsHsh Ptr, ByVal capacity As Long)
    'set Hash Table capacity
    'rebuild Hash Table with new capacity - data preserved
    'capacity should be about the same as number of stored Keys
    Local i, oldCap, newCap As Long
    Local arr As Long Ptr
    Local n, x As WsWsHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    oldCap = @p.cap
    newCap = Max&(capacity, 10)
    arr = MemAlloc(newCap * 4)
    ExitS(arr=0, LibErrM)
    Swap @p.arr, arr
    @p.cap = newCap
    @p.count = 0
    For i = 0 To oldCap - 1
        n = @arr[i] : @arr[i] = 0
        While n
            x = n : n = @n.next
            WsWsHshSet p, WsGet(@x.K), WsGet(@x.V)
            WsFinal(@x.K)
            WsFinal(@x.V)
            MemFree(x)
        Wend
    Next i
    MemFree(arr)
End Sub

Sub WsWsHshSet(ByVal p As WsWsHsh Ptr, ByRef key As WString, ByRef value As WString, Opt ByVal DontReplace As Byte)
    'add Key/Value to Tash Table - Value replaced if Key exist unless DontReplace = True
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local n, nn As WsWsHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    If n Then
        Do
            equal = WsEqual(@n.@K.mem, ps)
            If equal Then
                If IsFalse DontReplace Then WsSet @n.V, value
                Exit Sub
            ElseIf @n.next Then
                n = @n.next
            Else
                Exit Loop
            End If
        Loop
        nn = MemAlloc(SizeOf(WsWsHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = WsSetNew(key)
        @nn.V = WsSetNew(value)
        @n.next = nn
        Incr @p.count
    Else
        nn = MemAlloc(SizeOf(WsWsHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = WsSetNew(key)
        @nn.V = WsSetNew(value)
        @p.@arr[i] = nn
        Incr @p.count
    End If
End Sub

Function WsWsHshGet(ByVal p As WsWsHsh Ptr, ByRef key As WString) As WString
    'get Key's associated Value
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local n As WsWsHshNode Ptr
    ExitF(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        equal = WsEqual(@n.@K.mem, ps)
        If equal Then
            Function = WsGet(@n.V)
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Function WsWsHshGot(ByVal p As WsWsHsh Ptr, ByRef key As WString) As Byte
    'True/False if Key in Hash Table
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local n As WsWsHshNode Ptr
    ExitF(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        equal = WsEqual(@n.@K.mem, ps)
        If equal Then
            Function = 1
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Sub WsWsHshDel(ByVal p As WsWsHsh Ptr, ByRef key As WString)
    'remove Key and associated Value from Hash Table
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local prev, n As WsWsHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    prev = 0
    n = @p.@arr[i]
    While n
        equal = WsEqual(@n.@K.mem, ps)
        If equal Then
            If prev Then
                @prev.next = @n.next
            Else
                @p.@arr[i] = @n.next
            End If
            @n.K = WsFinal(@n.K)
            @n.V = WsFinal(@n.V)
            MemFree(n)
            ExitS(@p.count=0, LibErrU)
            Decr @p.count
            Exit Loop
        End If
        prev = n
        n = @n.next
    Wend
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function WsWsHshClone(ByVal p As WsWsHsh Ptr) As Long
    'create duplicate container - return handle to cloned container
    Local i As Long
    Local n As WsWsHshNode Ptr
    Local clone As Long
    Err = 0
    ExitF(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    clone = WsWsHshNew(@p.cap) : If Err Then Exit Function
    For i = 0 To @p.cap - 1
        n = @p.@arr[i]
        While n
            WsWsHshSet clone, WsGet(@n.K), WsGet(@n.V)
            n = @n.next
        Wend
    Next i
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function WsWsHshStore(ByVal p As WsWsHsh Ptr) As String
    'store container to string
    Local i As Long
    Local keys, vals, store As Long
    Local n As WsWsHshNode Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    keys = WsLstNew() : If Err Then Exit Function
    vals = WsLstNew() : If Err Then Exit Function
    store = SsLstNew() : If Err Then Exit Function
    If @p.count Then
        For i = 0 To @p.cap - 1
            n = @p.@arr[i]
            While n
                WsLstAdd keys, WsGet(@n.K)
                WsLstAdd vals, WsGet(@n.V)
                n = @n.next
            Wend
        Next i
        SsLstAdd store, Mkl$(@p.cap)
        SsLstAdd store, WsLstStore(keys)
        SsLstAdd store, WsLstStore(vals)
        Function = SsLstStore(store)
    End If
    keys = WsLstFinal(keys)
    vals = WsLstFinal(vals)
    store = SsLstFinal(store)
End Function

Sub WsWsHshRestore(ByVal p As WsWsHsh Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, store As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    WsWsHshClear p
    keys = WsLstNew() : If Err Then Exit Sub
    vals = WsLstNew() : If Err Then Exit Sub
    store = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore store, s : If Err Then Exit Sub
        ExitS(SsLstCount(store)<>3, LibErrU)
        WsWsHshCapSet p, Cvl(SsLstPopFirst(store))
        WsLstRestore keys, SsLstPopFirst(store)
        WsLstRestore vals, SsLstPopFirst(store)
        ExitS(WsLstCount(keys)<>WsLstCount(vals), LibErrU)
        While WsLstCount(keys)
            WsWsHshSet p, WsLstPopFirst(keys), WsLstPopFirst(vals)
        Wend
    End If
    keys = WsLstFinal(keys)
    vals = WsLstFinal(vals)
    store = SsLstFinal(store)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub WsWsHshFileStore(ByVal p As WsWsHsh Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    s = WsWsHshStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub WsWsHshFileRestore(ByVal p As WsWsHsh Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>WsWsHshTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        WsWsHshRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub
