#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro BtLstNodeTag = -264589208
Macro BtLstTag = -212996673
Type BtLstNode
    tag As Long
    next As BtLstNode Ptr
    prev As BtLstNode Ptr
    value As Byte
End Type
Type BtLst
    tag As Long
    count As Long
    first As BtLstNode Ptr
    last As BtLstNode Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Byte List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = BtLstNew() 'get handle for new container
    '       h = BtLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function BtLstNew() As Long
    'allocate new container - return handle
    Local pLst As BtLst Ptr
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = BtLstTag
    Function = pLst
End Function

Function BtLstFinal(ByVal pLst As BtLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>BtLstTag, LibErrH)
        BtLstClear pLst
        MemFree(pLst)
    End If
End Function

Function BtLstValidate(ByVal pLst As BtLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = BtLstTag Then Function = @pLst.tag
End Function

Sub BtLstClear(ByVal pLst As BtLst Ptr)
    'delete all data
    Local node As BtLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    While @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    Wend
    @pLst.last = 0
    @pLst.count = 0
End Sub

Function BtLstCount(ByVal pLst As BtLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    Function = @pLst.count
End Function

Sub BtLstAdd(ByVal pLst As BtLst Ptr, ByVal value As Byte)
    'append Value to end of List
    Local node As BtLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.count Then
        node = BtLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    Else
        node = BtLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Sub BtLstIns(ByVal pLst As BtLst Ptr, ByVal value As Byte)
    'insert Value at front of List
    Local node As BtLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.count Then
        node = BtLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    Else
        node = BtLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Function BtLstFirst(ByVal pLst As BtLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    Function = @pLst.first
End Function

Function BtLstLast(ByVal pLst As BtLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    Function = @pLst.last
End Function

Function BtLstNext(ByVal pNode As BtLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>BtLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function BtLstPrev(ByVal pNode As BtLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>BtLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function BtLstGet(ByVal pNode As BtLstNode Ptr) As Byte
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>BtLstNodeTag, LibErrH)
    Function = @pNode.value
End Function

Sub BtLstSet(ByVal pNode As BtLstNode Ptr, ByVal value As Byte)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>BtLstNodeTag, LibErrH)
    @pNode.value = value
End Sub

Sub BtLstInsPrev(ByVal pLst As BtLst Ptr, ByVal pNode As BtLstNode Ptr, ByVal value As Byte)
    'insert Value before node
    Local node As BtLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>BtLstTag Or @pNode.tag<>BtLstNodeTag, LibErrH)
    If pNode = @pLst.first Then
        BtLstIns pLst, value
    Else
        ExitS(@pNode.prev=0, LibErrU)
        node = BtLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    End If
End Sub

Sub BtLstInsNext(ByVal pLst As BtLst Ptr, ByVal pNode As BtLstNode Ptr, ByVal value As Byte)
    'insert Value after Cursor
    Local node As BtLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>BtLstTag Or @pNode.tag<>BtLstNodeTag, LibErrH)
    If pNode = @pLst.last Then
        BtLstAdd pLst, value
    Else
        ExitS(@pNode.next=0, LibErrU)
        node = BtLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    End If
End Sub

Sub BtLstDel(ByVal pLst As BtLst Ptr, ByVal pNode As BtLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>BtLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>BtLstNodeTag, LibErrH)
        If @pLst.first = pNode Then @pLst.first = @pNode.next
        If @pLst.last = pNode Then @pLst.last = @pNode.prev
        If @pNode.prev Then @pNode.@prev.next = @pNode.next
        If @pNode.next Then @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        Decr @pLst.count
        MemFree(pNode)
    End If
End Sub

Sub BtLstDelPrev(ByVal pLst As BtLst Ptr, ByVal pNode As BtLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>BtLstTag Or @pNode.tag<>BtLstNodeTag, LibErrH)
    BtLstDel pLst, @pNode.prev
End Sub

Sub BtLstDelNext(ByVal pLst As BtLst Ptr, ByVal pNode As BtLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>BtLstTag Or @pNode.tag<>BtLstNodeTag, LibErrH)
    BtLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub BtLstStkPush(ByVal pLst As BtLst Ptr, ByVal value As Byte)
    'Push Value on Stack
    BtLstAdd pLst, value
End Sub

Function BtLstStkPeek(ByVal pLst As BtLst Ptr) As Byte
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function BtLstStkPop(ByVal pLst As BtLst Ptr) As Byte
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        BtLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub BtLstQuePush(ByVal pLst As BtLst Ptr, ByVal value As Byte)
    'Add Value to end of Queue
    BtLstAdd pLst, value
End Sub

Function BtLstQuePeek(ByVal pLst As BtLst Ptr) As Byte
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function BtLstQuePop(ByVal pLst As BtLst Ptr) As Byte
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        BtLstDel pLst, @pLst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub BtLstPushFirst(ByVal pLst As BtLst Ptr, ByVal value As Byte)
    'Add Value at front of container
    BtLstIns pLst, value
End Sub

Sub BtLstPushLast(ByVal pLst As BtLst Ptr, ByVal value As Byte)
    'Add Value at end of container
    BtLstAdd pLst, value
End Sub

Function BtLstPeekFirst(ByVal pLst As BtLst Ptr) As Byte
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function BtLstPeekLast(ByVal pLst As BtLst Ptr) As Byte
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function BtLstPopFirst(ByVal pLst As BtLst Ptr) As Byte
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        BtLstDel pLst, @pLst.first
    End If
End Function

Function BtLstPopLast(ByVal pLst As BtLst Ptr) As Byte
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        BtLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function BtLstClone(ByVal pLst As BtLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As BtLst Ptr
    Local node As BtLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    Err = 0
    pClone = BtLstNew() : If Err Then Exit Function
    node = @pLst.first
    While node
        BtLstAdd pClone, @node.value
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function BtLstStore(ByVal pLst As BtLst Ptr) As String
    'store container to String
    Local s As String
    Local node As BtLstNode Ptr
    Local p As Byte Ptr
    ExitF(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    If @pLst.count Then
        s = Nul$(@pLst.count * %SizeBt)
        p = StrPtr(s)
        node = @pLst.first
        While node
            @p = @node.value : Incr p
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub BtLstRestore(ByVal pLst As BtLst Ptr, ByRef s As String)
    'restore container from string
    Local items As Long
    Local p As Byte Ptr
    ExitS(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    BtLstClear pLst
    If Len(s) Then
        p = StrPtr(s)
        items = Len(s) / %SizeBt
        While items
            BtLstAdd pLst, @p
            Incr p
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub BtLstFileStore(ByVal pLst As BtLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    s = BtLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub BtLstFileRestore(ByVal pLst As BtLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>BtLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        BtLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function BtLstNodeAlloc(ByVal pLst As BtLst Ptr) Private As Long
    Local node As BtLstNode Ptr
    node = MemAlloc(SizeOf(BtLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = BtLstNodeTag
    Incr @pLst.count
    Function = node
End Function
