#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro CuLstNodeTag = 572035680
Macro CuLstTag = -212996673
Type CuLstNode
    tag As Long
    next As CuLstNode Ptr
    prev As CuLstNode Ptr
    value As Currency
End Type
Type CuLst
    tag As Long
    count As Long
    first As CuLstNode Ptr
    last As CuLstNode Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Currency List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = CuLstNew() 'get handle for new container
    '       h = CuLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function CuLstNew() As Long
    'allocate new container - return handle
    Local pLst As CuLst Ptr
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = CuLstTag
    Function = pLst
End Function

Function CuLstFinal(ByVal pLst As CuLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>CuLstTag, LibErrH)
        CuLstClear pLst
        MemFree(pLst)
    End If
End Function

Function CuLstValidate(ByVal pLst As CuLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = CuLstTag Then Function = @pLst.tag
End Function

Sub CuLstClear(ByVal pLst As CuLst Ptr)
    'delete all data
    Local node As CuLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    While @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    Wend
    @pLst.last = 0
    @pLst.count = 0
End Sub

Function CuLstCount(ByVal pLst As CuLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    Function = @pLst.count
End Function

Sub CuLstAdd(ByVal pLst As CuLst Ptr, ByVal value As Currency)
    'append Value to end of List
    Local node As CuLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.count Then
        node = CuLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    Else
        node = CuLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Sub CuLstIns(ByVal pLst As CuLst Ptr, ByVal value As Currency)
    'insert Value at front of List
    Local node As CuLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.count Then
        node = CuLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    Else
        node = CuLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Function CuLstFirst(ByVal pLst As CuLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    Function = @pLst.first
End Function

Function CuLstLast(ByVal pLst As CuLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    Function = @pLst.last
End Function

Function CuLstNext(ByVal pNode As CuLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>CuLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function CuLstPrev(ByVal pNode As CuLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>CuLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function CuLstGet(ByVal pNode As CuLstNode Ptr) As Currency
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>CuLstNodeTag, LibErrH)
    Function = @pNode.value
End Function

Sub CuLstSet(ByVal pNode As CuLstNode Ptr, ByVal value As Currency)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>CuLstNodeTag, LibErrH)
    @pNode.value = value
End Sub

Sub CuLstInsPrev(ByVal pLst As CuLst Ptr, ByVal pNode As CuLstNode Ptr, ByVal value As Currency)
    'insert Value before node
    Local node As CuLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>CuLstTag Or @pNode.tag<>CuLstNodeTag, LibErrH)
    If pNode = @pLst.first Then
        CuLstIns pLst, value
    Else
        ExitS(@pNode.prev=0, LibErrU)
        node = CuLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    End If
End Sub

Sub CuLstInsNext(ByVal pLst As CuLst Ptr, ByVal pNode As CuLstNode Ptr, ByVal value As Currency)
    'insert Value after Cursor
    Local node As CuLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>CuLstTag Or @pNode.tag<>CuLstNodeTag, LibErrH)
    If pNode = @pLst.last Then
        CuLstAdd pLst, value
    Else
        ExitS(@pNode.next=0, LibErrU)
        node = CuLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    End If
End Sub

Sub CuLstDel(ByVal pLst As CuLst Ptr, ByVal pNode As CuLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>CuLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>CuLstNodeTag, LibErrH)
        If @pLst.first = pNode Then @pLst.first = @pNode.next
        If @pLst.last = pNode Then @pLst.last = @pNode.prev
        If @pNode.prev Then @pNode.@prev.next = @pNode.next
        If @pNode.next Then @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        Decr @pLst.count
        MemFree(pNode)
    End If
End Sub

Sub CuLstDelPrev(ByVal pLst As CuLst Ptr, ByVal pNode As CuLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>CuLstTag Or @pNode.tag<>CuLstNodeTag, LibErrH)
    CuLstDel pLst, @pNode.prev
End Sub

Sub CuLstDelNext(ByVal pLst As CuLst Ptr, ByVal pNode As CuLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>CuLstTag Or @pNode.tag<>CuLstNodeTag, LibErrH)
    CuLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub CuLstStkPush(ByVal pLst As CuLst Ptr, ByVal value As Currency)
    'Push Value on Stack
    CuLstAdd pLst, value
End Sub

Function CuLstStkPeek(ByVal pLst As CuLst Ptr) As Currency
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function CuLstStkPop(ByVal pLst As CuLst Ptr) As Currency
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        CuLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub CuLstQuePush(ByVal pLst As CuLst Ptr, ByVal value As Currency)
    'Add Value to end of Queue
    CuLstAdd pLst, value
End Sub

Function CuLstQuePeek(ByVal pLst As CuLst Ptr) As Currency
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function CuLstQuePop(ByVal pLst As CuLst Ptr) As Currency
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        CuLstDel pLst, @pLst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub CuLstPushFirst(ByVal pLst As CuLst Ptr, ByVal value As Currency)
    'Add Value at front of container
    CuLstIns pLst, value
End Sub

Sub CuLstPushLast(ByVal pLst As CuLst Ptr, ByVal value As Currency)
    'Add Value at end of container
    CuLstAdd pLst, value
End Sub

Function CuLstPeekFirst(ByVal pLst As CuLst Ptr) As Currency
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function CuLstPeekLast(ByVal pLst As CuLst Ptr) As Currency
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function CuLstPopFirst(ByVal pLst As CuLst Ptr) As Currency
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        CuLstDel pLst, @pLst.first
    End If
End Function

Function CuLstPopLast(ByVal pLst As CuLst Ptr) As Currency
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        CuLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function CuLstClone(ByVal pLst As CuLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As CuLst Ptr
    Local node As CuLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    Err = 0
    pClone = CuLstNew() : If Err Then Exit Function
    node = @pLst.first
    While node
        CuLstAdd pClone, @node.value
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function CuLstStore(ByVal pLst As CuLst Ptr) As String
    'store container to String
    Local s As String
    Local node As CuLstNode Ptr
    Local p As Currency Ptr
    ExitF(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    If @pLst.count Then
        s = Nul$(@pLst.count * %SizeCu)
        p = StrPtr(s)
        node = @pLst.first
        While node
            @p = @node.value : Incr p
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub CuLstRestore(ByVal pLst As CuLst Ptr, ByRef s As String)
    'restore container from string
    Local items As Long
    Local p As Currency Ptr
    ExitS(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    CuLstClear pLst
    If Len(s) Then
        p = StrPtr(s)
        items = Len(s) / %SizeCu
        While items
            CuLstAdd pLst, @p
            Incr p
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub CuLstFileStore(ByVal pLst As CuLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    s = CuLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub CuLstFileRestore(ByVal pLst As CuLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>CuLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        CuLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function CuLstNodeAlloc(ByVal pLst As CuLst Ptr) Private As Long
    Local node As CuLstNode Ptr
    node = MemAlloc(SizeOf(CuLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = CuLstNodeTag
    Incr @pLst.count
    Function = node
End Function
