#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro CxLstNodeTag = 1540788735
Macro CxLstTag = 217604277
Type CxLstNode
    tag As Long
    next As CxLstNode Ptr
    prev As CxLstNode Ptr
    value As CurrencyX
End Type
Type CxLst
    tag As Long
    count As Long
    first As CxLstNode Ptr
    last As CxLstNode Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   CurrencyX List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = CxLstNew() 'get handle for new container
    '       h = CxLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function CxLstNew() As Long
    'allocate new container - return handle
    Local pLst As CxLst Ptr
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = CxLstTag
    Function = pLst
End Function

Function CxLstFinal(ByVal pLst As CxLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>CxLstTag, LibErrH)
        CxLstClear pLst
        MemFree(pLst)
    End If
End Function

Function CxLstValidate(ByVal pLst As CxLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = CxLstTag Then Function = @pLst.tag
End Function

Sub CxLstClear(ByVal pLst As CxLst Ptr)
    'delete all data
    Local node As CxLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    While @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    Wend
    @pLst.last = 0
    @pLst.count = 0
End Sub

Function CxLstCount(ByVal pLst As CxLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    Function = @pLst.count
End Function

Sub CxLstAdd(ByVal pLst As CxLst Ptr, ByVal value As CurrencyX)
    'append Value to end of List
    Local node As CxLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.count Then
        node = CxLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    Else
        node = CxLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Sub CxLstIns(ByVal pLst As CxLst Ptr, ByVal value As CurrencyX)
    'insert Value at front of List
    Local node As CxLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.count Then
        node = CxLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    Else
        node = CxLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Function CxLstFirst(ByVal pLst As CxLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    Function = @pLst.first
End Function

Function CxLstLast(ByVal pLst As CxLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    Function = @pLst.last
End Function

Function CxLstNext(ByVal pNode As CxLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>CxLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function CxLstPrev(ByVal pNode As CxLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>CxLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function CxLstGet(ByVal pNode As CxLstNode Ptr) As CurrencyX
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>CxLstNodeTag, LibErrH)
    Function = @pNode.value
End Function

Sub CxLstSet(ByVal pNode As CxLstNode Ptr, ByVal value As CurrencyX)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>CxLstNodeTag, LibErrH)
    @pNode.value = value
End Sub

Sub CxLstInsPrev(ByVal pLst As CxLst Ptr, ByVal pNode As CxLstNode Ptr, ByVal value As CurrencyX)
    'insert Value before node
    Local node As CxLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>CxLstTag Or @pNode.tag<>CxLstNodeTag, LibErrH)
    If pNode = @pLst.first Then
        CxLstIns pLst, value
    Else
        ExitS(@pNode.prev=0, LibErrU)
        node = CxLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    End If
End Sub

Sub CxLstInsNext(ByVal pLst As CxLst Ptr, ByVal pNode As CxLstNode Ptr, ByVal value As CurrencyX)
    'insert Value after Cursor
    Local node As CxLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>CxLstTag Or @pNode.tag<>CxLstNodeTag, LibErrH)
    If pNode = @pLst.last Then
        CxLstAdd pLst, value
    Else
        ExitS(@pNode.next=0, LibErrU)
        node = CxLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    End If
End Sub

Sub CxLstDel(ByVal pLst As CxLst Ptr, ByVal pNode As CxLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>CxLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>CxLstNodeTag, LibErrH)
        If @pLst.first = pNode Then @pLst.first = @pNode.next
        If @pLst.last = pNode Then @pLst.last = @pNode.prev
        If @pNode.prev Then @pNode.@prev.next = @pNode.next
        If @pNode.next Then @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        Decr @pLst.count
        MemFree(pNode)
    End If
End Sub

Sub CxLstDelPrev(ByVal pLst As CxLst Ptr, ByVal pNode As CxLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>CxLstTag Or @pNode.tag<>CxLstNodeTag, LibErrH)
    CxLstDel pLst, @pNode.prev
End Sub

Sub CxLstDelNext(ByVal pLst As CxLst Ptr, ByVal pNode As CxLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>CxLstTag Or @pNode.tag<>CxLstNodeTag, LibErrH)
    CxLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub CxLstStkPush(ByVal pLst As CxLst Ptr, ByVal value As CurrencyX)
    'Push Value on Stack
    CxLstAdd pLst, value
End Sub

Function CxLstStkPeek(ByVal pLst As CxLst Ptr) As CurrencyX
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function CxLstStkPop(ByVal pLst As CxLst Ptr) As CurrencyX
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        CxLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub CxLstQuePush(ByVal pLst As CxLst Ptr, ByVal value As CurrencyX)
    'Add Value to end of Queue
    CxLstAdd pLst, value
End Sub

Function CxLstQuePeek(ByVal pLst As CxLst Ptr) As CurrencyX
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function CxLstQuePop(ByVal pLst As CxLst Ptr) As CurrencyX
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        CxLstDel pLst, @pLst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub CxLstPushFirst(ByVal pLst As CxLst Ptr, ByVal value As CurrencyX)
    'Add Value at front of container
    CxLstIns pLst, value
End Sub

Sub CxLstPushLast(ByVal pLst As CxLst Ptr, ByVal value As CurrencyX)
    'Add Value at end of container
    CxLstAdd pLst, value
End Sub

Function CxLstPeekFirst(ByVal pLst As CxLst Ptr) As CurrencyX
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function CxLstPeekLast(ByVal pLst As CxLst Ptr) As CurrencyX
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function CxLstPopFirst(ByVal pLst As CxLst Ptr) As CurrencyX
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        CxLstDel pLst, @pLst.first
    End If
End Function

Function CxLstPopLast(ByVal pLst As CxLst Ptr) As CurrencyX
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        CxLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function CxLstClone(ByVal pLst As CxLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As CxLst Ptr
    Local node As CxLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    Err = 0
    pClone = CxLstNew() : If Err Then Exit Function
    node = @pLst.first
    While node
        CxLstAdd pClone, @node.value
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function CxLstStore(ByVal pLst As CxLst Ptr) As String
    'store container to String
    Local s As String
    Local node As CxLstNode Ptr
    Local p As CurrencyX Ptr
    ExitF(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    If @pLst.count Then
        s = Nul$(@pLst.count * %SizeCx)
        p = StrPtr(s)
        node = @pLst.first
        While node
            @p = @node.value : Incr p
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub CxLstRestore(ByVal pLst As CxLst Ptr, ByRef s As String)
    'restore container from string
    Local items As Long
    Local p As CurrencyX Ptr
    ExitS(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    CxLstClear pLst
    If Len(s) Then
        p = StrPtr(s)
        items = Len(s) / %SizeCx
        While items
            CxLstAdd pLst, @p
            Incr p
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub CxLstFileStore(ByVal pLst As CxLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    s = CxLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub CxLstFileRestore(ByVal pLst As CxLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>CxLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        CxLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function CxLstNodeAlloc(ByVal pLst As CxLst Ptr) Private As Long
    Local node As CxLstNode Ptr
    node = MemAlloc(SizeOf(CxLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = CxLstNodeTag
    Incr @pLst.count
    Function = node
End Function
