#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro QdLstNodeTag = 1317754472
Macro QdLstTag = -439488143
Type QdLstNode
    tag As Long
    next As QdLstNode Ptr
    prev As QdLstNode Ptr
    value As Quad
End Type
Type QdLst
    tag As Long
    count As Long
    first As QdLstNode Ptr
    last As QdLstNode Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Quad List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = QdLstNew() 'get handle for new container
    '       h = QdLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function QdLstNew() As Long
    'allocate new container - return handle
    Local pLst As QdLst Ptr
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = QdLstTag
    Function = pLst
End Function

Function QdLstFinal(ByVal pLst As QdLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>QdLstTag, LibErrH)
        QdLstClear pLst
        MemFree(pLst)
    End If
End Function

Function QdLstValidate(ByVal pLst As QdLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = QdLstTag Then Function = @pLst.tag
End Function

Sub QdLstClear(ByVal pLst As QdLst Ptr)
    'delete all data
    Local node As QdLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    While @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    Wend
    @pLst.last = 0
    @pLst.count = 0
End Sub

Function QdLstCount(ByVal pLst As QdLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    Function = @pLst.count
End Function

Sub QdLstAdd(ByVal pLst As QdLst Ptr, ByVal value As Quad)
    'append Value to end of List
    Local node As QdLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.count Then
        node = QdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    Else
        node = QdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Sub QdLstIns(ByVal pLst As QdLst Ptr, ByVal value As Quad)
    'insert Value at front of List
    Local node As QdLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.count Then
        node = QdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    Else
        node = QdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Function QdLstFirst(ByVal pLst As QdLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    Function = @pLst.first
End Function

Function QdLstLast(ByVal pLst As QdLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    Function = @pLst.last
End Function

Function QdLstNext(ByVal pNode As QdLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>QdLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function QdLstPrev(ByVal pNode As QdLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>QdLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function QdLstGet(ByVal pNode As QdLstNode Ptr) As Quad
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>QdLstNodeTag, LibErrH)
    Function = @pNode.value
End Function

Sub QdLstSet(ByVal pNode As QdLstNode Ptr, ByVal value As Quad)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>QdLstNodeTag, LibErrH)
    @pNode.value = value
End Sub

Sub QdLstInsPrev(ByVal pLst As QdLst Ptr, ByVal pNode As QdLstNode Ptr, ByVal value As Quad)
    'insert Value before node
    Local node As QdLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>QdLstTag Or @pNode.tag<>QdLstNodeTag, LibErrH)
    If pNode = @pLst.first Then
        QdLstIns pLst, value
    Else
        ExitS(@pNode.prev=0, LibErrU)
        node = QdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    End If
End Sub

Sub QdLstInsNext(ByVal pLst As QdLst Ptr, ByVal pNode As QdLstNode Ptr, ByVal value As Quad)
    'insert Value after Cursor
    Local node As QdLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>QdLstTag Or @pNode.tag<>QdLstNodeTag, LibErrH)
    If pNode = @pLst.last Then
        QdLstAdd pLst, value
    Else
        ExitS(@pNode.next=0, LibErrU)
        node = QdLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    End If
End Sub

Sub QdLstDel(ByVal pLst As QdLst Ptr, ByVal pNode As QdLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>QdLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>QdLstNodeTag, LibErrH)
        If @pLst.first = pNode Then @pLst.first = @pNode.next
        If @pLst.last = pNode Then @pLst.last = @pNode.prev
        If @pNode.prev Then @pNode.@prev.next = @pNode.next
        If @pNode.next Then @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        Decr @pLst.count
        MemFree(pNode)
    End If
End Sub

Sub QdLstDelPrev(ByVal pLst As QdLst Ptr, ByVal pNode As QdLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>QdLstTag Or @pNode.tag<>QdLstNodeTag, LibErrH)
    QdLstDel pLst, @pNode.prev
End Sub

Sub QdLstDelNext(ByVal pLst As QdLst Ptr, ByVal pNode As QdLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>QdLstTag Or @pNode.tag<>QdLstNodeTag, LibErrH)
    QdLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub QdLstStkPush(ByVal pLst As QdLst Ptr, ByVal value As Quad)
    'Push Value on Stack
    QdLstAdd pLst, value
End Sub

Function QdLstStkPeek(ByVal pLst As QdLst Ptr) As Quad
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function QdLstStkPop(ByVal pLst As QdLst Ptr) As Quad
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        QdLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub QdLstQuePush(ByVal pLst As QdLst Ptr, ByVal value As Quad)
    'Add Value to end of Queue
    QdLstAdd pLst, value
End Sub

Function QdLstQuePeek(ByVal pLst As QdLst Ptr) As Quad
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function QdLstQuePop(ByVal pLst As QdLst Ptr) As Quad
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        QdLstDel pLst, @pLst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub QdLstPushFirst(ByVal pLst As QdLst Ptr, ByVal value As Quad)
    'Add Value at front of container
    QdLstIns pLst, value
End Sub

Sub QdLstPushLast(ByVal pLst As QdLst Ptr, ByVal value As Quad)
    'Add Value at end of container
    QdLstAdd pLst, value
End Sub

Function QdLstPeekFirst(ByVal pLst As QdLst Ptr) As Quad
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function QdLstPeekLast(ByVal pLst As QdLst Ptr) As Quad
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function QdLstPopFirst(ByVal pLst As QdLst Ptr) As Quad
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        QdLstDel pLst, @pLst.first
    End If
End Function

Function QdLstPopLast(ByVal pLst As QdLst Ptr) As Quad
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        QdLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function QdLstClone(ByVal pLst As QdLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As QdLst Ptr
    Local node As QdLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    Err = 0
    pClone = QdLstNew() : If Err Then Exit Function
    node = @pLst.first
    While node
        QdLstAdd pClone, @node.value
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function QdLstStore(ByVal pLst As QdLst Ptr) As String
    'store container to String
    Local s As String
    Local node As QdLstNode Ptr
    Local p As Quad Ptr
    ExitF(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    If @pLst.count Then
        s = Nul$(@pLst.count * %SizeQd)
        p = StrPtr(s)
        node = @pLst.first
        While node
            @p = @node.value : Incr p
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub QdLstRestore(ByVal pLst As QdLst Ptr, ByRef s As String)
    'restore container from string
    Local items As Long
    Local p As Quad Ptr
    ExitS(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    QdLstClear pLst
    If Len(s) Then
        p = StrPtr(s)
        items = Len(s) / %SizeQd
        While items
            QdLstAdd pLst, @p
            Incr p
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub QdLstFileStore(ByVal pLst As QdLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    s = QdLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub QdLstFileRestore(ByVal pLst As QdLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>QdLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        QdLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function QdLstNodeAlloc(ByVal pLst As QdLst Ptr) Private As Long
    Local node As QdLstNode Ptr
    node = MemAlloc(SizeOf(QdLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = QdLstNodeTag
    Incr @pLst.count
    Function = node
End Function
