#Include This Once
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\String\SsStr.inc"

Macro SsLstTag = 501260071
Type SsLst
    tag As Long
    lst As LnLst Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   String List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = SsLstNew() 'get handle for new container
    '       h = SsLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function SsLstNew() As Long
    'allocate new container - return handle
    Local pLst As SsLst Ptr
    Err = 0
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = SsLstTag
    @pLst.lst = LnLstNew() : If Err Then Exit Function
    Function = pLst
End Function

Function SsLstFinal(ByVal pLst As SsLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>SsLstTag, LibErrH)
        SsLstClear pLst
        @pLst.lst = LnLstFinal(@pLst.lst)
        MemFree(pLst)
    End If
End Function

Function SsLstValidate(ByVal pLst As SsLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = SsLstTag Then Function = @pLst.tag
End Function

Sub SsLstClear(ByVal pLst As SsLst Ptr)
    'delete all data
    Local node As LnLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    node = @pLst.@lst.first
    While node
        @node.value = SsFinal(@node.value)
        node = @node.next
    Wend
    LnLstClear @pLst.lst
End Sub

Function SsLstCount(ByVal pLst As SsLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    Function = @pLst.@lst.count
End Function

Sub SsLstAdd(ByVal pLst As SsLst Ptr, ByRef value As String)
    'append Value to end of List
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    LnLstAdd @pLst.lst, SsSetNew(value)
End Sub

Sub SsLstIns(ByVal pLst As SsLst Ptr, ByRef value As String)
    'insert Value at front of List
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    LnLstIns @pLst.lst, SsSetNew(value)
End Sub

Function SsLstFirst(ByVal pLst As SsLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    Function = @pLst.@lst.first
End Function

Function SsLstLast(ByVal pLst As SsLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    Function = @pLst.@lst.last
End Function

Function SsLstNext(ByVal pNode As LnLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function SsLstPrev(ByVal pNode As LnLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function SsLstGet(ByVal pNode As LnLstNode Ptr) As String
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    Function = SsGet(@pNode.value)
End Function

Sub SsLstSet(ByVal pNode As LnLstNode Ptr, ByRef value As String)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    SsSet @pNode.value, value
End Sub

Sub SsLstInsPrev(ByVal pLst As SsLst Ptr, ByVal pNode As LnLstNode Ptr, ByRef value As String)
    'insert Value before node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>SsLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstInsPrev @pLst.lst, pNode, SsSetNew(value)
End Sub

Sub SsLstInsNext(ByVal pLst As SsLst Ptr, ByVal pNode As LnLstNode Ptr, ByRef value As String)
    'insert Value after Cursor
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>SsLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstInsNext @pLst.lst, pNode, SsSetNew(value)
End Sub

Sub SsLstDel(ByVal pLst As SsLst Ptr, ByVal pNode As LnLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>LnLstNodeTag, LibErrH)
        SsFinal @pNode.value
        LnLstDel @pLst.lst, pNode
    End If
End Sub

Sub SsLstDelPrev(ByVal pLst As SsLst Ptr, ByVal pNode As LnLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>SsLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    SsLstDel pLst, @pNode.prev
End Sub

Sub SsLstDelNext(ByVal pLst As SsLst Ptr, ByVal pNode As LnLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>SsLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    SsLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SsLstStkPush(ByVal pLst As SsLst Ptr, ByRef value As String)
    'Push Value on Stack
    SsLstAdd pLst, value
End Sub

Function SsLstStkPeek(ByVal pLst As SsLst Ptr) As String
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    If @pLst.@lst.last Then Function = SsGet(@pLst.@lst.@last.value)
End Function

Function SsLstStkPop(ByVal pLst As SsLst Ptr) As String
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    If @pLst.@lst.last Then
        Function = SsGet(@pLst.@lst.@last.value)
        SsLstDel pLst, @pLst.@lst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SsLstQuePush(ByVal pLst As SsLst Ptr, ByRef value As String)
    'Add Value to end of Queue
    SsLstAdd pLst, value
End Sub

Function SsLstQuePeek(ByVal pLst As SsLst Ptr) As String
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    If @pLst.@lst.first Then Function = SsGet(@pLst.@lst.@first.value)
End Function

Function SsLstQuePop(ByVal pLst As SsLst Ptr) As String
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    If @pLst.@lst.first Then
        Function = SsGet(@pLst.@lst.@first.value)
        SsLstDel pLst, @pLst.@lst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub SsLstPushFirst(ByVal pLst As SsLst Ptr, ByRef value As String)
    'Add Value at front of container
    SsLstIns pLst, value
End Sub

Sub SsLstPushLast(ByVal pLst As SsLst Ptr, ByRef value As String)
    'Add Value at end of container
    SsLstAdd pLst, value
End Sub

Function SsLstPeekFirst(ByVal pLst As SsLst Ptr) As String
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    If @pLst.@lst.first Then Function = SsGet(@pLst.@lst.@first.value)
End Function

Function SsLstPeekLast(ByVal pLst As SsLst Ptr) As String
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    If @pLst.@lst.last Then Function = SsGet(@pLst.@lst.@last.value)
End Function

Function SsLstPopFirst(ByVal pLst As SsLst Ptr) As String
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    If @pLst.@lst.first Then
        Function = SsGet(@pLst.@lst.@first.value)
        SsLstDel pLst, @pLst.@lst.first
    End If
End Function

Function SsLstPopLast(ByVal pLst As SsLst Ptr) As String
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    If @pLst.@lst.last Then
        Function = SsGet(@pLst.@lst.@last.value)
        SsLstDel pLst, @pLst.@lst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SsLstClone(ByVal pLst As SsLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As SsLst Ptr
    Local node As LnLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    Err = 0
    pClone = SsLstNew() : If Err Then Exit Function
    node = @pLst.@lst.first
    While node
        SsLstAdd pClone, SsGet(@node.value)
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SsLstStore(ByVal pLst As SsLst Ptr) As String
    'store container to String
    Local tot, bytes As Long
    Local node As LnLstNode Ptr
    Local ps As SsStr Ptr
    Local s As String
    Local pl As Long Ptr
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    If @pLst.@lst.count Then
        tot = 4
        node = @pLst.@lst.first
        While node
            ps = @node.value
            ExitF(ps=0, LibErrN)
            tot += 4 + @ps.count
            node = @node.next
        Wend
        s = Nul$(tot)
        pl = StrPtr(s)
        @pl = @pLst.@lst.count : Incr pl
        node = @pLst.@lst.first
        While node
            ps = @node.value
            bytes = @ps.count
            @pl = bytes : Incr pl
            If bytes Then
                Memory Copy @ps.mem, pl, bytes : pl += bytes
            End If
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub SsLstRestore(ByVal pLst As SsLst Ptr, ByRef s As String)
    'restore container from string
    Local items, bytes As Long
    Local pl As Long Ptr
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    SsLstClear pLst
    If Len(s) Then
        pl = StrPtr(s)
        items = @pl : Incr pl
        While items
            bytes = @pl : Incr pl
            If bytes Then
                SsLstAdd pLst, Peek$(pl, bytes) : pl += bytes
            Else
                SsLstAdd pLst, ""
            End If
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SsLstFileStore(ByVal pLst As SsLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    s = SsLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsLstFileRestore(ByVal pLst As SsLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SsLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   String Builder Procedures
    '       use container's other procedures to Add/Insert/Modify/Remove string segments
    '----------------------------------------------------------------------------------------
'--

Function SsLstBuildStr(ByVal pLst As SsLst Ptr) As String
    'get complete string containing all string segments in container
    Local tot, hMem As Long
    Local node As LnLstNode Ptr
    Local ps As SsStr Ptr
    Local s As String
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    node = @pLst.@lst.first
    While node
        ps = @node.value
        tot += @ps.count
        node = @node.next
    Wend
    If tot Then
        s = Nul$(tot)
        hMem = StrPtr(s)
        node = @pLst.@lst.first
        While node
            ps = @node.value
            If @ps.count Then
                Memory Copy @ps.mem, hMem, @ps.count : hMem += @ps.count
            End If
            node = @node.next
        Wend
    End If
    Function = s
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Text Document Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SsLstTextLoad(ByVal pLst As SsLst Ptr, ByVal file As String)
    'load text file into container clearing current contents
    Local f  As Long
    Local s As String
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    SsLstClear pLst
    Try
        f = FreeFile
        Open file For Input As f
        While IsFalse Eof(f)
            Line Input# f, s
            SsLstAdd pLst, s
        Wend
    Catch
         ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsLstTextSave(ByVal pLst As SsLst Ptr, ByVal file As String)
    'save container to text file overwriting file's current contents
    Local ok, f  As Long
    Local node As LnLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Output As f
        node = @pLst.@lst.first
        While node
            Print# f, SsGet(@node.value)
            node = @node.next
        Wend
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Function SsLstTextGet(ByVal pLst As SsLst Ptr) As String
    'get container's contents as a Text Document, appending CrLf to each item
    Local tot As Long
    Local node As LnLstNode Ptr
    Local ps As SsStr Ptr
    Local pb As Byte Ptr
    Local s As String
    ExitF(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    node = @pLst.@lst.first
    While node
        ps = @node.value
        tot += @ps.count + 2
        node = @node.next
    Wend
    If tot Then
        s = Nul$(tot)
        pb = StrPtr(s)
        node = @pLst.@lst.first
        While node
            ps = @node.value
            If @ps.count Then
                Memory Copy @ps.mem, pb, @ps.count : pb += @ps.count
            End If
            @pb = 13 : Incr pb
            @pb = 10 : Incr pb
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub SsLstTextSet(ByVal pLst As SsLst Ptr, ByVal s As String)
    'replace container's contents with a Text Document - each line = item in container (removing CrLf)
    Local start, x, bytes As Long
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    SsLstClear pLst
    If Len(s) Then
        start = 1
        x = InStr(start, s, $CrLf)
        While x
            bytes = x - start
            SsLstAdd pLst, Mid$(s, start, bytes)
            start = x + 2
            x = InStr(start, s, $CrLf)
        Wend
        If start < Len(s) Then SsLstAdd pLst, Mid$(s, start)
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   File List Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SsLstGetFiles(ByVal pLst As SsLst Ptr, ByVal folder As String, ByVal mask As String)
    'load container with all files in folder matching the mask
    'loads file names without path
    Local file, s As String
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    SsLstClear pLst
    ExitS(IsFalse IsFolder(folder), LibErrA)
    s = RTrim$(folder, "\") + "\" + mask
    file = Dir$(s)
    While Len(file)
        SsLstAdd pLst, file
        file = Dir$(Next)
    Wend
End Sub

Sub SsLstGetPaths(ByVal pLst As SsLst Ptr, ByVal folder As String, ByVal mask As String)
    'load container with all files in folder matching the mask
    'loads full path
    Local file, s As String
    ExitS(pLst=0 Or @pLst.tag<>SsLstTag, LibErrH)
    SsLstClear pLst
    ExitS(IsFalse IsFolder(folder), LibErrA)
    folder = RTrim$(folder, "\") + "\"
    s = folder + mask
    file = Dir$(s)
    While Len(file)
        SsLstAdd pLst, folder + file
        file = Dir$(Next)
    Wend
End Sub
