#Include This Once
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\String\WsStr.inc"

Macro WsLstTag = -212996673
Type WsLst
    tag As Long
    lst As LnLst Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   WString List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = WsLstNew() 'get handle for new container
    '       h = WsLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function WsLstNew() As Long
    'allocate new container - return handle
    Local pLst As WsLst Ptr
    Err = 0
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = WsLstTag
    @pLst.lst = LnLstNew() : If Err Then Exit Function
    Function = pLst
End Function

Function WsLstFinal(ByVal pLst As WsLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>WsLstTag, LibErrH)
        WsLstClear pLst
        @pLst.lst = LnLstFinal(@pLst.lst)
        MemFree(pLst)
    End If
End Function

Function WsLstValidate(ByVal pLst As WsLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = WsLstTag Then Function = @pLst.tag
End Function

Sub WsLstClear(ByVal pLst As WsLst Ptr)
    'delete all data
    Local node As LnLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    node = @pLst.@lst.first
    While node
        @node.value = WsFinal(@node.value)
        node = @node.next
    Wend
    LnLstClear @pLst.lst
End Sub

Function WsLstCount(ByVal pLst As WsLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    Function = @pLst.@lst.count
End Function

Sub WsLstAdd(ByVal pLst As WsLst Ptr, ByRef value As WString)
    'append Value to end of List
    ExitS(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    LnLstAdd @pLst.lst, WsSetNew(value)
End Sub

Sub WsLstIns(ByVal pLst As WsLst Ptr, ByRef value As WString)
    'insert Value at front of List
    ExitS(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    LnLstIns @pLst.lst, WsSetNew(value)
End Sub

Function WsLstFirst(ByVal pLst As WsLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    Function = @pLst.@lst.first
End Function

Function WsLstLast(ByVal pLst As WsLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    Function = @pLst.@lst.last
End Function

Function WsLstNext(ByVal pNode As LnLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function WsLstPrev(ByVal pNode As LnLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function WsLstGet(ByVal pNode As LnLstNode Ptr) As WString
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    Function = WsGet(@pNode.value)
End Function

Sub WsLstSet(ByVal pNode As LnLstNode Ptr, ByRef value As WString)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    WsSet @pNode.value, value
End Sub

Sub WsLstInsPrev(ByVal pLst As WsLst Ptr, ByVal pNode As LnLstNode Ptr, ByRef value As WString)
    'insert Value before node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>WsLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstInsPrev @pLst.lst, pNode, WsSetNew(value)
End Sub

Sub WsLstInsNext(ByVal pLst As WsLst Ptr, ByVal pNode As LnLstNode Ptr, ByRef value As WString)
    'insert Value after Cursor
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>WsLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstInsNext @pLst.lst, pNode, WsSetNew(value)
End Sub

Sub WsLstDel(ByVal pLst As WsLst Ptr, ByVal pNode As LnLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>LnLstNodeTag, LibErrH)
        WsFinal @pNode.value
        LnLstDel @pLst.lst, pNode
    End If
End Sub

Sub WsLstDelPrev(ByVal pLst As WsLst Ptr, ByVal pNode As LnLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>WsLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    WsLstDel pLst, @pNode.prev
End Sub

Sub WsLstDelNext(ByVal pLst As WsLst Ptr, ByVal pNode As LnLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>WsLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    WsLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub WsLstStkPush(ByVal pLst As WsLst Ptr, ByRef value As WString)
    'Push Value on Stack
    WsLstAdd pLst, value
End Sub

Function WsLstStkPeek(ByVal pLst As WsLst Ptr) As WString
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    If @pLst.@lst.last Then Function = WsGet(@pLst.@lst.@last.value)
End Function

Function WsLstStkPop(ByVal pLst As WsLst Ptr) As WString
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    If @pLst.@lst.last Then
        Function = WsGet(@pLst.@lst.@last.value)
        WsLstDel pLst, @pLst.@lst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub WsLstQuePush(ByVal pLst As WsLst Ptr, ByRef value As WString)
    'Add Value to end of Queue
    WsLstAdd pLst, value
End Sub

Function WsLstQuePeek(ByVal pLst As WsLst Ptr) As WString
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    If @pLst.@lst.first Then Function = WsGet(@pLst.@lst.@first.value)
End Function

Function WsLstQuePop(ByVal pLst As WsLst Ptr) As WString
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    If @pLst.@lst.first Then
        Function = WsGet(@pLst.@lst.@first.value)
        WsLstDel pLst, @pLst.@lst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub WsLstPushFirst(ByVal pLst As WsLst Ptr, ByRef value As WString)
    'Add Value at front of container
    WsLstIns pLst, value
End Sub

Sub WsLstPushLast(ByVal pLst As WsLst Ptr, ByRef value As WString)
    'Add Value at end of container
    WsLstAdd pLst, value
End Sub

Function WsLstPeekFirst(ByVal pLst As WsLst Ptr) As WString
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    If @pLst.@lst.first Then Function = WsGet(@pLst.@lst.@first.value)
End Function

Function WsLstPeekLast(ByVal pLst As WsLst Ptr) As WString
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    If @pLst.@lst.last Then Function = WsGet(@pLst.@lst.@last.value)
End Function

Function WsLstPopFirst(ByVal pLst As WsLst Ptr) As WString
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    If @pLst.@lst.first Then
        Function = WsGet(@pLst.@lst.@first.value)
        WsLstDel pLst, @pLst.@lst.first
    End If
End Function

Function WsLstPopLast(ByVal pLst As WsLst Ptr) As WString
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    If @pLst.@lst.last Then
        Function = WsGet(@pLst.@lst.@last.value)
        WsLstDel pLst, @pLst.@lst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function WsLstClone(ByVal pLst As WsLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As WsLst Ptr
    Local node As LnLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    Err = 0
    pClone = WsLstNew() : If Err Then Exit Function
    node = @pLst.@lst.first
    While node
        WsLstAdd pClone, WsGet(@node.value)
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function WsLstStore(ByVal pLst As WsLst Ptr) As String
    'store container to String
    Local tot, bytes As Long
    Local node As LnLstNode Ptr
    Local ps As WsStr Ptr
    Local s As String
    Local pl As Long Ptr
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    If @pLst.@lst.count Then
        tot = 4
        node = @pLst.@lst.first
        While node
            ps = @node.value
            ExitF(ps=0, LibErrN)
            tot += 4 + (@ps.count * 2)
            node = @node.next
        Wend
        s = Nul$(tot)
        pl = StrPtr(s)
        @pl = @pLst.@lst.count : Incr pl
        node = @pLst.@lst.first
        While node
            ps = @node.value
            bytes = @ps.count * 2
            @pl = bytes : Incr pl
            If bytes Then
                Memory Copy @ps.mem, pl, bytes : pl += bytes
            End If
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub WsLstRestore(ByVal pLst As WsLst Ptr, ByRef s As String)
    'restore container from string
    Local items, bytes As Long
    Local pl As Long Ptr
    ExitS(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    WsLstClear pLst
    If Len(s) Then
        pl = StrPtr(s)
        items = @pl : Incr pl
        While items
            bytes = @pl : Incr pl
            If bytes Then
                WsLstAdd pLst, Peek$$(pl, bytes / 2) : pl += bytes
            Else
                WsLstAdd pLst, ""
            End If
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub WsLstFileStore(ByVal pLst As WsLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    s = WsLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub WsLstFileRestore(ByVal pLst As WsLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        WsLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   WString Builder Procedures
    '       use container's other procedures to Add/Insert/Modify/Remove string segments
    '----------------------------------------------------------------------------------------
'--

Function WsLstBuildWStr(ByVal pLst As WsLst Ptr) As WString
    'get complete string containing all string segments in container
    Local tot, hMem As Long
    Local node As LnLstNode Ptr
    Local ps As WsStr Ptr
    Local s As WString
    ExitF(pLst=0 Or @pLst.tag<>WsLstTag, LibErrH)
    node = @pLst.@lst.first
    While node
        ps = @node.value
        tot += @ps.count * 2
        node = @node.next
    Wend
    If tot Then
        s = Nul$(tot / 2)
        hMem = StrPtr(s)
        node = @pLst.@lst.first
        While node
            ps = @node.value
            If @ps.count Then
                Memory Copy @ps.mem, hMem, @ps.count * 2: hMem += (@ps.count * 2)
            End If
            node = @node.next
        Wend
    End If
    Function = s
End Function
