#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\CuLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'String/Currency ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - no limit on Key length
    '   - use SsCuTreComparison() to change how Keys compared
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = SsCuTreNew() 'get handle for new container
    '   h = SsCuTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro SsCuTreNodeTag = -2145892447
Macro SsCuTreTag = -174970846
Type SsCuTreNode
    tag As Long
    P As SsCuTreNode Ptr
    L As SsCuTreNode Ptr
    R As SsCuTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As Currency
End Type
Type SsCuTre
    tag As Long
    count As Long
    root As SsCuTreNode Ptr
    compareCB As Long
    collation As SsStr Ptr
End Type

Function SsCuTreNew() As Long
    'allocate new container - return handle
    Local p As SsCuTre Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = SsCuTreTag
    @p.compareCB = CodePtr(SsCompare)
    @p.collation = SsNew() : If Err Then Exit Function
    Function = p
End Function

Function SsCuTreFinal(ByVal pTree As SsCuTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>SsCuTreTag, LibErrH)
        @pTree.collation = SsFinal(@pTree.collation)
        SsCuTreClear pTree
        MemFree(pTree)
    End If
End Function

Function SsCuTreValidate(ByVal pTree As SsCuTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = SsCuTreTag Then Function = @pTree.tag
End Function

Sub SsCuTreComparison(ByVal pTree As SsCuTre Ptr, ByVal compareUCase As Long, ByVal collationSequence As String)
    'set how Strings compared
    'default = case ignored
    'if collationSequence String provided then
    '   Strings are compared using the order of the collation sequence String
    '   collation String must be 256 characters
    'else if compareUCase = True then
    '   Strings compared UCase
    ExitS(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    @pTree.compareCB = CodePtr(SsCompare)
    SsClear @pTree.collation
    If Len(collationSequence) Then
        ExitS(Len(collationSequence)<>256, LibErrS)
        SsSet @pTree.collation, collationSequence : If Err Then Exit Sub
        @pTree.compareCB = CodePtr(SsCompareCollate)
    ElseIf compareUCase Then
        @pTree.compareCB = CodePtr(SsCompareUCase)
    End If
End Sub

Sub SsCuTreClear(ByVal pTree As SsCuTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As SsCuTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = SsCuTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = SsCuTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            SsCuTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function SsCuTreCount(ByVal pTree As SsCuTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub SsCuTreSet(ByVal pTree As SsCuTre Ptr, ByRef key As String, ByVal value As Currency, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local compare, temp As Long
    Local n As SsCuTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Sub
    If @pTree.root Then
        n = @pTree.root
        While 1
            Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
            If compare > 0 Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = SsCuTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    SsSet @n.@R.K, key
                    @n.@R.V = value
                    SsCuTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf compare < 0 Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = SsCuTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    SsSet @n.@L.K, key
                    @n.@L.V = value
                    SsCuTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then @n.V = value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = SsCuTreAllocNode(pTree) : If Err Then Exit Sub
        SsSet @pTree.@root.K, key
        @pTree.@root.V = value
        @pTree.count = 1
    End If
    temp = SsFinal(temp)
End Sub

Function SsCuTreGet(ByVal pTree As SsCuTre Ptr, ByRef key As String) As Currency
    'get Key's associated Value
    Local compare, temp As Long
    Local n As SsCuTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = @n.V
            Exit Loop
        End If
    Wend
    temp = SsFinal(temp)
End Function

Function SsCuTreGot(ByVal pTree As SsCuTre Ptr, ByRef key As String) As Long
    'True/False if Key exist
    Local compare, temp As Long
    Local n As SsCuTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
    temp = SsFinal(temp)
End Function

Sub SsCuTreDel(ByVal pTree As SsCuTre Ptr, ByRef key As String)
    'remove Key and associated Value
    Local pNode As SsCuTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    pNode = SsCuTreGot(pTree, key)
    If pNode Then
        SsCuTreRemoveNode(pTree, pNode)
    End If
End Sub

Function SsCuTreFirst(ByVal pTree As SsCuTre Ptr) As Long
    'get handle to first node in tree
    Local n As SsCuTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function SsCuTreLast(ByVal pTree As SsCuTre Ptr) As Long
    'get handle to last node in tree
    Local n As SsCuTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function SsCuTreNext(ByVal pNode As SsCuTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As SsCuTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>SsCuTreNodeTag, LibErrH)
        minR = SsCuTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = SsCuTreParentGreater(pNode)
        End If
    End If
End Function

Function SsCuTrePrev(ByVal pNode As SsCuTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As SsCuTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>SsCuTreNodeTag, LibErrH)
        maxL = SsCuTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = SsCuTreParentLesser(pNode)
        End If
    End If
End Function

Function SsCuTreGetKey(ByVal pNode As SsCuTreNode Ptr) As String
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>SsCuTreNodeTag, LibErrH)
    Function = SsGet(@pNode.K)
End Function

Function SsCuTreGetVal(ByVal pNode As SsCuTreNode Ptr) As Currency
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>SsCuTreNodeTag, LibErrH)
    Function = @pNode.V
End Function

Sub SsCuTreSetVal(ByVal pNode As SsCuTreNode Ptr, ByVal value As Currency)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>SsCuTreNodeTag, LibErrH)
    @pNode.V = value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SsCuTreClone(ByVal pTree As SsCuTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    clone = SsCuTreNew() : If Err Then Exit Function
    h = SsCuTreFirst(pTree)
    While h
        SsCuTreSet clone, SsCuTreGetKey(h), SsCuTreGetVal(h)
        h = SsCuTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SsCuTreStore(ByVal pTree As SsCuTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    keys = SsLstNew() : If Err Then Exit Function
    vals = CuLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = SsCuTreFirst(pTree)
        While h
            SsLstAdd keys, SsCuTreGetKey(h)
            CuLstAdd vals, SsCuTreGetVal(h)
            h = SsCuTreNext(h)
        Wend
        SsLstAdd stor, SsLstStore(keys)
        SsLstAdd stor, CuLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = SsLstFinal(keys)
    vals = CuLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub SsCuTreRestore(ByVal pTree As SsCuTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    SsCuTreClear pTree
    keys = SsLstNew() : If Err Then Exit Sub
    vals = CuLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        SsLstRestore keys, SsLstPopFirst(stor)
        CuLstRestore vals, SsLstPopFirst(stor)
        ExitS(SsLstCount(keys)<>CuLstCount(vals), LibErrU)
        While SsLstCount(keys)
            SsCuTreSet pTree, SsLstPopFirst(keys), CuLstPopFirst(vals)
        Wend
    End If
    keys = SsLstFinal(keys)
    vals = CuLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SsCuTreFileStore(ByVal pTree As SsCuTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    s = SsCuTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsCuTreFileRestore(ByVal pTree As SsCuTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsCuTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SsCuTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub SsCuTreRemoveNode(ByVal p As SsCuTre Ptr, ByVal n As SsCuTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As SsCuTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, SsCuTreMaxLeft(n), SsCuTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        SsCuTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = SsCuTreFreeNode(p, n)
        SsCuTreBalanceBranch p, nP
    End If
End Sub

Function SsCuTreAllocNode(ByVal p As SsCuTre Ptr) Private As Long
    Local n As SsCuTreNode Ptr
    n = MemAlloc(SizeOf(SsCuTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = SsCuTreNodeTag
    @n.K = SsNew() : If Err Then Exit Function
    Incr @p.count
    Function = n
End Function

Function SsCuTreFreeNode(ByVal p As SsCuTre Ptr, ByVal n As SsCuTreNode Ptr) As Long
    If n Then
        @n.K = SsFinal(@n.K)
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub SsCuTreBalanceBranch(ByVal p As SsCuTre Ptr, ByVal n As SsCuTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = SsCuTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = SsCuTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function SsCuTreMaxLeft(ByVal n As SsCuTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function SsCuTreMinRight(ByVal n As SsCuTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function SsCuTreParentGreater(ByVal n As SsCuTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function SsCuTreParentLesser(ByVal n As SsCuTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function SsCuTreRotateLeft(ByVal p As SsCuTre Ptr, ByVal n As SsCuTreNode Ptr) Private As Long
    Local nR, nRL As SsCuTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function SsCuTreRotateRight(ByVal p As SsCuTre Ptr, ByVal n As SsCuTreNode Ptr) Private As Long
    Local nL, nLR As SsCuTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
