#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\CxLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'String/CurrencyX ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - no limit on Key length
    '   - use SsCxTreComparison() to change how Keys compared
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = SsCxTreNew() 'get handle for new container
    '   h = SsCxTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro SsCxTreNodeTag = -662146394
Macro SsCxTreTag = -281548639
Type SsCxTreNode
    tag As Long
    P As SsCxTreNode Ptr
    L As SsCxTreNode Ptr
    R As SsCxTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As CurrencyX
End Type
Type SsCxTre
    tag As Long
    count As Long
    root As SsCxTreNode Ptr
    compareCB As Long
    collation As SsStr Ptr
End Type

Function SsCxTreNew() As Long
    'allocate new container - return handle
    Local p As SsCxTre Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = SsCxTreTag
    @p.compareCB = CodePtr(SsCompare)
    @p.collation = SsNew() : If Err Then Exit Function
    Function = p
End Function

Function SsCxTreFinal(ByVal pTree As SsCxTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>SsCxTreTag, LibErrH)
        @pTree.collation = SsFinal(@pTree.collation)
        SsCxTreClear pTree
        MemFree(pTree)
    End If
End Function

Function SsCxTreValidate(ByVal pTree As SsCxTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = SsCxTreTag Then Function = @pTree.tag
End Function

Sub SsCxTreComparison(ByVal pTree As SsCxTre Ptr, ByVal compareUCase As Long, ByVal collationSequence As String)
    'set how Strings compared
    'default = case ignored
    'if collationSequence String provided then
    '   Strings are compared using the order of the collation sequence String
    '   collation String must be 256 characters
    'else if compareUCase = True then
    '   Strings compared UCase
    ExitS(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    @pTree.compareCB = CodePtr(SsCompare)
    SsClear @pTree.collation
    If Len(collationSequence) Then
        ExitS(Len(collationSequence)<>256, LibErrS)
        SsSet @pTree.collation, collationSequence : If Err Then Exit Sub
        @pTree.compareCB = CodePtr(SsCompareCollate)
    ElseIf compareUCase Then
        @pTree.compareCB = CodePtr(SsCompareUCase)
    End If
End Sub

Sub SsCxTreClear(ByVal pTree As SsCxTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As SsCxTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = SsCxTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = SsCxTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            SsCxTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function SsCxTreCount(ByVal pTree As SsCxTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub SsCxTreSet(ByVal pTree As SsCxTre Ptr, ByRef key As String, ByVal value As CurrencyX, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local compare, temp As Long
    Local n As SsCxTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Sub
    If @pTree.root Then
        n = @pTree.root
        While 1
            Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
            If compare > 0 Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = SsCxTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    SsSet @n.@R.K, key
                    @n.@R.V = value
                    SsCxTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf compare < 0 Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = SsCxTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    SsSet @n.@L.K, key
                    @n.@L.V = value
                    SsCxTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then @n.V = value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = SsCxTreAllocNode(pTree) : If Err Then Exit Sub
        SsSet @pTree.@root.K, key
        @pTree.@root.V = value
        @pTree.count = 1
    End If
    temp = SsFinal(temp)
End Sub

Function SsCxTreGet(ByVal pTree As SsCxTre Ptr, ByRef key As String) As CurrencyX
    'get Key's associated Value
    Local compare, temp As Long
    Local n As SsCxTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = @n.V
            Exit Loop
        End If
    Wend
    temp = SsFinal(temp)
End Function

Function SsCxTreGot(ByVal pTree As SsCxTre Ptr, ByRef key As String) As Long
    'True/False if Key exist
    Local compare, temp As Long
    Local n As SsCxTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
    temp = SsFinal(temp)
End Function

Sub SsCxTreDel(ByVal pTree As SsCxTre Ptr, ByRef key As String)
    'remove Key and associated Value
    Local pNode As SsCxTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    pNode = SsCxTreGot(pTree, key)
    If pNode Then
        SsCxTreRemoveNode(pTree, pNode)
    End If
End Sub

Function SsCxTreFirst(ByVal pTree As SsCxTre Ptr) As Long
    'get handle to first node in tree
    Local n As SsCxTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function SsCxTreLast(ByVal pTree As SsCxTre Ptr) As Long
    'get handle to last node in tree
    Local n As SsCxTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function SsCxTreNext(ByVal pNode As SsCxTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As SsCxTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>SsCxTreNodeTag, LibErrH)
        minR = SsCxTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = SsCxTreParentGreater(pNode)
        End If
    End If
End Function

Function SsCxTrePrev(ByVal pNode As SsCxTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As SsCxTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>SsCxTreNodeTag, LibErrH)
        maxL = SsCxTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = SsCxTreParentLesser(pNode)
        End If
    End If
End Function

Function SsCxTreGetKey(ByVal pNode As SsCxTreNode Ptr) As String
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>SsCxTreNodeTag, LibErrH)
    Function = SsGet(@pNode.K)
End Function

Function SsCxTreGetVal(ByVal pNode As SsCxTreNode Ptr) As CurrencyX
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>SsCxTreNodeTag, LibErrH)
    Function = @pNode.V
End Function

Sub SsCxTreSetVal(ByVal pNode As SsCxTreNode Ptr, ByVal value As CurrencyX)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>SsCxTreNodeTag, LibErrH)
    @pNode.V = value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SsCxTreClone(ByVal pTree As SsCxTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    clone = SsCxTreNew() : If Err Then Exit Function
    h = SsCxTreFirst(pTree)
    While h
        SsCxTreSet clone, SsCxTreGetKey(h), SsCxTreGetVal(h)
        h = SsCxTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SsCxTreStore(ByVal pTree As SsCxTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    keys = SsLstNew() : If Err Then Exit Function
    vals = CxLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = SsCxTreFirst(pTree)
        While h
            SsLstAdd keys, SsCxTreGetKey(h)
            CxLstAdd vals, SsCxTreGetVal(h)
            h = SsCxTreNext(h)
        Wend
        SsLstAdd stor, SsLstStore(keys)
        SsLstAdd stor, CxLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = SsLstFinal(keys)
    vals = CxLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub SsCxTreRestore(ByVal pTree As SsCxTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    SsCxTreClear pTree
    keys = SsLstNew() : If Err Then Exit Sub
    vals = CxLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        SsLstRestore keys, SsLstPopFirst(stor)
        CxLstRestore vals, SsLstPopFirst(stor)
        ExitS(SsLstCount(keys)<>CxLstCount(vals), LibErrU)
        While SsLstCount(keys)
            SsCxTreSet pTree, SsLstPopFirst(keys), CxLstPopFirst(vals)
        Wend
    End If
    keys = SsLstFinal(keys)
    vals = CxLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SsCxTreFileStore(ByVal pTree As SsCxTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    s = SsCxTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsCxTreFileRestore(ByVal pTree As SsCxTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsCxTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SsCxTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub SsCxTreRemoveNode(ByVal p As SsCxTre Ptr, ByVal n As SsCxTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As SsCxTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, SsCxTreMaxLeft(n), SsCxTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        SsCxTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = SsCxTreFreeNode(p, n)
        SsCxTreBalanceBranch p, nP
    End If
End Sub

Function SsCxTreAllocNode(ByVal p As SsCxTre Ptr) Private As Long
    Local n As SsCxTreNode Ptr
    n = MemAlloc(SizeOf(SsCxTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = SsCxTreNodeTag
    @n.K = SsNew() : If Err Then Exit Function
    Incr @p.count
    Function = n
End Function

Function SsCxTreFreeNode(ByVal p As SsCxTre Ptr, ByVal n As SsCxTreNode Ptr) As Long
    If n Then
        @n.K = SsFinal(@n.K)
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub SsCxTreBalanceBranch(ByVal p As SsCxTre Ptr, ByVal n As SsCxTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = SsCxTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = SsCxTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function SsCxTreMaxLeft(ByVal n As SsCxTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function SsCxTreMinRight(ByVal n As SsCxTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function SsCxTreParentGreater(ByVal n As SsCxTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function SsCxTreParentLesser(ByVal n As SsCxTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function SsCxTreRotateLeft(ByVal p As SsCxTre Ptr, ByVal n As SsCxTreNode Ptr) Private As Long
    Local nR, nRL As SsCxTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function SsCxTreRotateRight(ByVal p As SsCxTre Ptr, ByVal n As SsCxTreNode Ptr) Private As Long
    Local nL, nLR As SsCxTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
