#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro BtArrTag = 1620720082
Type BtArr
    tag As Long
    count As Long
    arr As Byte Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Byte Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = BtArrNew() 'get handle for new container
    '       h = BtArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function BtArrNew() As Long
    'allocate new container - return handle
    Local p As BtArr Ptr
    p = MemAlloc(SizeOf(BtArr))
    ExitF(p=0, LibErrM)
    @p.tag = BtArrTag
    Function = p
End Function

Function BtArrFinal(ByVal p As BtArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>BtArrTag, LibErrH)
        BtArrClear p
        MemFree(p)
    End If
End Function

Function BtArrValidate(ByVal p As BtArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = BtArrTag Then Function = @p.tag
End Function

Sub BtArrClear(ByVal p As BtArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeBt
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function BtArrCount(ByVal p As BtArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    Function = @p.count
End Function

Sub BtArrReDim(ByVal p As BtArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        BtArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeBt
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeBt)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeBt
    End If
End Sub

Function BtArrGet(ByVal p As BtArr Ptr, ByVal index As Long) As Byte
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub BtArrSet(ByVal p As BtArr Ptr, ByVal index As Long, ByVal value As Byte)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub BtArrAdd(ByVal p As BtArr Ptr, ByVal value As Byte)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    BtArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub BtArrIns(ByVal p As BtArr Ptr, ByVal index As Long, ByVal value As Byte)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    BtArrReDim p, @p.count + 1 : If Err Then Exit Sub
    BtArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub BtArrDel(ByVal p As BtArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then BtArrMove @p.arr, i + 1, i, @p.count - i
    BtArrReDim p, @p.count - 1
End Sub

Sub BtArrSort(ByVal p As BtArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As Byte
    Local arr As Byte Ptr
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub BtArrUniqueSort(ByVal p As BtArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As Byte Ptr
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    BtArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then BtArrDel p, i + 1
    Next i
End Sub

Sub BtArrReverse(ByVal p As BtArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As Byte Ptr
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function BtArrBinSearch(ByVal p As BtArr Ptr, ByVal value As Byte) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As Byte Ptr
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub BtArrBinInsert(ByVal p As BtArr Ptr, ByVal value As Byte)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    i = BtArrBinPos(p, value, compare)
    If i = 0 Then
        BtArrAdd p, value
    ElseIf compare <= 0 Then
        BtArrIns p, i, value
    ElseIf i < @p.count Then
        BtArrIns p, i + 1, value
    Else
        BtArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Byte Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub BtArrStkPush(ByVal p As BtArr Ptr, ByVal value As Byte)
    'push value on top of Stack
    BtArrAdd p, value
End Sub

Function BtArrStkPeek(ByVal p As BtArr Ptr) As Byte
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function BtArrStkPop(ByVal p As BtArr Ptr) As Byte
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        BtArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Byte Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub BtArrQuePush(ByVal p As BtArr Ptr, ByVal value As Byte)
    'add value to end of Queue
    BtArrAdd p, value
End Sub

Function BtArrQuePeek(ByVal p As BtArr Ptr) As Byte
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function BtArrQuePop(ByVal p As BtArr Ptr) As Byte
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        BtArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Byte Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub BtArrPushFirst(ByVal p As BtArr Ptr, ByVal value As Byte)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then BtArrIns p, 1, value Else BtArrAdd p, value
End Sub

Sub BtArrPushLast(ByVal p As BtArr Ptr, ByVal value As Byte)
    'add value to and of Deque (double-ended Queue)
    BtArrAdd p, value
End Sub

Function BtArrPeekFirst(ByVal p As BtArr Ptr) As Byte
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function BtArrPeekLast(ByVal p As BtArr Ptr) As Byte
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function BtArrPopFirst(ByVal p As BtArr Ptr) As Byte
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        BtArrDel p, 1
    End If
End Function

Function BtArrPopLast(ByVal p As BtArr Ptr) As Byte
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        BtArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function BtArrClone(ByVal p As BtArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As BtArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    clone = BtArrNew() : If Err Then Exit Function
    If @p.count Then
        BtArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeBt, @clone.arr + %SizeBt, @p.count * %SizeBt
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function BtArrStore(ByVal p As BtArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>BtArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeBt, @p.count * %SizeBt)
End Function

Sub BtArrRestore(ByVal p As BtArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    BtArrClear p
    If strLen Then
        BtArrReDim p, strLen / %SizeBt : If Err Then Exit Sub
        Poke$ @p.arr + %SizeBt, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub BtArrFileStore(ByVal p As BtArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    s = BtArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub BtArrFileRestore(ByVal p As BtArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>BtArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        BtArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub BtArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeBt), arr + (toIndex * %SizeBt),  Count * %SizeBt
End Sub

Function BtArrBinPos(ByVal p As BtArr Ptr, ByVal value As Byte, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Byte Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function
