#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro SnArrTag = 84537623
Type SnArr
    tag As Long
    count As Long
    arr As Single Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Single Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = SnArrNew() 'get handle for new container
    '       h = SnArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function SnArrNew() As Long
    'allocate new container - return handle
    Local p As SnArr Ptr
    p = MemAlloc(SizeOf(SnArr))
    ExitF(p=0, LibErrM)
    @p.tag = SnArrTag
    Function = p
End Function

Function SnArrFinal(ByVal p As SnArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>SnArrTag, LibErrH)
        SnArrClear p
        MemFree(p)
    End If
End Function

Function SnArrValidate(ByVal p As SnArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = SnArrTag Then Function = @p.tag
End Function

Sub SnArrClear(ByVal p As SnArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeSn
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function SnArrCount(ByVal p As SnArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    Function = @p.count
End Function

Sub SnArrReDim(ByVal p As SnArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        SnArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeSn
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeSn)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeSn
    End If
End Sub

Function SnArrGet(ByVal p As SnArr Ptr, ByVal index As Long) As Single
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub SnArrSet(ByVal p As SnArr Ptr, ByVal index As Long, ByVal value As Single)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub SnArrAdd(ByVal p As SnArr Ptr, ByVal value As Single)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    SnArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub SnArrIns(ByVal p As SnArr Ptr, ByVal index As Long, ByVal value As Single)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    SnArrReDim p, @p.count + 1 : If Err Then Exit Sub
    SnArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub SnArrDel(ByVal p As SnArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then SnArrMove @p.arr, i + 1, i, @p.count - i
    SnArrReDim p, @p.count - 1
End Sub

Sub SnArrSort(ByVal p As SnArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As Single
    Local arr As Single Ptr
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub SnArrUniqueSort(ByVal p As SnArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As Single Ptr
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    SnArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then SnArrDel p, i + 1
    Next i
End Sub

Sub SnArrReverse(ByVal p As SnArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As Single Ptr
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function SnArrBinSearch(ByVal p As SnArr Ptr, ByVal value As Single) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As Single Ptr
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub SnArrBinInsert(ByVal p As SnArr Ptr, ByVal value As Single)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    i = SnArrBinPos(p, value, compare)
    If i = 0 Then
        SnArrAdd p, value
    ElseIf compare <= 0 Then
        SnArrIns p, i, value
    ElseIf i < @p.count Then
        SnArrIns p, i + 1, value
    Else
        SnArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Single Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SnArrStkPush(ByVal p As SnArr Ptr, ByVal value As Single)
    'push value on top of Stack
    SnArrAdd p, value
End Sub

Function SnArrStkPeek(ByVal p As SnArr Ptr) As Single
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function SnArrStkPop(ByVal p As SnArr Ptr) As Single
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        SnArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Single Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SnArrQuePush(ByVal p As SnArr Ptr, ByVal value As Single)
    'add value to end of Queue
    SnArrAdd p, value
End Sub

Function SnArrQuePeek(ByVal p As SnArr Ptr) As Single
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function SnArrQuePop(ByVal p As SnArr Ptr) As Single
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        SnArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Single Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SnArrPushFirst(ByVal p As SnArr Ptr, ByVal value As Single)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then SnArrIns p, 1, value Else SnArrAdd p, value
End Sub

Sub SnArrPushLast(ByVal p As SnArr Ptr, ByVal value As Single)
    'add value to and of Deque (double-ended Queue)
    SnArrAdd p, value
End Sub

Function SnArrPeekFirst(ByVal p As SnArr Ptr) As Single
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function SnArrPeekLast(ByVal p As SnArr Ptr) As Single
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function SnArrPopFirst(ByVal p As SnArr Ptr) As Single
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        SnArrDel p, 1
    End If
End Function

Function SnArrPopLast(ByVal p As SnArr Ptr) As Single
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        SnArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SnArrClone(ByVal p As SnArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As SnArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    clone = SnArrNew() : If Err Then Exit Function
    If @p.count Then
        SnArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeSn, @clone.arr + %SizeSn, @p.count * %SizeSn
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SnArrStore(ByVal p As SnArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>SnArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeSn, @p.count * %SizeSn)
End Function

Sub SnArrRestore(ByVal p As SnArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    SnArrClear p
    If strLen Then
        SnArrReDim p, strLen / %SizeSn : If Err Then Exit Sub
        Poke$ @p.arr + %SizeSn, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SnArrFileStore(ByVal p As SnArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    s = SnArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SnArrFileRestore(ByVal p As SnArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>SnArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SnArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub SnArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeSn), arr + (toIndex * %SizeSn),  Count * %SizeSn
End Sub

Function SnArrBinPos(ByVal p As SnArr Ptr, ByVal value As Single, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Single Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function
