#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\Hash.inc"
#Include Once "C:\HLib3\String\SsStr.inc"
#Include Once "C:\HLib3\List\SsLst.inc"

'++
    '----------------------------------------------------------------------------------------
    '   String/String ~ Hash Table Container
    '       http://en.wikipedia.org/wiki/Hash_table
    '       Key must be unique String
    '       Key is case sensitive - no nulls
    '       Value stored/retrieved/removed using unique lookup Key
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = SsSsHshNew() 'get handle for new container
    '       h = SsSsHshFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Macro SsSsHshTag = 667850440
Type SsSsHshNode
    next As SsSsHshNode Ptr
    K As SsStr Ptr
    V As SsStr Ptr
End Type
Type SsSsHsh
    tag As Long
    count As Long
    cap As Long
    arr As Long Ptr
End Type

Function SsSsHshNew(ByVal capacity As Long) As Long
    'allocate new container - return handle
    'capacity = number of expected Keys (minium 100)
    Local i As Long
    Local p As SsSsHsh Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = SsSsHshTag
    @p.cap = Max&(capacity, 10)
    @p.arr = MemAlloc(@p.cap * 4)
    ExitF(@p.arr=0, LibErrM)
    Function = p
End Function

Function SsSsHshFinal(ByVal p As SsSsHsh Ptr) As Long
    'free allocated container - return null
    Local i As Long
    Local n, x As SsSsHshNode Ptr
    If p Then
        ExitF(@p.tag<>SsSsHshTag, LibErrH)
        For i = 0 To @p.cap - 1
            n = @p.@arr[i] : @p.@arr[i] = 0
            While n
                x = n : n = @n.next
                SsFinal(@x.K)
                SsFinal(@x.V)
                MemFree(x)
            Wend
        Next i
        @p.arr = MemFree(@p.arr)
    End If
End Function

Function SsSsHshValidate(ByVal p As SsSsHsh Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = SsSsHshTag Then Function = @p.tag
End Function

Sub SsSsHshClear(ByVal p As SsSsHsh Ptr)
    'delete all data
    Local i As Long
    Local n, x As SsSsHshNode Ptr
    ExitS(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    For i = 0 To  @p.cap - 1
        n = @p.@arr[i] : @p.@arr[i] = 0
        While n
            x = n : n = @n.next
            SsFinal(@x.K)
            SsFinal(@x.V)
            MemFree(x)
        Wend
    Next i
    @p.count = 0
End Sub

Function SsSsHshCount(ByVal p As SsSsHsh Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    Function = @p.count
End Function

Function SsSsHshCapGet(ByVal p As SsSsHsh Ptr) As Long
    'get Hash Table capacity
    ExitF(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    Function = @p.cap
End Function

Sub SsSsHshCapSet(ByVal p As SsSsHsh Ptr, ByVal capacity As Long)
    'set Hash Table capacity
    'rebuild Hash Table with new capacity - data preserved
    'capacity should be about the same as number of stored Keys
    Local i, oldCap, newCap As Long
    Local arr As Long Ptr
    Local n, x As SsSsHshNode Ptr
    ExitS(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    oldCap = @p.cap
    newCap = Max&(capacity, 10)
    arr = MemAlloc(newCap * 4)
    ExitS(arr=0, LibErrM)
    Swap @p.arr, arr
    @p.cap = newCap
    @p.count = 0
    For i = 0 To oldCap - 1
        n = @arr[i] : @arr[i] = 0
        While n
            x = n : n = @n.next
            SsSsHshSet p, SsGet(@x.K), SsGet(@x.V)
            SsFinal(@x.K)
            SsFinal(@x.V)
            MemFree(x)
        Wend
    Next i
    MemFree(arr)
End Sub

Sub SsSsHshSet(ByVal p As SsSsHsh Ptr, ByRef key As String, ByRef value As String, Opt ByVal DontReplace As Byte)
    'add Key/Value to Tash Table - Value replaced if Key exist unless DontReplace = True
    Local i, equal As Long
    Local ps As String Ptr : ps = StrPtr(key)
    Local n, nn As SsSsHshNode Ptr
    ExitS(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    i = StrHash(ps, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    If n Then
        Do
            equal = SsEqual(@n.@K.mem, ps)
            If equal Then
                If IsFalse DontReplace Then SsSet @n.V, value
                Exit Sub
            ElseIf @n.next Then
                n = @n.next
            Else
                Exit Loop
            End If
        Loop
        nn = MemAlloc(SizeOf(SsSsHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = SsSetNew(key)
        @nn.V = SsSetNew(value)
        @n.next = nn
        Incr @p.count
    Else
        nn = MemAlloc(SizeOf(SsSsHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = SsSetNew(key)
        @nn.V = SsSetNew(value)
        @p.@arr[i] = nn
        Incr @p.count
    End If
End Sub

Function SsSsHshGet(ByVal p As SsSsHsh Ptr, ByRef key As String) As String
    'get Key's associated Value
    Local i, equal As Long
    Local ps As String Ptr : ps = StrPtr(key)
    Local n As SsSsHshNode Ptr
    ExitF(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    i = StrHash(ps, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        equal = SsEqual(@n.@K.mem, ps)
        If equal Then
            Function = SsGet(@n.V)
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Function SsSsHshGot(ByVal p As SsSsHsh Ptr, ByRef key As String) As Byte
    'True/False if Key in Hash Table
    Local i, equal As Long
    Local ps As String Ptr : ps = StrPtr(key)
    Local n As SsSsHshNode Ptr
    ExitF(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    i = StrHash(ps, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        equal = SsEqual(@n.@K.mem, ps)
        If equal Then
            Function = 1
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Sub SsSsHshDel(ByVal p As SsSsHsh Ptr, ByRef key As String)
    'remove Key and associated Value from Hash Table
    Local i, equal As Long
    Local ps As String Ptr : ps = StrPtr(key)
    Local prev, n As SsSsHshNode Ptr
    ExitS(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    i = StrHash(ps, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    prev = 0
    n = @p.@arr[i]
    While n
        equal = SsEqual(@n.@K.mem, ps)
        If equal Then
            If prev Then
                @prev.next = @n.next
            Else
                @p.@arr[i] = @n.next
            End If
            @n.K = SsFinal(@n.K)
            @n.V = SsFinal(@n.V)
            MemFree(n)
            ExitS(@p.count=0, LibErrU)
            Decr @p.count
            Exit Loop
        End If
        prev = n
        n = @n.next
    Wend
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SsSsHshClone(ByVal p As SsSsHsh Ptr) As Long
    'create duplicate container - return handle to cloned container
    Local i As Long
    Local n As SsSsHshNode Ptr
    Local clone As Long
    Err = 0
    ExitF(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    clone = SsSsHshNew(@p.cap) : If Err Then Exit Function
    For i = 0 To @p.cap - 1
        n = @p.@arr[i]
        While n
            SsSsHshSet clone, SsGet(@n.K), SsGet(@n.V)
            n = @n.next
        Wend
    Next i
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SsSsHshStore(ByVal p As SsSsHsh Ptr) As String
    'store container to string
    Local i As Long
    Local keys, vals, store As Long
    Local n As SsSsHshNode Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    keys = SsLstNew() : If Err Then Exit Function
    vals = SsLstNew() : If Err Then Exit Function
    store = SsLstNew() : If Err Then Exit Function
    If @p.count Then
        For i = 0 To @p.cap - 1
            n = @p.@arr[i]
            While n
                SsLstAdd keys, SsGet(@n.K)
                SsLstAdd vals, SsGet(@n.V)
                n = @n.next
            Wend
        Next i
        SsLstAdd store, Mkl$(@p.cap)
        SsLstAdd store, SsLstStore(keys)
        SsLstAdd store, SsLstStore(vals)
        Function = SsLstStore(store)
    End If
    keys = SsLstFinal(keys)
    vals = SsLstFinal(vals)
    store = SsLstFinal(store)
End Function

Sub SsSsHshRestore(ByVal p As SsSsHsh Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, store As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    SsSsHshClear p
    keys = SsLstNew() : If Err Then Exit Sub
    vals = SsLstNew() : If Err Then Exit Sub
    store = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore store, s : If Err Then Exit Sub
        ExitS(SsLstCount(store)<>3, LibErrU)
        SsSsHshCapSet p, Cvl(SsLstPopFirst(store))
        SsLstRestore keys, SsLstPopFirst(store)
        SsLstRestore vals, SsLstPopFirst(store)
        ExitS(SsLstCount(keys)<>SsLstCount(vals), LibErrU)
        While SsLstCount(keys)
            SsSsHshSet p, SsLstPopFirst(keys), SsLstPopFirst(vals)
        Wend
    End If
    keys = SsLstFinal(keys)
    vals = SsLstFinal(vals)
    store = SsLstFinal(store)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SsSsHshFileStore(ByVal p As SsSsHsh Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    s = SsSsHshStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsSsHshFileRestore(ByVal p As SsSsHsh Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>SsSsHshTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SsSsHshRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub
