#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\Hash.inc"
#Include Once "C:\HLib3\String\WsStr.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\WsLst.inc"
#Include Once "C:\HLib3\List\DbLst.inc"

'++
    '----------------------------------------------------------------------------------------
    '   WString/Double ~ Hash Table Container
    '       http://en.wikipedia.org/wiki/Hash_table
    '       Key must be unique WString
    '       Key is case sensitive - no nulls
    '       Value stored/retrieved/removed using unique lookup Key
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = WsDbHshNew() 'get handle for new container
    '       h = WsDbHshFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Macro WsDbHshTag = -252380731
Type WsDbHshNode
    next As WsDbHshNode Ptr
    K As WsStr Ptr
    V As Double
End Type
Type WsDbHsh
    tag As Long
    count As Long
    cap As Long
    arr As Long Ptr
End Type

Function WsDbHshNew(ByVal capacity As Long) As Long
    'allocate new container - return handle
    'capacity = number of expected Keys (minium 100)
    Local i As Long
    Local p As WsDbHsh Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = WsDbHshTag
    @p.cap = Max&(capacity, 10)
    @p.arr = MemAlloc(@p.cap * 4)
    ExitF(@p.arr=0, LibErrM)
    Function = p
End Function

Function WsDbHshFinal(ByVal p As WsDbHsh Ptr) As Long
    'free allocated container - return null
    Local i As Long
    Local n, x As WsDbHshNode Ptr
    If p Then
        ExitF(@p.tag<>WsDbHshTag, LibErrH)
        For i = 0 To @p.cap - 1
            n = @p.@arr[i] : @p.@arr[i] = 0
            While n
                x = n : n = @n.next
                WsFinal(@x.K)
                MemFree(x)
            Wend
        Next i
        @p.arr = MemFree(@p.arr)
    End If
End Function

Function WsDbHshValidate(ByVal p As WsDbHsh Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = WsDbHshTag Then Function = @p.tag
End Function

Sub WsDbHshClear(ByVal p As WsDbHsh Ptr)
    'delete all data
    Local i As Long
    Local n, x As WsDbHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    For i = 0 To  @p.cap - 1
        n = @p.@arr[i] : @p.@arr[i] = 0
        While n
            x = n : n = @n.next
            WsFinal(@x.K)
            MemFree(x)
        Wend
    Next i
    @p.count = 0
End Sub

Function WsDbHshCount(ByVal p As WsDbHsh Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    Function = @p.count
End Function

Function WsDbHshCapGet(ByVal p As WsDbHsh Ptr) As Long
    'get Hash Table capacity
    ExitF(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    Function = @p.cap
End Function

Sub WsDbHshCapSet(ByVal p As WsDbHsh Ptr, ByVal capacity As Long)
    'set Hash Table capacity
    'rebuild Hash Table with new capacity - data preserved
    'capacity should be about the same as number of stored Keys
    Local i, oldCap, newCap As Long
    Local arr As Long Ptr
    Local n, x As WsDbHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    oldCap = @p.cap
    newCap = Max&(capacity, 10)
    arr = MemAlloc(newCap * 4)
    ExitS(arr=0, LibErrM)
    Swap @p.arr, arr
    @p.cap = newCap
    @p.count = 0
    For i = 0 To oldCap - 1
        n = @arr[i] : @arr[i] = 0
        While n
            x = n : n = @n.next
            WsDbHshSet p, WsGet(@x.K), @x.V
            WsFinal(@x.K)
            MemFree(x)
        Wend
    Next i
    MemFree(arr)
End Sub

Sub WsDbHshSet(ByVal p As WsDbHsh Ptr, ByRef key As WString, ByVal value As Double, Opt ByVal DontReplace As Byte)
    'add Key/Value to Tash Table - Value replaced if Key exist unless DontReplace = True
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local n, nn As WsDbHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    If n Then
        Do
            equal = WsEqual(@n.@K.mem, ps)
            If equal Then
                If IsFalse DontReplace Then @n.V = value
                Exit Sub
            ElseIf @n.next Then
                n = @n.next
            Else
                Exit Loop
            End If
        Loop
        nn = MemAlloc(SizeOf(WsDbHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = WsSetNew(key)
        @nn.V = value
        @n.next = nn
        Incr @p.count
    Else
        nn = MemAlloc(SizeOf(WsDbHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = WsSetNew(key)
        @nn.V = value
        @p.@arr[i] = nn
        Incr @p.count
    End If
End Sub

Function WsDbHshGet(ByVal p As WsDbHsh Ptr, ByRef key As WString) As Double
    'get Key's associated Value
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local n As WsDbHshNode Ptr
    ExitF(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        equal = WsEqual(@n.@K.mem, ps)
        If equal Then
            Function = @n.V
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Function WsDbHshGot(ByVal p As WsDbHsh Ptr, ByRef key As WString) As Byte
    'True/False if Key in Hash Table
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local n As WsDbHshNode Ptr
    ExitF(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        equal = WsEqual(@n.@K.mem, ps)
        If equal Then
            Function = 1
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Sub WsDbHshDel(ByVal p As WsDbHsh Ptr, ByRef key As WString)
    'remove Key and associated Value from Hash Table
    Local i, equal As Long
    Local ps As WString Ptr : ps = StrPtr(key)
    Local prev, n As WsDbHshNode Ptr
    ExitS(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    i = WStrHash(ps, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    prev = 0
    n = @p.@arr[i]
    While n
        equal = WsEqual(@n.@K.mem, ps)
        If equal Then
            If prev Then
                @prev.next = @n.next
            Else
                @p.@arr[i] = @n.next
            End If
            @n.K = WsFinal(@n.K)
            MemFree(n)
            ExitS(@p.count=0, LibErrU)
            Decr @p.count
            Exit Loop
        End If
        prev = n
        n = @n.next
    Wend
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function WsDbHshClone(ByVal p As WsDbHsh Ptr) As Long
    'create duplicate container - return handle to cloned container
    Local i As Long
    Local n As WsDbHshNode Ptr
    Local clone As Long
    Err = 0
    ExitF(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    clone = WsDbHshNew(@p.cap) : If Err Then Exit Function
    For i = 0 To @p.cap - 1
        n = @p.@arr[i]
        While n
            WsDbHshSet clone, WsGet(@n.K), @n.V
            n = @n.next
        Wend
    Next i
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function WsDbHshStore(ByVal p As WsDbHsh Ptr) As String
    'store container to string
    Local i As Long
    Local keys, vals, store As Long
    Local n As WsDbHshNode Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    keys = WsLstNew() : If Err Then Exit Function
    vals = DbLstNew() : If Err Then Exit Function
    store = SsLstNew() : If Err Then Exit Function
    If @p.count Then
        For i = 0 To @p.cap - 1
            n = @p.@arr[i]
            While n
                WsLstAdd keys, WsGet(@n.K)
                DbLstAdd vals, @n.V
                n = @n.next
            Wend
        Next i
        SsLstAdd store, Mkl$(@p.cap)
        SsLstAdd store, WsLstStore(keys)
        SsLstAdd store, DbLstStore(vals)
        Function = SsLstStore(store)
    End If
    keys = WsLstFinal(keys)
    vals = DbLstFinal(vals)
    store = SsLstFinal(store)
End Function

Sub WsDbHshRestore(ByVal p As WsDbHsh Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, store As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    WsDbHshClear p
    keys = WsLstNew() : If Err Then Exit Sub
    vals = DbLstNew() : If Err Then Exit Sub
    store = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore store, s : If Err Then Exit Sub
        ExitS(SsLstCount(store)<>3, LibErrU)
        WsDbHshCapSet p, Cvl(SsLstPopFirst(store))
        WsLstRestore keys, SsLstPopFirst(store)
        DbLstRestore vals, SsLstPopFirst(store)
        ExitS(WsLstCount(keys)<>DbLstCount(vals), LibErrU)
        While WsLstCount(keys)
            WsDbHshSet p, WsLstPopFirst(keys), DbLstPopFirst(vals)
        Wend
    End If
    keys = WsLstFinal(keys)
    vals = DbLstFinal(vals)
    store = SsLstFinal(store)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub WsDbHshFileStore(ByVal p As WsDbHsh Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    s = WsDbHshStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub WsDbHshFileRestore(ByVal p As WsDbHsh Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>WsDbHshTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        WsDbHshRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub
