#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\List\CuLst.inc"
#Include Once "C:\HLib3\List\SsLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'Long/Currency ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = LnCuTreNew() 'get handle for new container
    '   h = LnCuTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro LnCuTreNodeTag = 1255889709
Macro LnCuTreTag = 1012397857
Type LnCuTreNode
    tag As Long
    P As LnCuTreNode Ptr
    L As LnCuTreNode Ptr
    R As LnCuTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As Currency
End Type
Type LnCuTre
    tag As Long
    count As Long
    root As LnCuTreNode Ptr
End Type

Function LnCuTreNew() As Long
    'allocate new container - return handle
    Local p As LnCuTre Ptr
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = LnCuTreTag
    Function = p
End Function

Function LnCuTreFinal(ByVal pTree As LnCuTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>LnCuTreTag, LibErrH)
        LnCuTreClear pTree
        MemFree(pTree)
    End If
End Function

Function LnCuTreValidate(ByVal pTree As LnCuTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = LnCuTreTag Then Function = @pTree.tag
End Function

Sub LnCuTreClear(ByVal pTree As LnCuTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As LnCuTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = LnCuTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = LnCuTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            LnCuTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function LnCuTreCount(ByVal pTree As LnCuTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub LnCuTreSet(ByVal pTree As LnCuTre Ptr, ByVal key As Long, ByVal value As Currency, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local n As LnCuTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    If @pTree.root Then
        n = @pTree.root
        While 1
            If key > @n.K Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = LnCuTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    @n.@R.K = key
                    @n.@R.V = value
                    LnCuTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf key < @n.K Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = LnCuTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    @n.@L.K = key
                    @n.@L.V = value
                    LnCuTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then @n.V = value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = LnCuTreAllocNode(pTree) : If Err Then Exit Sub
        @pTree.@root.K = key
        @pTree.@root.V = value
        @pTree.count = 1
    End If
End Sub

Function LnCuTreGet(ByVal pTree As LnCuTre Ptr, ByVal key As Long) As Currency
    'get Key's associated Value
    Register k As Long : k = key
    Local n As LnCuTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = @n.V
            Exit Loop
        End If
    Wend
End Function

Function LnCuTreGot(ByVal pTree As LnCuTre Ptr, ByVal key As Long) As Long
    'True/False if Key exist
    Register k As Long : k = key
    Local n As LnCuTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
End Function

Sub LnCuTreDel(ByVal pTree As LnCuTre Ptr, ByVal key As Long)
    'remove Key and associated Value
    Local pNode As LnCuTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    pNode = LnCuTreGot(pTree, key)
    If pNode Then
        LnCuTreRemoveNode(pTree, pNode)
    End If
End Sub

Function LnCuTreFirst(ByVal pTree As LnCuTre Ptr) As Long
    'get handle to first node in tree
    Local n As LnCuTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function LnCuTreLast(ByVal pTree As LnCuTre Ptr) As Long
    'get handle to last node in tree
    Local n As LnCuTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function LnCuTreNext(ByVal pNode As LnCuTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As LnCuTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnCuTreNodeTag, LibErrH)
        minR = LnCuTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = LnCuTreParentGreater(pNode)
        End If
    End If
End Function

Function LnCuTrePrev(ByVal pNode As LnCuTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As LnCuTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnCuTreNodeTag, LibErrH)
        maxL = LnCuTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = LnCuTreParentLesser(pNode)
        End If
    End If
End Function

Function LnCuTreGetKey(ByVal pNode As LnCuTreNode Ptr) As Long
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>LnCuTreNodeTag, LibErrH)
    Function = @pNode.K
End Function

Function LnCuTreGetVal(ByVal pNode As LnCuTreNode Ptr) As Currency
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>LnCuTreNodeTag, LibErrH)
    Function = @pNode.V
End Function

Sub LnCuTreSetVal(ByVal pNode As LnCuTreNode Ptr, ByVal value As Currency)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>LnCuTreNodeTag, LibErrH)
    @pNode.V = value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnCuTreClone(ByVal pTree As LnCuTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    clone = LnCuTreNew() : If Err Then Exit Function
    h = LnCuTreFirst(pTree)
    While h
        LnCuTreSet clone, LnCuTreGetKey(h), LnCuTreGetVal(h)
        h = LnCuTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnCuTreStore(ByVal pTree As LnCuTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    keys = LnLstNew() : If Err Then Exit Function
    vals = CuLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = LnCuTreFirst(pTree)
        While h
            LnLstAdd keys, LnCuTreGetKey(h)
            CuLstAdd vals, LnCuTreGetVal(h)
            h = LnCuTreNext(h)
        Wend
        SsLstAdd stor, LnLstStore(keys)
        SsLstAdd stor, CuLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = LnLstFinal(keys)
    vals = CuLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub LnCuTreRestore(ByVal pTree As LnCuTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    LnCuTreClear pTree
    keys = LnLstNew() : If Err Then Exit Sub
    vals = CuLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        LnLstRestore keys, SsLstPopFirst(stor)
        CuLstRestore vals, SsLstPopFirst(stor)
        ExitS(LnLstCount(keys)<>CuLstCount(vals), LibErrU)
        While LnLstCount(keys)
            LnCuTreSet pTree, LnLstPopFirst(keys), CuLstPopFirst(vals)
        Wend
    End If
    keys = LnLstFinal(keys)
    vals = CuLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnCuTreFileStore(ByVal pTree As LnCuTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    s = LnCuTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnCuTreFileRestore(ByVal pTree As LnCuTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnCuTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnCuTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub LnCuTreRemoveNode(ByVal p As LnCuTre Ptr, ByVal n As LnCuTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As LnCuTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, LnCuTreMaxLeft(n), LnCuTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        LnCuTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = LnCuTreFreeNode(p, n)
        LnCuTreBalanceBranch p, nP
    End If
End Sub

Function LnCuTreAllocNode(ByVal p As LnCuTre Ptr) Private As Long
    Local n As LnCuTreNode Ptr
    n = MemAlloc(SizeOf(LnCuTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = LnCuTreNodeTag
    Incr @p.count
    Function = n
End Function

Function LnCuTreFreeNode(ByVal p As LnCuTre Ptr, ByVal n As LnCuTreNode Ptr) As Long
    If n Then
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub LnCuTreBalanceBranch(ByVal p As LnCuTre Ptr, ByVal n As LnCuTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = LnCuTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = LnCuTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function LnCuTreMaxLeft(ByVal n As LnCuTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function LnCuTreMinRight(ByVal n As LnCuTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function LnCuTreParentGreater(ByVal n As LnCuTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnCuTreParentLesser(ByVal n As LnCuTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnCuTreRotateLeft(ByVal p As LnCuTre Ptr, ByVal n As LnCuTreNode Ptr) Private As Long
    Local nR, nRL As LnCuTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function LnCuTreRotateRight(ByVal p As LnCuTre Ptr, ByVal n As LnCuTreNode Ptr) Private As Long
    Local nL, nLR As LnCuTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
