#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\WsLst.inc"
#Include Once "C:\HLib3\List\CuLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'WString/Currency ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - no limit on Key length
    '   - use WsCuTreComparison() to change how Keys compared
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = WsCuTreNew() 'get handle for new container
    '   h = WsCuTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro WsCuTreNodeTag = 1317754472
Macro WsCuTreTag = -1900523422
Type WsCuTreNode
    tag As Long
    P As WsCuTreNode Ptr
    L As WsCuTreNode Ptr
    R As WsCuTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As Currency
End Type
Type WsCuTre
    tag As Long
    count As Long
    root As WsCuTreNode Ptr
    compareCB As Long
    collation As WsStr Ptr
End Type

Function WsCuTreNew() As Long
    'allocate new container - return handle
    Local p As WsCuTre Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = WsCuTreTag
    @p.compareCB = CodePtr(WsCompare)
    @p.collation = WsNew() : If Err Then Exit Function
    Function = p
End Function

Function WsCuTreFinal(ByVal pTree As WsCuTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>WsCuTreTag, LibErrH)
        @pTree.collation = WsFinal(@pTree.collation)
        WsCuTreClear pTree
        MemFree(pTree)
    End If
End Function

Function WsCuTreValidate(ByVal pTree As WsCuTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = WsCuTreTag Then Function = @pTree.tag
End Function

Sub WsCuTreComparison(ByVal pTree As WsCuTre Ptr, ByVal compareUCase As Long, ByVal collationSequence As WString)
    'set how WStrings compared
    'default = case ignored
    'if collationSequence WString provided then
    '   WStrings are compared using the order of the collation sequence WString
    '   collation WString must be 65536 characters
    'else if compareUCase = True then
    '   WStrings compared UCase
    ExitS(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    @pTree.compareCB = CodePtr(WsCompare)
    WsClear @pTree.collation
    If Len(collationSequence) Then
        ExitS(Len(collationSequence)<>65536, LibErrS)
        WsSet @pTree.collation, collationSequence : If Err Then Exit Sub
        @pTree.compareCB = CodePtr(WsCompareCollate)
    ElseIf compareUCase Then
        @pTree.compareCB = CodePtr(WsCompareUCase)
    End If
End Sub

Sub WsCuTreClear(ByVal pTree As WsCuTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As WsCuTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = WsCuTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = WsCuTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            WsCuTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function WsCuTreCount(ByVal pTree As WsCuTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub WsCuTreSet(ByVal pTree As WsCuTre Ptr, ByRef key As WString, ByVal value As Currency, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local compare, temp As Long
    Local n As WsCuTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    temp = WsSetNew(key) : If Err Then Exit Sub
    If @pTree.root Then
        n = @pTree.root
        While 1
            Call Dword @pTree.compareCB Using WsCompareCB(temp, @n.K, @pTree.collation) To compare
            If compare > 0 Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = WsCuTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    WsSet @n.@R.K, key
                    @n.@R.V = value
                    WsCuTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf compare < 0 Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = WsCuTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    WsSet @n.@L.K, key
                    @n.@L.V = value
                    WsCuTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then @n.V = value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = WsCuTreAllocNode(pTree) : If Err Then Exit Sub
        WsSet @pTree.@root.K, key
        @pTree.@root.V = value
        @pTree.count = 1
    End If
    temp = WsFinal(temp)
End Sub

Function WsCuTreGet(ByVal pTree As WsCuTre Ptr, ByRef key As WString) As Currency
    'get Key's associated Value
    Local compare, temp As Long
    Local n As WsCuTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    temp = WsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using WsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = @n.V
            Exit Loop
        End If
    Wend
    temp = WsFinal(temp)
End Function

Function WsCuTreGot(ByVal pTree As WsCuTre Ptr, ByRef key As WString) As Long
    'True/False if Key exist
    Local compare, temp As Long
    Local n As WsCuTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    temp = WsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using WsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
    temp = WsFinal(temp)
End Function

Sub WsCuTreDel(ByVal pTree As WsCuTre Ptr, ByRef key As WString)
    'remove Key and associated Value
    Local pNode As WsCuTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    pNode = WsCuTreGot(pTree, key)
    If pNode Then
        WsCuTreRemoveNode(pTree, pNode)
    End If
End Sub

Function WsCuTreFirst(ByVal pTree As WsCuTre Ptr) As Long
    'get handle to first node in tree
    Local n As WsCuTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function WsCuTreLast(ByVal pTree As WsCuTre Ptr) As Long
    'get handle to last node in tree
    Local n As WsCuTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function WsCuTreNext(ByVal pNode As WsCuTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As WsCuTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>WsCuTreNodeTag, LibErrH)
        minR = WsCuTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = WsCuTreParentGreater(pNode)
        End If
    End If
End Function

Function WsCuTrePrev(ByVal pNode As WsCuTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As WsCuTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>WsCuTreNodeTag, LibErrH)
        maxL = WsCuTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = WsCuTreParentLesser(pNode)
        End If
    End If
End Function

Function WsCuTreGetKey(ByVal pNode As WsCuTreNode Ptr) As WString
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>WsCuTreNodeTag, LibErrH)
    Function = WsGet(@pNode.K)
End Function

Function WsCuTreGetVal(ByVal pNode As WsCuTreNode Ptr) As Currency
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>WsCuTreNodeTag, LibErrH)
    Function = @pNode.V
End Function

Sub WsCuTreSetVal(ByVal pNode As WsCuTreNode Ptr, ByVal value As Currency)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>WsCuTreNodeTag, LibErrH)
    @pNode.V = value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function WsCuTreClone(ByVal pTree As WsCuTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    clone = WsCuTreNew() : If Err Then Exit Function
    h = WsCuTreFirst(pTree)
    While h
        WsCuTreSet clone, WsCuTreGetKey(h), WsCuTreGetVal(h)
        h = WsCuTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function WsCuTreStore(ByVal pTree As WsCuTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    keys = WsLstNew() : If Err Then Exit Function
    vals = CuLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = WsCuTreFirst(pTree)
        While h
            WsLstAdd keys, WsCuTreGetKey(h)
            CuLstAdd vals, WsCuTreGetVal(h)
            h = WsCuTreNext(h)
        Wend
        SsLstAdd stor, WsLstStore(keys)
        SsLstAdd stor, CuLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = WsLstFinal(keys)
    vals = CuLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub WsCuTreRestore(ByVal pTree As WsCuTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    WsCuTreClear pTree
    keys = WsLstNew() : If Err Then Exit Sub
    vals = CuLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        WsLstRestore keys, SsLstPopFirst(stor)
        CuLstRestore vals, SsLstPopFirst(stor)
        ExitS(WsLstCount(keys)<>CuLstCount(vals), LibErrU)
        While WsLstCount(keys)
            WsCuTreSet pTree, WsLstPopFirst(keys), CuLstPopFirst(vals)
        Wend
    End If
    keys = WsLstFinal(keys)
    vals = CuLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub WsCuTreFileStore(ByVal pTree As WsCuTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    s = WsCuTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub WsCuTreFileRestore(ByVal pTree As WsCuTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>WsCuTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        WsCuTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub WsCuTreRemoveNode(ByVal p As WsCuTre Ptr, ByVal n As WsCuTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As WsCuTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, WsCuTreMaxLeft(n), WsCuTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        WsCuTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = WsCuTreFreeNode(p, n)
        WsCuTreBalanceBranch p, nP
    End If
End Sub

Function WsCuTreAllocNode(ByVal p As WsCuTre Ptr) Private As Long
    Local n As WsCuTreNode Ptr
    n = MemAlloc(SizeOf(WsCuTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = WsCuTreNodeTag
    @n.K = WsNew() : If Err Then Exit Function
    Incr @p.count
    Function = n
End Function

Function WsCuTreFreeNode(ByVal p As WsCuTre Ptr, ByVal n As WsCuTreNode Ptr) As Long
    If n Then
        @n.K = WsFinal(@n.K)
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub WsCuTreBalanceBranch(ByVal p As WsCuTre Ptr, ByVal n As WsCuTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = WsCuTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = WsCuTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function WsCuTreMaxLeft(ByVal n As WsCuTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function WsCuTreMinRight(ByVal n As WsCuTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function WsCuTreParentGreater(ByVal n As WsCuTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function WsCuTreParentLesser(ByVal n As WsCuTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function WsCuTreRotateLeft(ByVal p As WsCuTre Ptr, ByVal n As WsCuTreNode Ptr) Private As Long
    Local nR, nRL As WsCuTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function WsCuTreRotateRight(ByVal p As WsCuTre Ptr, ByVal n As WsCuTreNode Ptr) Private As Long
    Local nL, nLR As WsCuTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
