#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\WsLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'WString/WString ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - no limit on Key length
    '   - use WsWsTreComparison() to change how Keys compared
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = WsWsTreNew() 'get handle for new container
    '   h = WsWsTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro WsWsTreNodeTag = -1851175339
Macro WsWsTreTag = 1734344665
Type WsWsTreNode
    tag As Long
    P As WsWsTreNode Ptr
    L As WsWsTreNode Ptr
    R As WsWsTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As Long
End Type
Type WsWsTre
    tag As Long
    count As Long
    root As WsWsTreNode Ptr
    compareCB As Long
    collation As WsStr Ptr
End Type

Function WsWsTreNew() As Long
    'allocate new container - return handle
    Local p As WsWsTre Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = WsWsTreTag
    @p.compareCB = CodePtr(WsCompare)
    @p.collation = WsNew() : If Err Then Exit Function
    Function = p
End Function

Function WsWsTreFinal(ByVal pTree As WsWsTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>WsWsTreTag, LibErrH)
        @pTree.collation = WsFinal(@pTree.collation)
        WsWsTreClear pTree
        MemFree(pTree)
    End If
End Function

Function WsWsTreValidate(ByVal pTree As WsWsTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = WsWsTreTag Then Function = @pTree.tag
End Function

Sub WsWsTreComparison(ByVal pTree As WsWsTre Ptr, ByVal compareUCase As Long, ByVal collationSequence As WString)
    'set how WStrings compared
    'default = case ignored
    'if collationSequence WString provided then
    '   WStrings are compared using the order of the collation sequence WString
    '   collation WString must be 65536 characters
    'else if compareUCase = True then
    '   WStrings compared UCase
    ExitS(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    @pTree.compareCB = CodePtr(WsCompare)
    WsClear @pTree.collation
    If Len(collationSequence) Then
        ExitS(Len(collationSequence)<>65536, LibErrS)
        WsSet @pTree.collation, collationSequence : If Err Then Exit Sub
        @pTree.compareCB = CodePtr(WsCompareCollate)
    ElseIf compareUCase Then
        @pTree.compareCB = CodePtr(WsCompareUCase)
    End If
End Sub

Sub WsWsTreClear(ByVal pTree As WsWsTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As WsWsTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = WsWsTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = WsWsTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            WsWsTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function WsWsTreCount(ByVal pTree As WsWsTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub WsWsTreSet(ByVal pTree As WsWsTre Ptr, ByRef key As WString, ByRef value As WString, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local compare, temp As Long
    Local n As WsWsTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    temp = WsSetNew(key) : If Err Then Exit Sub
    If @pTree.root Then
        n = @pTree.root
        While 1
            Call Dword @pTree.compareCB Using WsCompareCB(temp, @n.K, @pTree.collation) To compare
            If compare > 0 Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = WsWsTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    WsSet @n.@R.K, key
                    WsSet @n.@R.V, value
                    WsWsTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf compare < 0 Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = WsWsTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    WsSet @n.@L.K, key
                    WsSet @n.@L.V, value
                    WsWsTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then WsSet @n.V, value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = WsWsTreAllocNode(pTree) : If Err Then Exit Sub
        WsSet @pTree.@root.K, key
        WsSet @pTree.@root.V, value
        @pTree.count = 1
    End If
    temp = WsFinal(temp)
End Sub

Function WsWsTreGet(ByVal pTree As WsWsTre Ptr, ByRef key As WString) As WString
    'get Key's associated Value
    Local compare, temp As Long
    Local n As WsWsTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    temp = WsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using WsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = WsGet(@n.V)
            Exit Loop
        End If
    Wend
    temp = WsFinal(temp)
End Function

Function WsWsTreGot(ByVal pTree As WsWsTre Ptr, ByRef key As WString) As Long
    'True/False if Key exist
    Local compare, temp As Long
    Local n As WsWsTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    temp = WsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using WsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
    temp = WsFinal(temp)
End Function

Sub WsWsTreDel(ByVal pTree As WsWsTre Ptr, ByRef key As WString)
    'remove Key and associated Value
    Local pNode As WsWsTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    pNode = WsWsTreGot(pTree, key)
    If pNode Then
        WsWsTreRemoveNode(pTree, pNode)
    End If
End Sub

Function WsWsTreFirst(ByVal pTree As WsWsTre Ptr) As Long
    'get handle to first node in tree
    Local n As WsWsTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function WsWsTreLast(ByVal pTree As WsWsTre Ptr) As Long
    'get handle to last node in tree
    Local n As WsWsTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function WsWsTreNext(ByVal pNode As WsWsTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As WsWsTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>WsWsTreNodeTag, LibErrH)
        minR = WsWsTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = WsWsTreParentGreater(pNode)
        End If
    End If
End Function

Function WsWsTrePrev(ByVal pNode As WsWsTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As WsWsTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>WsWsTreNodeTag, LibErrH)
        maxL = WsWsTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = WsWsTreParentLesser(pNode)
        End If
    End If
End Function

Function WsWsTreGetKey(ByVal pNode As WsWsTreNode Ptr) As WString
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>WsWsTreNodeTag, LibErrH)
    Function = WsGet(@pNode.K)
End Function

Function WsWsTreGetVal(ByVal pNode As WsWsTreNode Ptr) As WString
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>WsWsTreNodeTag, LibErrH)
    Function = WsGet(@pNode.V)
End Function

Sub WsWsTreSetVal(ByVal pNode As WsWsTreNode Ptr, ByRef value As WString)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>WsWsTreNodeTag, LibErrH)
    WsSet @pNode.V, value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function WsWsTreClone(ByVal pTree As WsWsTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    clone = WsWsTreNew() : If Err Then Exit Function
    h = WsWsTreFirst(pTree)
    While h
        WsWsTreSet clone, WsWsTreGetKey(h), WsWsTreGetVal(h)
        h = WsWsTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function WsWsTreStore(ByVal pTree As WsWsTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    keys = WsLstNew() : If Err Then Exit Function
    vals = WsLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = WsWsTreFirst(pTree)
        While h
            WsLstAdd keys, WsWsTreGetKey(h)
            WsLstAdd vals, WsWsTreGetVal(h)
            h = WsWsTreNext(h)
        Wend
        SsLstAdd stor, WsLstStore(keys)
        SsLstAdd stor, WsLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = WsLstFinal(keys)
    vals = WsLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub WsWsTreRestore(ByVal pTree As WsWsTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    WsWsTreClear pTree
    keys = WsLstNew() : If Err Then Exit Sub
    vals = WsLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        WsLstRestore keys, SsLstPopFirst(stor)
        WsLstRestore vals, SsLstPopFirst(stor)
        ExitS(WsLstCount(keys)<>WsLstCount(vals), LibErrU)
        While WsLstCount(keys)
            WsWsTreSet pTree, WsLstPopFirst(keys), WsLstPopFirst(vals)
        Wend
    End If
    keys = WsLstFinal(keys)
    vals = WsLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub WsWsTreFileStore(ByVal pTree As WsWsTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    s = WsWsTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub WsWsTreFileRestore(ByVal pTree As WsWsTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>WsWsTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        WsWsTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub WsWsTreRemoveNode(ByVal p As WsWsTre Ptr, ByVal n As WsWsTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As WsWsTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, WsWsTreMaxLeft(n), WsWsTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        WsWsTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = WsWsTreFreeNode(p, n)
        WsWsTreBalanceBranch p, nP
    End If
End Sub

Function WsWsTreAllocNode(ByVal p As WsWsTre Ptr) Private As Long
    Local n As WsWsTreNode Ptr
    n = MemAlloc(SizeOf(WsWsTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = WsWsTreNodeTag
    @n.K = WsNew() : If Err Then Exit Function
    @n.V = WsNew() : If Err Then Exit Function
    Incr @p.count
    Function = n
End Function

Function WsWsTreFreeNode(ByVal p As WsWsTre Ptr, ByVal n As WsWsTreNode Ptr) As Long
    If n Then
        @n.K = WsFinal(@n.K)
        @n.V = WsFinal(@n.V)
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub WsWsTreBalanceBranch(ByVal p As WsWsTre Ptr, ByVal n As WsWsTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = WsWsTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = WsWsTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function WsWsTreMaxLeft(ByVal n As WsWsTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function WsWsTreMinRight(ByVal n As WsWsTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function WsWsTreParentGreater(ByVal n As WsWsTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function WsWsTreParentLesser(ByVal n As WsWsTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function WsWsTreRotateLeft(ByVal p As WsWsTre Ptr, ByVal n As WsWsTreNode Ptr) Private As Long
    Local nR, nRL As WsWsTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function WsWsTreRotateRight(ByVal p As WsWsTre Ptr, ByVal n As WsWsTreNode Ptr) Private As Long
    Local nL, nLR As WsWsTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
