#Include This Once
#Include Once "C:\HLib3\Array\LnArr.inc"
#Include Once "C:\HLib3\Array\SsArr.inc"

Macro OjArrTag = 1734344665
Type OjArr
    tag As Long
    arr As LnArr Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Object Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = OjArrNew() 'get handle for new container
    '       h = OjArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

'++
Declare Function ObjArrCompareCB(a As IUnknown, b As IUnknown) As Long
    'callback function to handle comparing two any two objects stored in container
    '   a < b : return < 0
    '   a = b : return = 0
    '   a > b : return > 0

Declare Function OjArrStoreCB(o As IUnknown) As String
    'callback function must pack the object into a string and return the string

Declare Function OjArrRestoreCB(ByVal s As String) As IUnknown
    'callback function must restore the object from the string and return the object
'--

Function OjArrNew() As Long
    'allocate new container - return handle
    Local p As OjArr Ptr
    Err = 0
    p = MemAlloc(SizeOf(OjArr))
    ExitF(p=0, LibErrM)
    @p.tag = OjArrTag
    @p.arr = LnArrNew() : If Err Then Exit Function
    Function = p
End Function

Function OjArrFinal(ByVal p As OjArr Ptr) As Long
    'free allocated container - return null
    Local i As Long
    If p Then
        ExitF(@p.tag<>OjArrTag, LibErrH)
        OjArrClear p
        @p.arr = LnArrFinal(@p.arr)
        MemFree(p)
    End If
End Function

Function OjArrValidate(ByVal p As OjArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = OjArrTag Then Function = @p.tag
End Function

Sub OjArrClear(ByVal p As OjArr Ptr)
    'delete all data
    Local i As Long
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    For i = 1 To @p.@arr.count
        ObjMacFree(@p.@arr.@arr[i])
    Next i
    LnArrClear @p.arr
End Sub

Function OjArrCount(ByVal p As OjArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    Function = @p.@arr.count
End Function

Sub OjArrReDim(ByVal p As OjArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local i, items As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        OjArrClear p
    ElseIf Count <> @p.@arr.count Then
        items = @p.@arr.count
        For i = Count + 1 To items
            ObjMacFree(@p.@arr.@arr[i])
        Next i
        LnArrReDim @p.arr, Count : If Err Then Exit Sub
    End If
End Sub

Function OjArrGet(ByVal p As OjArr Ptr, ByVal index As Long) As IUnknown
    'get value at index - one-based index
    Register i As Long : i = index
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitF(i<1 Or i>@p.@arr.count, LibErrB)
    ExitF(@p.@arr.@arr[i]=0, LibErrN)
    Poke Long, VarPtr(o), @p.@arr.@arr[i]
    o.AddRef()
    Function = o
End Function

Sub OjArrSet(ByVal p As OjArr Ptr, ByVal index As Long, value As IUnknown)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    ExitS(IsNothing(value), LibErrO)
    ObjMacSet(@p.@arr.@arr[i], value)
End Sub

Sub OjArrAdd(ByVal p As OjArr Ptr, value As IUnknown)
    'append value to end of array - ReDim automatic
    Local h As Long
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(IsNothing(value), LibErrO)
    ObjMacSet(h, value)
    LnArrAdd @p.arr, h
End Sub

Sub OjArrIns(ByVal p As OjArr Ptr, ByVal index As Long, value As IUnknown)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Local h As Long
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    ExitS(IsNothing(value), LibErrO)
    ObjMacSet(h, value)
    LnArrIns @p.arr, i, h
End Sub

Sub OjArrDel(ByVal p As OjArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(i<1 Or i>@p.@arr.count, LibErrB)
    ObjMacFree(@p.@arr.@arr[i])
    LnArrDel @p.arr, i
End Sub

Sub OjArrSort(ByVal p As OjArr Ptr, ByVal compareCallback As Long)
    'sort array - fast non-recursive Quick Sort
    'all stored objects must be same type
    'compareCallback =  CodePtr(callback function to handle comparing stored objects)
    '   Function ObjArrCompareCB(a As IUnknown, b As IUnknown) As Long
    '       a < b : return < 0
    '       a = b : return = 0
    '       a > b : return > 0
    Local i, j, k, leftIndex, rightIndex, counter, compare As Long
    Local arr As Long Ptr
    Local a, b As IUnknown
    Local pa, pb As Long Ptr : pa = VarPtr(a) : pb = VarPtr(b)
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(compareCallback=0, LibErrC)
    If @p.@arr.count > 1 Then
        arr = @p.@arr.arr
        leftIndex = 1
        rightIndex = @p.@arr.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            ExitS(@arr[k]=0, LibErrO)
            @pa = @arr[k]
            While i <= j
                ExitS(@arr[i]=0, LibErrO)
                @pb = @arr[i]
                Call Dword compareCallback Using ObjArrCompareCB(b, a) To compare
                @pb = 0
                While compare < 0
                    Incr i
                    ExitS(@arr[i]=0, LibErrO)
                    @pb = @arr[i]
                    Call Dword compareCallback Using ObjArrCompareCB(b, a) To compare
                    @pb = 0
                Wend
                ExitS(@arr[j]=0, LibErrO)
                @pb = @arr[j]
                Call Dword compareCallback Using ObjArrCompareCB(b, a) To compare
                @pb = 0
                While compare > 0
                    Decr j
                    ExitS(@arr[j]=0, LibErrO)
                    @pb = @arr[j]
                    Call Dword compareCallback Using ObjArrCompareCB(b, a) To compare
                    @pb = 0
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            @pa = 0
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub OjArrUniqueSort(ByVal p As OjArr Ptr, ByVal compareCallback As Long)
    'sort array and delete all duplicates
    'all stored objects must be same type
    'compareCallback =  CodePtr(callback function to handle comparing stored objects)
    '   Function ObjArrCompareCB(a As IUnknown, b As IUnknown) As Long
    '       a < b : return < 0
    '       a = b : return = 0
    '       a > b : return > 0
    Local i, compare As Long
    Local arr As Long Ptr
    Local a, b As IUnknown
    Local pa, pb As Long Ptr : pa = VarPtr(a) : pb = VarPtr(b)
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(compareCallback=0, LibErrC)
    OjArrSort p, compareCallback : If Err Then Exit Sub
    arr = @p.@arr.arr
    For i = @p.@arr.count - 1 To 1 Step -1
        @pa = @arr[i + 1] : @pb = @arr[i]
        Call Dword compareCallback Using ObjArrCompareCB(a, b) To compare
        @pa = 0 : @pb = 0
        If compare = 0 Then OjArrDel p, i + 1
    Next i
End Sub

Sub OjArrReverse(ByVal p As OjArr Ptr)
    'reverse array
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    LnArrReverse @p.arr
End Sub

Function OjArrBinSearch(ByVal p As OjArr Ptr, value As IUnknown, ByVal compareCallback As Long) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    'compareCallback =  CodePtr(callback function to handle comparing stored objects)
    '   Function ObjArrCompareCB(a As IUnknown, b As IUnknown) As Long
    '       a < b : return < 0
    '       a = b : return = 0
    '       a > b : return > 0
    Local i, top, bot, compare As Long
    Local arr As Long Ptr
    Local b As IUnknown
    Local pa, pb As Long Ptr
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitF(IsNothing(value), LibErrO)
    bot = 1
    top = @p.@arr.count
    arr = @p.@arr.arr
    pa = VarPtr(value) : pb = VarPtr(b)
    While top >= bot
        i = bot + top
        Shift Right i, 1
        ExitF(@arr[i]=0, LibErrO)
        @pa = @arr[i]
        Call Dword compareCallback Using ObjArrCompareCB(value, b) To compare
        @pa = 0
        If compare > 0 Then
            bot = i + 1
        ElseIf compare < 0 Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub OjArrBinInsert(ByVal p As OjArr Ptr, value As IUnknown, ByVal compareCallback As Long)
    'binary insert value into array
    'ok if array empty
    'array must be sorted for valid results
    'compareCallback =  CodePtr(callback function to handle comparing stored objects)
    '   Function ObjArrCompareCB(a As IUnknown, b As IUnknown) As Long
    '       a < b : return < 0
    '       a = b : return = 0
    '       a > b : return > 0
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(IsNothing(value), LibErrO)
    ExitS(compareCallback=0, LibErrC)
    i = OjArrBinPos(p, value, compare, compareCallback)
    If i = 0 Then
        OjArrAdd p, value
    ElseIf compare <= 0 Then
        OjArrIns p, i, value
    ElseIf i < @p.@arr.count Then
        OjArrIns p, i + 1, value
    Else
        OjArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Object Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub OjArrStkPush(ByVal p As OjArr Ptr, value As IUnknown)
    'push value on top of Stack
    OjArrAdd p, value
End Sub

Function OjArrStkPeek(ByVal p As OjArr Ptr) As IUnknown
    'get top value on Stack
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ObjMacGet(@p.@arr.@arr[@p.@arr.count], o)
        Function = o
    End If
End Function

Function OjArrStkPop(ByVal p As OjArr Ptr) As IUnknown
    'get and remove top value on Stack
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ObjMacGet(@p.@arr.@arr[@p.@arr.count], o)
        Function = o
        OjArrDel p, @p.@arr.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   String Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub OjArrQuePush(ByVal p As OjArr Ptr, value As IUnknown)
    'add value to end of Queue
    OjArrAdd p, value
End Sub

Function OjArrQuePeek(ByVal p As OjArr Ptr) As IUnknown
    'get first value in Queue
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ObjMacGet(@p.@arr.@arr[1], o)
        Function = o
    End If
End Function

Function OjArrQuePop(ByVal p As OjArr Ptr) As IUnknown
    'get and remove first value in Queue
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ObjMacGet(@p.@arr.@arr[1], o)
        Function = o
        OjArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   String Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub OjArrPushFirst(ByVal p As OjArr Ptr, value As IUnknown)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then OjArrIns p, 1, value Else OjArrAdd p, value
End Sub

Sub OjArrPushLast(ByVal p As OjArr Ptr, value As IUnknown)
    'add value to and of Deque (double-ended Queue)
    OjArrAdd p, value
End Sub

Function OjArrPeekFirst(ByVal p As OjArr Ptr) As IUnknown
    'get first value in Deque (double-ended Queue)
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ObjMacGet(@p.@arr.@arr[1], o)
        Function = o
    End If
End Function

Function OjArrPeekLast(ByVal p As OjArr Ptr) As IUnknown
    'get last value in Deque (double-ended Queue)
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ObjMacGet(@p.@arr.@arr[@p.@arr.count], o)
        Function = o
    End If
End Function

Function OjArrPopFirst(ByVal p As OjArr Ptr) As IUnknown
    'get and remove first value in Deque (double-ended Queue)
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ObjMacGet(@p.@arr.@arr[1], o)
        Function = o
        OjArrDel p, 1
    End If
End Function

Function OjArrPopLast(ByVal p As OjArr Ptr) As IUnknown
    'get and remove last value in Deque (double-ended Queue)
    Local o As IUnknown
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    If @p.@arr.count Then
        ObjMacGet(@p.@arr.@arr[@p.@arr.count], o)
        Function = o
        OjArrDel p, @p.@arr.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function OjArrClone(ByVal p As OjArr Ptr) As Long
    'duplicate container - return clone's handle
    Local i As Long
    Local clone As OjArr Ptr
    Local o As IUnknown
    Err = 0
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    clone = OjArrNew()
    If @p.@arr.count Then
        OjArrReDim clone, @p.@arr.count : If Err Then Exit Function
        For i = 1 To @p.@arr.count
            o = OjArrGet(p, i)
            If Err Then Exit Function
            OjArrSet clone, i, o
            o = Nothing
        Next i
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function OjArrStore(ByVal p As OjArr Ptr, ByVal storeCallback As Long) As String
    'store container to string
    'storeCallback = CodePtr( function to convert each object into a string )
    '   Function OjArrStoreCB(o As IUnknown) As String
    Local i As Long
    Local str As Long
    Local o As IUnknown
    Local s As String
    Err = 0
    ExitF(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitF(storeCallback=0, LibErrC)
    str = SsArrNew()
    If @p.@arr.count Then
        SsArrReDim str, @p.@arr.count : If Err Then Exit Function
        For i = 1 To @p.@arr.count
            o = OjArrGet(p, i) : If Err Then Exit Function
            Call Dword storeCallback Using OjArrStoreCB(o) To s
            SsArrSet str, i, s
        Next i
        Function = SsArrStore(str)
    End If
    str = SsArrFinal(str)
End Function

Sub OjArrRestore(ByVal p As OjArr Ptr, ByVal s As String, ByVal restoreCallback As Long)
    'restore container from string
    'restoreCallback = CodePtr( callback function to convert string back into object )
    '   Function OjArrRestoreCB(ByVal s As String) As IUnknown
    Local i, Count As Long
    Local str As Long
    Local o As IUnknown
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    ExitS(restoreCallback=0, LibErrC)
    OjArrClear p
    If Len(s) Then
        SsArrRestore str, s : If Err Then Exit Sub
        Count = SsArrCount(str)
        If Count Then
            OjArrReDim p, Count : If Err Then Exit Sub
            For i = 1 To Count
                Call Dword restoreCallback Using OjArrRestoreCB(SsArrGet(str, i)) To o
                OjArrSet p, i, o : If Err Then Exit Sub
                o = Nothing
            Next i
        End If
    End If
    str = SsArrFinal(str)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub OjArrFileStore(ByVal p As OjArr Ptr, ByVal file As String, ByVal storeCallback As Long)
    'store container to file
    'storeCallback = CodePtr( function to convert each object into a string )
    '   Function OjArrStoreCB(o As IUnknown) As String
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    s = OjArrStore(p, storeCallback) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub OjArrFileRestore(ByVal p As OjArr Ptr, ByVal file As String, ByVal restoreCallback As Long)
    'restore container from file
    'restoreCallback = CodePtr( callback function to convert string back into object )
    '   Function OjArrRestoreCB(ByVal s As String) As IUnknown
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>OjArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        OjArrRestore p, s, restoreCallback
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function OjArrBinPos(ByVal p As OjArr Ptr, value As IUnknown, ByRef compare As Long, ByVal compareCallback As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Long Ptr
    Local b As IUnknown
    Local pa, pb As Long Ptr
    compare = -1
    bot = 1
    top = @p.@arr.count
    arr = @p.@arr.arr
    pa = VarPtr(value) : pb = VarPtr(b)
    While top >= bot
        i = bot + top
        Shift Right i, 1
        ExitF(@arr[i]=0, LibErrO)
        @pa = @arr[i]
        Call Dword compareCallback Using ObjArrCompareCB(value, b) To compare
        @pa = 0
        If compare > 0 Then
            bot = i + 1
        ElseIf compare < 0 Then
            top = i - 1
        Else
            Exit Loop
        End If
    Wend
    Function = i
End Function
