#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro CxArrTag = 1834559302
Type CxArr
    tag As Long
    count As Long
    arr As CurrencyX Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   CurrencyX Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = CxArrNew() 'get handle for new container
    '       h = CxArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function CxArrNew() As Long
    'allocate new container - return handle
    Local p As CxArr Ptr
    p = MemAlloc(SizeOf(CxArr))
    ExitF(p=0, LibErrM)
    @p.tag = CxArrTag
    Function = p
End Function

Function CxArrFinal(ByVal p As CxArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>CxArrTag, LibErrH)
        CxArrClear p
        MemFree(p)
    End If
End Function

Function CxArrValidate(ByVal p As CxArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = CxArrTag Then Function = @p.tag
End Function

Sub CxArrClear(ByVal p As CxArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeCx
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function CxArrCount(ByVal p As CxArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    Function = @p.count
End Function

Sub CxArrReDim(ByVal p As CxArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        CxArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeCx
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeCx)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeCx
    End If
End Sub

Function CxArrGet(ByVal p As CxArr Ptr, ByVal index As Long) As CurrencyX
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub CxArrSet(ByVal p As CxArr Ptr, ByVal index As Long, ByVal value As CurrencyX)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub CxArrAdd(ByVal p As CxArr Ptr, ByVal value As CurrencyX)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    CxArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub CxArrIns(ByVal p As CxArr Ptr, ByVal index As Long, ByVal value As CurrencyX)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    CxArrReDim p, @p.count + 1 : If Err Then Exit Sub
    CxArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub CxArrDel(ByVal p As CxArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then CxArrMove @p.arr, i + 1, i, @p.count - i
    CxArrReDim p, @p.count - 1
End Sub

Sub CxArrSort(ByVal p As CxArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As CurrencyX
    Local arr As CurrencyX Ptr
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub CxArrUniqueSort(ByVal p As CxArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As CurrencyX Ptr
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    CxArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then CxArrDel p, i + 1
    Next i
End Sub

Sub CxArrReverse(ByVal p As CxArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As CurrencyX Ptr
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function CxArrBinSearch(ByVal p As CxArr Ptr, ByVal value As CurrencyX) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As CurrencyX Ptr
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub CxArrBinInsert(ByVal p As CxArr Ptr, ByVal value As CurrencyX)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    i = CxArrBinPos(p, value, compare)
    If i = 0 Then
        CxArrAdd p, value
    ElseIf compare <= 0 Then
        CxArrIns p, i, value
    ElseIf i < @p.count Then
        CxArrIns p, i + 1, value
    Else
        CxArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   CurrencyX Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub CxArrStkPush(ByVal p As CxArr Ptr, ByVal value As CurrencyX)
    'push value on top of Stack
    CxArrAdd p, value
End Sub

Function CxArrStkPeek(ByVal p As CxArr Ptr) As CurrencyX
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function CxArrStkPop(ByVal p As CxArr Ptr) As CurrencyX
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        CxArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   CurrencyX Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub CxArrQuePush(ByVal p As CxArr Ptr, ByVal value As CurrencyX)
    'add value to end of Queue
    CxArrAdd p, value
End Sub

Function CxArrQuePeek(ByVal p As CxArr Ptr) As CurrencyX
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function CxArrQuePop(ByVal p As CxArr Ptr) As CurrencyX
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        CxArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   CurrencyX Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub CxArrPushFirst(ByVal p As CxArr Ptr, ByVal value As CurrencyX)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then CxArrIns p, 1, value Else CxArrAdd p, value
End Sub

Sub CxArrPushLast(ByVal p As CxArr Ptr, ByVal value As CurrencyX)
    'add value to and of Deque (double-ended Queue)
    CxArrAdd p, value
End Sub

Function CxArrPeekFirst(ByVal p As CxArr Ptr) As CurrencyX
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function CxArrPeekLast(ByVal p As CxArr Ptr) As CurrencyX
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function CxArrPopFirst(ByVal p As CxArr Ptr) As CurrencyX
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        CxArrDel p, 1
    End If
End Function

Function CxArrPopLast(ByVal p As CxArr Ptr) As CurrencyX
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        CxArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function CxArrClone(ByVal p As CxArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As CxArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    clone = CxArrNew() : If Err Then Exit Function
    If @p.count Then
        CxArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeCx, @clone.arr + %SizeCx, @p.count * %SizeCx
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function CxArrStore(ByVal p As CxArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>CxArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeCx, @p.count * %SizeCx)
End Function

Sub CxArrRestore(ByVal p As CxArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    CxArrClear p
    If strLen Then
        CxArrReDim p, strLen / %SizeCx : If Err Then Exit Sub
        Poke$ @p.arr + %SizeCx, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub CxArrFileStore(ByVal p As CxArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    s = CxArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub CxArrFileRestore(ByVal p As CxArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>CxArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        CxArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub CxArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeCx), arr + (toIndex * %SizeCx),  Count * %SizeCx
End Sub

Function CxArrBinPos(ByVal p As CxArr Ptr, ByVal value As CurrencyX, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As CurrencyX Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function
