#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro DwArrTag = -33103557
Type DwArr
    tag As Long
    count As Long
    arr As Dword Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Dword Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = DwArrNew() 'get handle for new container
    '       h = DwArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function DwArrNew() As Long
    'allocate new container - return handle
    Local p As DwArr Ptr
    p = MemAlloc(SizeOf(DwArr))
    ExitF(p=0, LibErrM)
    @p.tag = DwArrTag
    Function = p
End Function

Function DwArrFinal(ByVal p As DwArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>DwArrTag, LibErrH)
        DwArrClear p
        MemFree(p)
    End If
End Function

Function DwArrValidate(ByVal p As DwArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = DwArrTag Then Function = @p.tag
End Function

Sub DwArrClear(ByVal p As DwArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeDw
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function DwArrCount(ByVal p As DwArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    Function = @p.count
End Function

Sub DwArrReDim(ByVal p As DwArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        DwArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeDw
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeDw)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeDw
    End If
End Sub

Function DwArrGet(ByVal p As DwArr Ptr, ByVal index As Long) As Dword
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub DwArrSet(ByVal p As DwArr Ptr, ByVal index As Long, ByVal value As Dword)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub DwArrAdd(ByVal p As DwArr Ptr, ByVal value As Dword)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    DwArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub DwArrIns(ByVal p As DwArr Ptr, ByVal index As Long, ByVal value As Dword)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    DwArrReDim p, @p.count + 1 : If Err Then Exit Sub
    DwArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub DwArrDel(ByVal p As DwArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then DwArrMove @p.arr, i + 1, i, @p.count - i
    DwArrReDim p, @p.count - 1
End Sub

Sub DwArrSort(ByVal p As DwArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As Dword
    Local arr As Dword Ptr
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub DwArrUniqueSort(ByVal p As DwArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As Dword Ptr
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    DwArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then DwArrDel p, i + 1
    Next i
End Sub

Sub DwArrReverse(ByVal p As DwArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As Dword Ptr
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function DwArrBinSearch(ByVal p As DwArr Ptr, ByVal value As Dword) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As Dword Ptr
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub DwArrBinInsert(ByVal p As DwArr Ptr, ByVal value As Dword)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    i = DwArrBinPos(p, value, compare)
    If i = 0 Then
        DwArrAdd p, value
    ElseIf compare <= 0 Then
        DwArrIns p, i, value
    ElseIf i < @p.count Then
        DwArrIns p, i + 1, value
    Else
        DwArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Dword Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub DwArrStkPush(ByVal p As DwArr Ptr, ByVal value As Dword)
    'push value on top of Stack
    DwArrAdd p, value
End Sub

Function DwArrStkPeek(ByVal p As DwArr Ptr) As Dword
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function DwArrStkPop(ByVal p As DwArr Ptr) As Dword
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        DwArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Dword Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub DwArrQuePush(ByVal p As DwArr Ptr, ByVal value As Dword)
    'add value to end of Queue
    DwArrAdd p, value
End Sub

Function DwArrQuePeek(ByVal p As DwArr Ptr) As Dword
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function DwArrQuePop(ByVal p As DwArr Ptr) As Dword
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        DwArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Dword Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub DwArrPushFirst(ByVal p As DwArr Ptr, ByVal value As Dword)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then DwArrIns p, 1, value Else DwArrAdd p, value
End Sub

Sub DwArrPushLast(ByVal p As DwArr Ptr, ByVal value As Dword)
    'add value to and of Deque (double-ended Queue)
    DwArrAdd p, value
End Sub

Function DwArrPeekFirst(ByVal p As DwArr Ptr) As Dword
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function DwArrPeekLast(ByVal p As DwArr Ptr) As Dword
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function DwArrPopFirst(ByVal p As DwArr Ptr) As Dword
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        DwArrDel p, 1
    End If
End Function

Function DwArrPopLast(ByVal p As DwArr Ptr) As Dword
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        DwArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function DwArrClone(ByVal p As DwArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As DwArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    clone = DwArrNew() : If Err Then Exit Function
    If @p.count Then
        DwArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeDw, @clone.arr + %SizeDw, @p.count * %SizeDw
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function DwArrStore(ByVal p As DwArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>DwArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeDw, @p.count * %SizeDw)
End Function

Sub DwArrRestore(ByVal p As DwArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    DwArrClear p
    If strLen Then
        DwArrReDim p, strLen / %SizeDw : If Err Then Exit Sub
        Poke$ @p.arr + %SizeDw, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub DwArrFileStore(ByVal p As DwArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    s = DwArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub DwArrFileRestore(ByVal p As DwArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>DwArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        DwArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub DwArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeDw), arr + (toIndex * %SizeDw),  Count * %SizeDw
End Sub

Function DwArrBinPos(ByVal p As DwArr Ptr, ByVal value As Dword, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Dword Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function
