#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro ExArrTag = 1012397857
Type ExArr
    tag As Long
    count As Long
    arr As Extended Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Extended Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = ExArrNew() 'get handle for new container
    '       h = ExArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function ExArrNew() As Long
    'allocate new container - return handle
    Local p As ExArr Ptr
    p = MemAlloc(SizeOf(ExArr))
    ExitF(p=0, LibErrM)
    @p.tag = ExArrTag
    Function = p
End Function

Function ExArrFinal(ByVal p As ExArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>ExArrTag, LibErrH)
        ExArrClear p
        MemFree(p)
    End If
End Function

Function ExArrValidate(ByVal p As ExArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = ExArrTag Then Function = @p.tag
End Function

Sub ExArrClear(ByVal p As ExArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeEx
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function ExArrCount(ByVal p As ExArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    Function = @p.count
End Function

Sub ExArrReDim(ByVal p As ExArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        ExArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeEx
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeEx)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeEx
    End If
End Sub

Function ExArrGet(ByVal p As ExArr Ptr, ByVal index As Long) As Extended
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub ExArrSet(ByVal p As ExArr Ptr, ByVal index As Long, ByVal value As Extended)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub ExArrAdd(ByVal p As ExArr Ptr, ByVal value As Extended)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    ExArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub ExArrIns(ByVal p As ExArr Ptr, ByVal index As Long, ByVal value As Extended)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    ExArrReDim p, @p.count + 1 : If Err Then Exit Sub
    ExArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub ExArrDel(ByVal p As ExArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then ExArrMove @p.arr, i + 1, i, @p.count - i
    ExArrReDim p, @p.count - 1
End Sub

Sub ExArrSort(ByVal p As ExArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As Extended
    Local arr As Extended Ptr
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub ExArrUniqueSort(ByVal p As ExArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As Extended Ptr
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    ExArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then ExArrDel p, i + 1
    Next i
End Sub

Sub ExArrReverse(ByVal p As ExArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As Extended Ptr
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function ExArrBinSearch(ByVal p As ExArr Ptr, ByVal value As Extended) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As Extended Ptr
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub ExArrBinInsert(ByVal p As ExArr Ptr, ByVal value As Extended)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    i = ExArrBinPos(p, value, compare)
    If i = 0 Then
        ExArrAdd p, value
    ElseIf compare <= 0 Then
        ExArrIns p, i, value
    ElseIf i < @p.count Then
        ExArrIns p, i + 1, value
    Else
        ExArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Extended Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub ExArrStkPush(ByVal p As ExArr Ptr, ByVal value As Extended)
    'push value on top of Stack
    ExArrAdd p, value
End Sub

Function ExArrStkPeek(ByVal p As ExArr Ptr) As Extended
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function ExArrStkPop(ByVal p As ExArr Ptr) As Extended
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        ExArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Extended Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub ExArrQuePush(ByVal p As ExArr Ptr, ByVal value As Extended)
    'add value to end of Queue
    ExArrAdd p, value
End Sub

Function ExArrQuePeek(ByVal p As ExArr Ptr) As Extended
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function ExArrQuePop(ByVal p As ExArr Ptr) As Extended
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        ExArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Extended Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub ExArrPushFirst(ByVal p As ExArr Ptr, ByVal value As Extended)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then ExArrIns p, 1, value Else ExArrAdd p, value
End Sub

Sub ExArrPushLast(ByVal p As ExArr Ptr, ByVal value As Extended)
    'add value to and of Deque (double-ended Queue)
    ExArrAdd p, value
End Sub

Function ExArrPeekFirst(ByVal p As ExArr Ptr) As Extended
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function ExArrPeekLast(ByVal p As ExArr Ptr) As Extended
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function ExArrPopFirst(ByVal p As ExArr Ptr) As Extended
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        ExArrDel p, 1
    End If
End Function

Function ExArrPopLast(ByVal p As ExArr Ptr) As Extended
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        ExArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function ExArrClone(ByVal p As ExArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As ExArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    clone = ExArrNew() : If Err Then Exit Function
    If @p.count Then
        ExArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeEx, @clone.arr + %SizeEx, @p.count * %SizeEx
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function ExArrStore(ByVal p As ExArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>ExArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeEx, @p.count * %SizeEx)
End Function

Sub ExArrRestore(ByVal p As ExArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    ExArrClear p
    If strLen Then
        ExArrReDim p, strLen / %SizeEx : If Err Then Exit Sub
        Poke$ @p.arr + %SizeEx, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub ExArrFileStore(ByVal p As ExArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    s = ExArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub ExArrFileRestore(ByVal p As ExArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>ExArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        ExArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub ExArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeEx), arr + (toIndex * %SizeEx),  Count * %SizeEx
End Sub

Function ExArrBinPos(ByVal p As ExArr Ptr, ByVal value As Extended, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Extended Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function
