#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro InArrTag = -1821650520
Type InArr
    tag As Long
    count As Long
    arr As Integer Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Integer Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = InArrNew() 'get handle for new container
    '       h = InArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function InArrNew() As Long
    'allocate new container - return handle
    Local p As InArr Ptr
    p = MemAlloc(SizeOf(InArr))
    ExitF(p=0, LibErrM)
    @p.tag = InArrTag
    Function = p
End Function

Function InArrFinal(ByVal p As InArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>InArrTag, LibErrH)
        InArrClear p
        MemFree(p)
    End If
End Function

Function InArrValidate(ByVal p As InArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = InArrTag Then Function = @p.tag
End Function

Sub InArrClear(ByVal p As InArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeIn
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function InArrCount(ByVal p As InArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    Function = @p.count
End Function

Sub InArrReDim(ByVal p As InArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        InArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeIn
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeIn)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeIn
    End If
End Sub

Function InArrGet(ByVal p As InArr Ptr, ByVal index As Long) As Integer
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub InArrSet(ByVal p As InArr Ptr, ByVal index As Long, ByVal value As Integer)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub InArrAdd(ByVal p As InArr Ptr, ByVal value As Integer)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    InArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub InArrIns(ByVal p As InArr Ptr, ByVal index As Long, ByVal value As Integer)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    InArrReDim p, @p.count + 1 : If Err Then Exit Sub
    InArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub InArrDel(ByVal p As InArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then InArrMove @p.arr, i + 1, i, @p.count - i
    InArrReDim p, @p.count - 1
End Sub

Sub InArrSort(ByVal p As InArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As Integer
    Local arr As Integer Ptr
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub InArrUniqueSort(ByVal p As InArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As Integer Ptr
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    InArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then InArrDel p, i + 1
    Next i
End Sub

Sub InArrReverse(ByVal p As InArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As Integer Ptr
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function InArrBinSearch(ByVal p As InArr Ptr, ByVal value As Integer) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As Integer Ptr
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub InArrBinInsert(ByVal p As InArr Ptr, ByVal value As Integer)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    i = InArrBinPos(p, value, compare)
    If i = 0 Then
        InArrAdd p, value
    ElseIf compare <= 0 Then
        InArrIns p, i, value
    ElseIf i < @p.count Then
        InArrIns p, i + 1, value
    Else
        InArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Integer Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub InArrStkPush(ByVal p As InArr Ptr, ByVal value As Integer)
    'push value on top of Stack
    InArrAdd p, value
End Sub

Function InArrStkPeek(ByVal p As InArr Ptr) As Integer
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function InArrStkPop(ByVal p As InArr Ptr) As Integer
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        InArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Integer Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub InArrQuePush(ByVal p As InArr Ptr, ByVal value As Integer)
    'add value to end of Queue
    InArrAdd p, value
End Sub

Function InArrQuePeek(ByVal p As InArr Ptr) As Integer
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function InArrQuePop(ByVal p As InArr Ptr) As Integer
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        InArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Integer Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub InArrPushFirst(ByVal p As InArr Ptr, ByVal value As Integer)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then InArrIns p, 1, value Else InArrAdd p, value
End Sub

Sub InArrPushLast(ByVal p As InArr Ptr, ByVal value As Integer)
    'add value to and of Deque (double-ended Queue)
    InArrAdd p, value
End Sub

Function InArrPeekFirst(ByVal p As InArr Ptr) As Integer
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function InArrPeekLast(ByVal p As InArr Ptr) As Integer
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function InArrPopFirst(ByVal p As InArr Ptr) As Integer
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        InArrDel p, 1
    End If
End Function

Function InArrPopLast(ByVal p As InArr Ptr) As Integer
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        InArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function InArrClone(ByVal p As InArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As InArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    clone = InArrNew() : If Err Then Exit Function
    If @p.count Then
        InArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeIn, @clone.arr + %SizeIn, @p.count * %SizeIn
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function InArrStore(ByVal p As InArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>InArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeIn, @p.count * %SizeIn)
End Function

Sub InArrRestore(ByVal p As InArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    InArrClear p
    If strLen Then
        InArrReDim p, strLen / %SizeIn : If Err Then Exit Sub
        Poke$ @p.arr + %SizeIn, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub InArrFileStore(ByVal p As InArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    s = InArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub InArrFileRestore(ByVal p As InArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>InArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        InArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub InArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeIn), arr + (toIndex * %SizeIn),  Count * %SizeIn
End Sub

Function InArrBinPos(ByVal p As InArr Ptr, ByVal value As Integer, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Integer Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function
