#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro WdArrTag = 264591370
Type WdArr
    tag As Long
    count As Long
    arr As Word Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Word Array Container
    '       one-based index
    '       ReDim automatic for all operations
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = WdArrNew() 'get handle for new container
    '       h = WdArrFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function WdArrNew() As Long
    'allocate new container - return handle
    Local p As WdArr Ptr
    p = MemAlloc(SizeOf(WdArr))
    ExitF(p=0, LibErrM)
    @p.tag = WdArrTag
    Function = p
End Function

Function WdArrFinal(ByVal p As WdArr Ptr) As Long
    'free allocated container - return null
    If p Then
        ExitF(@p.tag<>WdArrTag, LibErrH)
        WdArrClear p
        MemFree(p)
    End If
End Function

Function WdArrValidate(ByVal p As WdArr Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = WdArrTag Then Function = @p.tag
End Function

Sub WdArrClear(ByVal p As WdArr Ptr)
    'delete all data
    Local hMem As Long
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then
        hMem = @p.arr + %SizeWd
        MemFree(hMem)
    End If
    @p.arr = 0
    @p.count = 0
End Sub

Function WdArrCount(ByVal p As WdArr Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    Function = @p.count
End Function

Sub WdArrReDim(ByVal p As WdArr Ptr, ByVal Count As Long)
    'ReDim array - data preserved
    Local mem As Long
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    ExitS(Count<0, LibErrA)
    If Count = 0 Then
        WdArrClear p
    ElseIf Count <> @p.count Then
        If @p.count Then mem = @p.arr + %SizeWd
        @p.count = 0
        @p.arr = 0
        mem = MemReAlloc(mem, Count * %SizeWd)
        ExitS(mem=0, LibErrM)
        @p.count = Count
        @p.arr = mem - %SizeWd
    End If
End Sub

Function WdArrGet(ByVal p As WdArr Ptr, ByVal index As Long) As Word
    'get value at index - one-based index
    Register i As Long : i = index
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    ExitF(i<1 Or i>@p.count, LibErrB)
    Function = @p.@arr[i]
End Function

Sub WdArrSet(ByVal p As WdArr Ptr, ByVal index As Long, ByVal value As Word)
    'set value at index - one-based index
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    @p.@arr[i] = value
End Sub

Sub WdArrAdd(ByVal p As WdArr Ptr, ByVal value As Word)
    'append value to end of array - ReDim automatic
    Err = 0
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    WdArrReDim p, @p.count + 1 : If Err Then Exit Sub
    @p.@arr[@p.count] = value
End Sub

Sub WdArrIns(ByVal p As WdArr Ptr, ByVal index As Long, ByVal value As Word)
    'insert value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    Err = 0
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    WdArrReDim p, @p.count + 1 : If Err Then Exit Sub
    WdArrMove @p.arr, i, i + 1, @p.count - i
    @p.@arr[i] = value
End Sub

Sub WdArrDel(ByVal p As WdArr Ptr, ByVal index As Long)
    'remove value at index - one-based index - ReDim automatic
    Register i As Long : i = index
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    ExitS(i<1 Or i>@p.count, LibErrB)
    If i < @p.count Then WdArrMove @p.arr, i + 1, i, @p.count - i
    WdArrReDim p, @p.count - 1
End Sub

Sub WdArrSort(ByVal p As WdArr Ptr)
    'sort array - fast non-recursive Quick Sort
    Local i, j, k, leftIndex, rightIndex, counter As Long
    Local value As Word
    Local arr As Word Ptr
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count > 1 Then
        arr = @p.arr
        leftIndex = 1
        rightIndex = @p.count
        !PUSH leftIndex
        !PUSH rightIndex
        counter = 1
        While counter
            !POP rightIndex
            !POP leftIndex
            Decr counter
            i = leftIndex
            j = rightIndex
            k = i + j
            Shift Right k, 1
            value = @arr[k]
            While i <= j
                While @arr[i] < value
                    Incr i
                Wend
                While @arr[j] > value
                    Decr j
                Wend
                If i <= j Then
                    Swap @arr[i], @arr[j]
                    Incr i : Decr j
                End If
            Wend
            If leftIndex < j Then
                !PUSH leftIndex
                !PUSH j
                Incr counter
            End If
            If i < rightIndex Then
                !PUSH i
                !PUSH rightIndex
                Incr counter
            End If
        Wend
    End If
End Sub

Sub WdArrUniqueSort(ByVal p As WdArr Ptr)
    'sort array and delete all duplicates
    Register i As Long
    Local arr As Word Ptr
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    WdArrSort p
    arr = @p.arr
    For i = @p.count - 1 To 1 Step -1
        If @arr[i + 1] = @arr[i] Then WdArrDel p, i + 1
    Next i
End Sub

Sub WdArrReverse(ByVal p As WdArr Ptr)
    'reverse array
    Register i As Long
    Register j As Long
    Local arr As Word Ptr
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    i = 1
    j = @p.count
    arr = @p.arr
    While i < j
        Swap @arr[i], @arr[j]
        Incr i
        Decr j
    Wend
End Sub

Function WdArrBinSearch(ByVal p As WdArr Ptr, ByVal value As Word) As Long
    'binary search for item
    'return index - zero if not found
    'array must be sorted for valid results
    Local i, top, bot As Long
    Local arr As Word Ptr
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
        ElseIf value < @arr[i] Then
            top = i - 1
        Else
            Function = i
            Exit Loop
        End If
    Wend
End Function

Sub WdArrBinInsert(ByVal p As WdArr Ptr, ByVal value As Word)
    'binary insert value into array - ok if array empty - array must be sorted for valid results
    Local i, compare As Long
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    i = WdArrBinPos(p, value, compare)
    If i = 0 Then
        WdArrAdd p, value
    ElseIf compare <= 0 Then
        WdArrIns p, i, value
    ElseIf i < @p.count Then
        WdArrIns p, i + 1, value
    Else
        WdArrAdd p, value
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Word Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub WdArrStkPush(ByVal p As WdArr Ptr, ByVal value As Word)
    'push value on top of Stack
    WdArrAdd p, value
End Sub

Function WdArrStkPeek(ByVal p As WdArr Ptr) As Word
    'get top value on Stack
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function WdArrStkPop(ByVal p As WdArr Ptr) As Word
    'get and remove top value on Stack
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        WdArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Word Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub WdArrQuePush(ByVal p As WdArr Ptr, ByVal value As Word)
    'add value to end of Queue
    WdArrAdd p, value
End Sub

Function WdArrQuePeek(ByVal p As WdArr Ptr) As Word
    'get first value in Queue
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function WdArrQuePop(ByVal p As WdArr Ptr) As Word
    'get and remove first value in Queue
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        WdArrDel p, 1
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Word Deque Procedures
    '----------------------------------------------------------------------------------------
'--

Sub WdArrPushFirst(ByVal p As WdArr Ptr, ByVal value As Word)
    'add value to front of Deque (double-ended Queue)
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then WdArrIns p, 1, value Else WdArrAdd p, value
End Sub

Sub WdArrPushLast(ByVal p As WdArr Ptr, ByVal value As Word)
    'add value to and of Deque (double-ended Queue)
    WdArrAdd p, value
End Sub

Function WdArrPeekFirst(ByVal p As WdArr Ptr) As Word
    'get first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[1]
End Function

Function WdArrPeekLast(ByVal p As WdArr Ptr) As Word
    'get last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then Function = @p.@arr[@p.count]
End Function

Function WdArrPopFirst(ByVal p As WdArr Ptr) As Word
    'get and remove first value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[1]
        WdArrDel p, 1
    End If
End Function

Function WdArrPopLast(ByVal p As WdArr Ptr) As Word
    'get and remove last value in Deque (double-ended Queue)
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then
        Function = @p.@arr[@p.count]
        WdArrDel p, @p.count
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function WdArrClone(ByVal p As WdArr Ptr) As Long
    'duplicate container - return clone's handle
    Local clone As WdArr Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    clone = WdArrNew()  : If Err Then Exit Function
    If @p.count Then
        WdArrReDim clone, @p.count : If Err Then Exit Function
        Memory Copy @p.arr + %SizeWd, @clone.arr + %SizeWd, @p.count * %SizeWd
    End If
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function WdArrStore(ByVal p As WdArr Ptr) As String
    'store container to string
    ExitF(p=0 Or @p.tag<>WdArrTag, LibErrH)
    If @p.count Then Function = Peek$(@p.arr + %SizeWd, @p.count * %SizeWd)
End Function

Sub WdArrRestore(ByVal p As WdArr Ptr, ByVal s As String)
    'restore container from string
    Register strLen As Long : strLen = Len(s)
    Err = 0
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    WdArrClear p
    If strLen Then
        WdArrReDim p, strLen / %SizeWd : If Err Then Exit Sub
        Poke$ @p.arr + %SizeWd, s
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub WdArrFileStore(ByVal p As WdArr Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    s = WdArrStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub WdArrFileRestore(ByVal p As WdArr Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>WdArrTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        WdArrRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub WdArrMove(ByVal arr As Long, ByVal fromIndex As Long, ByVal toIndex As Long, ByVal Count As Long) Private
    Memory Copy arr + (fromIndex * %SizeWd), arr + (toIndex * %SizeWd),  Count * %SizeWd
End Sub

Function WdArrBinPos(ByVal p As WdArr Ptr, ByVal value As Word, ByRef compare As Long) Private As Long
    'return value's binary index position in array
    'array must be sorted for valid results
    '   returned index = 0 if array empty
    '   compare < 0 : value < returned index
    '   compare = 0 : value = returned index
    '   compare > 0 : value > returned index
    '   if return = last index, and compare > 0, then value > all values in array
    Local i, top, bot As Long
    Local arr As Word Ptr
    compare = -1
    bot = 1
    top = @p.count
    arr = @p.arr
    While top >= bot
        i = bot + top
        Shift Right i, 1
        If value > @arr[i] Then
            bot = i + 1
            compare = 1
        ElseIf value < @arr[i] Then
            top = i - 1
            compare = -1
        Else
            compare = 0
            Exit Loop
        End If
    Wend
    Function = i
End Function
