#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\Hash.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\List\QdLst.inc"

'++
    '----------------------------------------------------------------------------------------
    '   Long/Quad ~ Hash Table Container
    '       http://en.wikipedia.org/wiki/Hash_table
    '       Key must be unique Long
    '       Value stored/retrieved/removed using unique lookup Key
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = LnQdHshNew() 'get handle for new container
    '       h = LnQdHshFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Macro LnQdHshTag = 1600944301
Type LnQdHshNode
    next As LnQdHshNode Ptr
    K As Long
    V As Quad
End Type
Type LnQdHsh
    tag As Long
    count As Long
    cap As Long
    arr As Long Ptr
End Type

Function LnQdHshNew(ByVal capacity As Long) As Long
    'allocate new container - return handle
    'capacity = number of expected Keys (minium 100)
    Local i As Long
    Local p As LnQdHsh Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = LnQdHshTag
    @p.cap = Max&(capacity, 10)
    @p.arr = MemAlloc(@p.cap * 4)
    ExitF(@p.arr=0, LibErrM)
    Function = p
End Function

Function LnQdHshFinal(ByVal p As LnQdHsh Ptr) As Long
    'free allocated container - return null
    Local i As Long
    Local n, x As LnQdHshNode Ptr
    If p Then
        ExitF(@p.tag<>LnQdHshTag, LibErrH)
        For i = 0 To @p.cap - 1
            n = @p.@arr[i] : @p.@arr[i] = 0
            While n
                x = n : n = @n.next
                MemFree(x)
            Wend
        Next i
        @p.arr = MemFree(@p.arr)
    End If
End Function

Function LnQdHshValidate(ByVal p As LnQdHsh Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = LnQdHshTag Then Function = @p.tag
End Function

Sub LnQdHshClear(ByVal p As LnQdHsh Ptr)
    'delete all data
    Local i As Long
    Local n, x As LnQdHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    For i = 0 To  @p.cap - 1
        n = @p.@arr[i] : @p.@arr[i] = 0
        While n
            x = n : n = @n.next
            MemFree(x)
        Wend
    Next i
    @p.count = 0
End Sub

Function LnQdHshCount(ByVal p As LnQdHsh Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    Function = @p.count
End Function

Function LnQdHshCapGet(ByVal p As LnQdHsh Ptr) As Long
    'get Hash Table capacity
    ExitF(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    Function = @p.cap
End Function

Sub LnQdHshCapSet(ByVal p As LnQdHsh Ptr, ByVal capacity As Long)
    'set Hash Table capacity
    'rebuild Hash Table with new capacity - data preserved
    'capacity should be about the same as number of stored Keys
    Local i, oldCap, newCap As Long
    Local arr As Long Ptr
    Local n, x As LnQdHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    oldCap = @p.cap
    newCap = Max&(capacity, 10)
    arr = MemAlloc(newCap * 4)
    ExitS(arr=0, LibErrM)
    Swap @p.arr, arr
    @p.cap = newCap
    @p.count = 0
    For i = 0 To oldCap - 1
        n = @arr[i] : @arr[i] = 0
        While n
            x = n : n = @n.next
            LnQdHshSet p, @x.K, @x.V
            MemFree(x)
        Wend
    Next i
    MemFree(arr)
End Sub

Sub LnQdHshSet(ByVal p As LnQdHsh Ptr, ByVal key As Long, ByVal value As Quad, Opt ByVal DontReplace As Byte)
    'add Key/Value to Tash Table - Value replaced if Key exist unless DontReplace = True
    Local i As Long
    Local n, nn As LnQdHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    If n Then
        Do
            If @n.K = key Then
                If IsFalse DontReplace Then @n.V = value
                Exit Sub
            ElseIf @n.next Then
                n = @n.next
            Else
                Exit Loop
            End If
        Loop
        nn = MemAlloc(SizeOf(LnQdHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = key
        @nn.V = value
        @n.next = nn
        Incr @p.count
    Else
        nn = MemAlloc(SizeOf(LnQdHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = key
        @nn.V = value
        @p.@arr[i] = nn
        Incr @p.count
    End If
End Sub

Function LnQdHshGet(ByVal p As LnQdHsh Ptr, ByVal key As Long) As Quad
    'get Key's associated Value
    Local i As Long
    Local n As LnQdHshNode Ptr
    ExitF(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        If @n.K = key Then
            Function = @n.V
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Function LnQdHshGot(ByVal p As LnQdHsh Ptr, ByVal key As Long) As Byte
    'True/False if Key in Hash Table
    Local i As Long
    Local n As LnQdHshNode Ptr
    ExitF(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        If @n.K = key Then
            Function = 1
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Sub LnQdHshDel(ByVal p As LnQdHsh Ptr, ByVal key As Long)
    'remove Key and associated Value from Hash Table
    Local i As Long
    Local prev, n As LnQdHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    prev = 0
    n = @p.@arr[i]
    While n
        If @n.K = key Then
            If prev Then
                @prev.next = @n.next
            Else
                @p.@arr[i] = @n.next
            End If
            MemFree(n)
            ExitS(@p.count=0, LibErrU)
            Decr @p.count
            Exit Loop
        End If
        prev = n
        n = @n.next
    Wend
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnQdHshClone(ByVal p As LnQdHsh Ptr) As Long
    'create duplicate container - return handle to cloned container
    Local i As Long
    Local n As LnQdHshNode Ptr
    Local clone As Long
    Err = 0
    ExitF(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    clone = LnQdHshNew(@p.cap) : If Err Then Exit Function
    For i = 0 To @p.cap - 1
        n = @p.@arr[i]
        While n
            LnQdHshSet clone, @n.K, @n.V
            n = @n.next
        Wend
    Next i
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnQdHshStore(ByVal p As LnQdHsh Ptr) As String
    'store container to string
    Local i As Long
    Local keys, vals, store As Long
    Local n As LnQdHshNode Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    keys = LnLstNew() : If Err Then Exit Function
    vals = QdLstNew() : If Err Then Exit Function
    store = SsLstNew() : If Err Then Exit Function
    If @p.count Then
        For i = 0 To @p.cap - 1
            n = @p.@arr[i]
            While n
                LnLstAdd keys, @n.K
                QdLstAdd vals, @n.V
                n = @n.next
            Wend
        Next i
        SsLstAdd store, Mkl$(@p.cap)
        SsLstAdd store, LnLstStore(keys)
        SsLstAdd store, QdLstStore(vals)
        Function = SsLstStore(store)
    End If
    keys = LnLstFinal(keys)
    vals = QdLstFinal(vals)
    store = SsLstFinal(store)
End Function

Sub LnQdHshRestore(ByVal p As LnQdHsh Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, store As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    LnQdHshClear p
    keys = LnLstNew() : If Err Then Exit Sub
    vals = QdLstNew() : If Err Then Exit Sub
    store = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore store, s : If Err Then Exit Sub
        ExitS(SsLstCount(store)<>3, LibErrU)
        LnQdHshCapSet p, Cvl(SsLstPopFirst(store))
        LnLstRestore keys, SsLstPopFirst(store)
        QdLstRestore vals, SsLstPopFirst(store)
        ExitS(LnLstCount(keys)<>QdLstCount(vals), LibErrU)
        While LnLstCount(keys)
            LnQdHshSet p, LnLstPopFirst(keys), QdLstPopFirst(vals)
        Wend
    End If
    keys = LnLstFinal(keys)
    vals = QdLstFinal(vals)
    store = SsLstFinal(store)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnQdHshFileStore(ByVal p As LnQdHsh Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    s = LnQdHshStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnQdHshFileRestore(ByVal p As LnQdHsh Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>LnQdHshTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnQdHshRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub
