#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\Hash.inc"
#Include Once "C:\HLib3\String\SsStr.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\LnLst.inc"

'++
    '----------------------------------------------------------------------------------------
    '   Long/String ~ Hash Table Container
    '       http://en.wikipedia.org/wiki/Hash_table
    '       Key must be unique Long
    '       Value stored/retrieved/removed using unique lookup Key
    '
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = LnSsHshNew() 'get handle for new container
    '       h = LnSsHshFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Macro LnSsHshTag = -324365412
Type LnSsHshNode
    next As LnSsHshNode Ptr
    K As Long
    V As Long
End Type
Type LnSsHsh
    tag As Long
    count As Long
    cap As Long
    arr As Long Ptr
End Type

Function LnSsHshNew(ByVal capacity As Long) As Long
    'allocate new container - return handle
    'capacity = number of expected Keys (minium 100)
    Local i As Long
    Local p As LnSsHsh Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = LnSsHshTag
    @p.cap = Max&(capacity, 10)
    @p.arr = MemAlloc(@p.cap * 4)
    ExitF(@p.arr=0, LibErrM)
    Function = p
End Function

Function LnSsHshFinal(ByVal p As LnSsHsh Ptr) As Long
    'free allocated container - return null
    Local i As Long
    Local n, x As LnSsHshNode Ptr
    If p Then
        ExitF(@p.tag<>LnSsHshTag, LibErrH)
        For i = 0 To @p.cap - 1
            n = @p.@arr[i] : @p.@arr[i] = 0
            While n
                x = n : n = @n.next
                SsFinal(@x.V)
                MemFree(x)
            Wend
        Next i
        @p.arr = MemFree(@p.arr)
    End If
End Function

Function LnSsHshValidate(ByVal p As LnSsHsh Ptr) As Long
    'True/False if valid handle for this container
    If p And @p.tag = LnSsHshTag Then Function = @p.tag
End Function

Sub LnSsHshClear(ByVal p As LnSsHsh Ptr)
    'delete all data
    Local i As Long
    Local n, x As LnSsHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    For i = 0 To  @p.cap - 1
        n = @p.@arr[i] : @p.@arr[i] = 0
        While n
            x = n : n = @n.next
            SsFinal(@x.V)
            MemFree(x)
        Wend
    Next i
    @p.count = 0
End Sub

Function LnSsHshCount(ByVal p As LnSsHsh Ptr) As Long
    'get item count
    ExitF(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    Function = @p.count
End Function

Function LnSsHshCapGet(ByVal p As LnSsHsh Ptr) As Long
    'get Hash Table capacity
    ExitF(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    Function = @p.cap
End Function

Sub LnSsHshCapSet(ByVal p As LnSsHsh Ptr, ByVal capacity As Long)
    'set Hash Table capacity
    'rebuild Hash Table with new capacity - data preserved
    'capacity should be about the same as number of stored Keys
    Local i, oldCap, newCap As Long
    Local arr As Long Ptr
    Local n, x As LnSsHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    oldCap = @p.cap
    newCap = Max&(capacity, 10)
    arr = MemAlloc(newCap * 4)
    ExitS(arr=0, LibErrM)
    Swap @p.arr, arr
    @p.cap = newCap
    @p.count = 0
    For i = 0 To oldCap - 1
        n = @arr[i] : @arr[i] = 0
        While n
            x = n : n = @n.next
            LnSsHshSet p, @x.K, SsGet(@x.V)
            SsFinal(@x.V)
            MemFree(x)
        Wend
    Next i
    MemFree(arr)
End Sub

Sub LnSsHshSet(ByVal p As LnSsHsh Ptr, ByVal key As Long, ByRef value As String, Opt ByVal DontReplace As Byte)
    'add Key/Value to Tash Table - Value replaced if Key exist unless DontReplace = True
    Local i As Long
    Local n, nn As LnSsHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    If n Then
        Do
            If @n.K = key Then
                If IsFalse DontReplace Then SsSet @n.V, value
                Exit Sub
            ElseIf @n.next Then
                n = @n.next
            Else
                Exit Loop
            End If
        Loop
        nn = MemAlloc(SizeOf(LnSsHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = key
        @nn.V = SsSetNew(value)
        @n.next = nn
        Incr @p.count
    Else
        nn = MemAlloc(SizeOf(LnSsHshNode))
        ExitS(nn=0, LibErrM)
        @nn.K = key
        @nn.V = SsSetNew(value)
        @p.@arr[i] = nn
        Incr @p.count
    End If
End Sub

Function LnSsHshGet(ByVal p As LnSsHsh Ptr, ByVal key As Long) As String
    'get Key's associated Value
    Local i As Long
    Local n As LnSsHshNode Ptr
    ExitF(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        If @n.K = key Then
            Function = SsGet(@n.V)
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Function LnSsHshGot(ByVal p As LnSsHsh Ptr, ByVal key As Long) As Byte
    'True/False if Key in Hash Table
    Local i As Long
    Local n As LnSsHshNode Ptr
    ExitF(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitF(i>=@p.cap, LibErrU)
    n = @p.@arr[i]
    While n
        If @n.K = key Then
            Function = 1
            Exit Loop
        End If
        n = @n.next
    Wend
End Function

Sub LnSsHshDel(ByVal p As LnSsHsh Ptr, ByVal key As Long)
    'remove Key and associated Value from Hash Table
    Local i As Long
    Local prev, n As LnSsHshNode Ptr
    ExitS(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    i = LnHash(key, @p.cap)
    ExitS(i>=@p.cap, LibErrU)
    prev = 0
    n = @p.@arr[i]
    While n
        If @n.K = key Then
            If prev Then
                @prev.next = @n.next
            Else
                @p.@arr[i] = @n.next
            End If
            SsFinal(@n.V)
            MemFree(n)
            ExitS(@p.count=0, LibErrU)
            Decr @p.count
            Exit Loop
        End If
        prev = n
        n = @n.next
    Wend
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnSsHshClone(ByVal p As LnSsHsh Ptr) As Long
    'create duplicate container - return handle to cloned container
    Local i As Long
    Local n As LnSsHshNode Ptr
    Local clone As Long
    Err = 0
    ExitF(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    clone = LnSsHshNew(@p.cap) : If Err Then Exit Function
    For i = 0 To @p.cap - 1
        n = @p.@arr[i]
        While n
            LnSsHshSet clone, @n.K, SsGet(@n.V)
            n = @n.next
        Wend
    Next i
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnSsHshStore(ByVal p As LnSsHsh Ptr) As String
    'store container to string
    Local i As Long
    Local keys, vals, store As Long
    Local n As LnSsHshNode Ptr
    Err = 0
    ExitF(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    keys = LnLstNew() : If Err Then Exit Function
    vals = SsLstNew() : If Err Then Exit Function
    store = SsLstNew() : If Err Then Exit Function
    If @p.count Then
        For i = 0 To @p.cap - 1
            n = @p.@arr[i]
            While n
                LnLstAdd keys, @n.K
                SsLstAdd vals, SsGet(@n.V)
                n = @n.next
            Wend
        Next i
        SsLstAdd store, Mkl$(@p.cap)
        SsLstAdd store, LnLstStore(keys)
        SsLstAdd store, SsLstStore(vals)
        Function = SsLstStore(store)
    End If
    keys = LnLstFinal(keys)
    vals = SsLstFinal(vals)
    store = SsLstFinal(store)
End Function

Sub LnSsHshRestore(ByVal p As LnSsHsh Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, store As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    LnSsHshClear p
    keys = LnLstNew() : If Err Then Exit Sub
    vals = SsLstNew() : If Err Then Exit Sub
    store = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore store, s : If Err Then Exit Sub
        ExitS(SsLstCount(store)<>3, LibErrU)
        LnSsHshCapSet p, Cvl(SsLstPopFirst(store))
        LnLstRestore keys, SsLstPopFirst(store)
        SsLstRestore vals, SsLstPopFirst(store)
        ExitS(LnLstCount(keys)<>SsLstCount(vals), LibErrU)
        While LnLstCount(keys)
            LnSsHshSet p, LnLstPopFirst(keys), SsLstPopFirst(vals)
        Wend
    End If
    keys = LnLstFinal(keys)
    vals = SsLstFinal(vals)
    store = SsLstFinal(store)
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnSsHshFileStore(ByVal p As LnSsHsh Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    s = LnSsHshStore(p) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnSsHshFileRestore(ByVal p As LnSsHsh Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(p=0 Or @p.tag<>LnSsHshTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnSsHshRestore p, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub
