#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro LnLstNodeTag = 1540788735
Macro LnLstTag = 667850440
Type LnLstNode
    tag As Long
    next As LnLstNode Ptr
    prev As LnLstNode Ptr
    value As Long
End Type
Type LnLst
    tag As Long
    count As Long
    first As LnLstNode Ptr
    last As LnLstNode Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Long List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = LnLstNew() 'get handle for new container
    '       h = LnLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function LnLstNew() As Long
    'allocate new container - return handle
    Local pLst As LnLst Ptr
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = LnLstTag
    Function = pLst
End Function

Function LnLstFinal(ByVal pLst As LnLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>LnLstTag, LibErrH)
        LnLstClear pLst
        MemFree(pLst)
    End If
End Function

Function LnLstValidate(ByVal pLst As LnLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = LnLstTag Then Function = @pLst.tag
End Function

Sub LnLstClear(ByVal pLst As LnLst Ptr)
    'delete all data
    Local node As LnLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    While @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    Wend
    @pLst.last = 0
    @pLst.count = 0
End Sub

Function LnLstCount(ByVal pLst As LnLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    Function = @pLst.count
End Function

Sub LnLstAdd(ByVal pLst As LnLst Ptr, ByVal value As Long)
    'append Value to end of List
    Local node As LnLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.count Then
        node = LnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    Else
        node = LnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Sub LnLstIns(ByVal pLst As LnLst Ptr, ByVal value As Long)
    'insert Value at front of List
    Local node As LnLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.count Then
        node = LnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    Else
        node = LnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Function LnLstFirst(ByVal pLst As LnLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    Function = @pLst.first
End Function

Function LnLstLast(ByVal pLst As LnLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    Function = @pLst.last
End Function

Function LnLstNext(ByVal pNode As LnLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function LnLstPrev(ByVal pNode As LnLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function LnLstGet(ByVal pNode As LnLstNode Ptr) As Long
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    Function = @pNode.value
End Function

Sub LnLstSet(ByVal pNode As LnLstNode Ptr, ByVal value As Long)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    @pNode.value = value
End Sub

Sub LnLstInsPrev(ByVal pLst As LnLst Ptr, ByVal pNode As LnLstNode Ptr, ByVal value As Long)
    'insert Value before node
    Local node As LnLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>LnLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    If pNode = @pLst.first Then
        LnLstIns pLst, value
    Else
        ExitS(@pNode.prev=0, LibErrU)
        node = LnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    End If
End Sub

Sub LnLstInsNext(ByVal pLst As LnLst Ptr, ByVal pNode As LnLstNode Ptr, ByVal value As Long)
    'insert Value after Cursor
    Local node As LnLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>LnLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    If pNode = @pLst.last Then
        LnLstAdd pLst, value
    Else
        ExitS(@pNode.next=0, LibErrU)
        node = LnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    End If
End Sub

Sub LnLstDel(ByVal pLst As LnLst Ptr, ByVal pNode As LnLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>LnLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>LnLstNodeTag, LibErrH)
        If @pLst.first = pNode Then @pLst.first = @pNode.next
        If @pLst.last = pNode Then @pLst.last = @pNode.prev
        If @pNode.prev Then @pNode.@prev.next = @pNode.next
        If @pNode.next Then @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        Decr @pLst.count
        MemFree(pNode)
    End If
End Sub

Sub LnLstDelPrev(ByVal pLst As LnLst Ptr, ByVal pNode As LnLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>LnLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstDel pLst, @pNode.prev
End Sub

Sub LnLstDelNext(ByVal pLst As LnLst Ptr, ByVal pNode As LnLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>LnLstTag Or @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub LnLstStkPush(ByVal pLst As LnLst Ptr, ByVal value As Long)
    'Push Value on Stack
    LnLstAdd pLst, value
End Sub

Function LnLstStkPeek(ByVal pLst As LnLst Ptr) As Long
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function LnLstStkPop(ByVal pLst As LnLst Ptr) As Long
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        LnLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub LnLstQuePush(ByVal pLst As LnLst Ptr, ByVal value As Long)
    'Add Value to end of Queue
    LnLstAdd pLst, value
End Sub

Function LnLstQuePeek(ByVal pLst As LnLst Ptr) As Long
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function LnLstQuePop(ByVal pLst As LnLst Ptr) As Long
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        LnLstDel pLst, @pLst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub LnLstPushFirst(ByVal pLst As LnLst Ptr, ByVal value As Long)
    'Add Value at front of container
    LnLstIns pLst, value
End Sub

Sub LnLstPushLast(ByVal pLst As LnLst Ptr, ByVal value As Long)
    'Add Value at end of container
    LnLstAdd pLst, value
End Sub

Function LnLstPeekFirst(ByVal pLst As LnLst Ptr) As Long
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function LnLstPeekLast(ByVal pLst As LnLst Ptr) As Long
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function LnLstPopFirst(ByVal pLst As LnLst Ptr) As Long
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        LnLstDel pLst, @pLst.first
    End If
End Function

Function LnLstPopLast(ByVal pLst As LnLst Ptr) As Long
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        LnLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnLstClone(ByVal pLst As LnLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As LnLst Ptr
    Local node As LnLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    Err = 0
    pClone = LnLstNew() : If Err Then Exit Function
    node = @pLst.first
    While node
        LnLstAdd pClone, @node.value
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnLstStore(ByVal pLst As LnLst Ptr) As String
    'store container to String
    Local s As String
    Local node As LnLstNode Ptr
    Local p As Long Ptr
    ExitF(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    If @pLst.count Then
        s = Nul$(@pLst.count * %SizeLn)
        p = StrPtr(s)
        node = @pLst.first
        While node
            @p = @node.value : Incr p
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub LnLstRestore(ByVal pLst As LnLst Ptr, ByRef s As String)
    'restore container from string
    Local items As Long
    Local p As Long Ptr
    ExitS(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    LnLstClear pLst
    If Len(s) Then
        p = StrPtr(s)
        items = Len(s) / %SizeLn
        While items
            LnLstAdd pLst, @p
            Incr p
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnLstFileStore(ByVal pLst As LnLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    s = LnLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnLstFileRestore(ByVal pLst As LnLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>LnLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function LnLstNodeAlloc(ByVal pLst As LnLst Ptr) Private As Long
    Local node As LnLstNode Ptr
    node = MemAlloc(SizeOf(LnLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = LnLstNodeTag
    Incr @pLst.count
    Function = node
End Function
