#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro InLstNodeTag = 1834559302
Macro InLstTag = 2136895274
Type InLstNode
    tag As Long
    next As InLstNode Ptr
    prev As InLstNode Ptr
    value As Integer
End Type
Type InLst
    tag As Long
    count As Long
    first As InLstNode Ptr
    last As InLstNode Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Integer List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = InLstNew() 'get handle for new container
    '       h = InLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function InLstNew() As Long
    'allocate new container - return handle
    Local pLst As InLst Ptr
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = InLstTag
    Function = pLst
End Function

Function InLstFinal(ByVal pLst As InLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>InLstTag, LibErrH)
        InLstClear pLst
        MemFree(pLst)
    End If
End Function

Function InLstValidate(ByVal pLst As InLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = InLstTag Then Function = @pLst.tag
End Function

Sub InLstClear(ByVal pLst As InLst Ptr)
    'delete all data
    Local node As InLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    While @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    Wend
    @pLst.last = 0
    @pLst.count = 0
End Sub

Function InLstCount(ByVal pLst As InLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    Function = @pLst.count
End Function

Sub InLstAdd(ByVal pLst As InLst Ptr, ByVal value As Integer)
    'append Value to end of List
    Local node As InLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.count Then
        node = InLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    Else
        node = InLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Sub InLstIns(ByVal pLst As InLst Ptr, ByVal value As Integer)
    'insert Value at front of List
    Local node As InLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.count Then
        node = InLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    Else
        node = InLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Function InLstFirst(ByVal pLst As InLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    Function = @pLst.first
End Function

Function InLstLast(ByVal pLst As InLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    Function = @pLst.last
End Function

Function InLstNext(ByVal pNode As InLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>InLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function InLstPrev(ByVal pNode As InLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>InLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function InLstGet(ByVal pNode As InLstNode Ptr) As Integer
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>InLstNodeTag, LibErrH)
    Function = @pNode.value
End Function

Sub InLstSet(ByVal pNode As InLstNode Ptr, ByVal value As Integer)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>InLstNodeTag, LibErrH)
    @pNode.value = value
End Sub

Sub InLstInsPrev(ByVal pLst As InLst Ptr, ByVal pNode As InLstNode Ptr, ByVal value As Integer)
    'insert Value before node
    Local node As InLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>InLstTag Or @pNode.tag<>InLstNodeTag, LibErrH)
    If pNode = @pLst.first Then
        InLstIns pLst, value
    Else
        ExitS(@pNode.prev=0, LibErrU)
        node = InLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    End If
End Sub

Sub InLstInsNext(ByVal pLst As InLst Ptr, ByVal pNode As InLstNode Ptr, ByVal value As Integer)
    'insert Value after Cursor
    Local node As InLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>InLstTag Or @pNode.tag<>InLstNodeTag, LibErrH)
    If pNode = @pLst.last Then
        InLstAdd pLst, value
    Else
        ExitS(@pNode.next=0, LibErrU)
        node = InLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    End If
End Sub

Sub InLstDel(ByVal pLst As InLst Ptr, ByVal pNode As InLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>InLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>InLstNodeTag, LibErrH)
        If @pLst.first = pNode Then @pLst.first = @pNode.next
        If @pLst.last = pNode Then @pLst.last = @pNode.prev
        If @pNode.prev Then @pNode.@prev.next = @pNode.next
        If @pNode.next Then @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        Decr @pLst.count
        MemFree(pNode)
    End If
End Sub

Sub InLstDelPrev(ByVal pLst As InLst Ptr, ByVal pNode As InLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>InLstTag Or @pNode.tag<>InLstNodeTag, LibErrH)
    InLstDel pLst, @pNode.prev
End Sub

Sub InLstDelNext(ByVal pLst As InLst Ptr, ByVal pNode As InLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>InLstTag Or @pNode.tag<>InLstNodeTag, LibErrH)
    InLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub InLstStkPush(ByVal pLst As InLst Ptr, ByVal value As Integer)
    'Push Value on Stack
    InLstAdd pLst, value
End Sub

Function InLstStkPeek(ByVal pLst As InLst Ptr) As Integer
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function InLstStkPop(ByVal pLst As InLst Ptr) As Integer
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        InLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub InLstQuePush(ByVal pLst As InLst Ptr, ByVal value As Integer)
    'Add Value to end of Queue
    InLstAdd pLst, value
End Sub

Function InLstQuePeek(ByVal pLst As InLst Ptr) As Integer
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function InLstQuePop(ByVal pLst As InLst Ptr) As Integer
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        InLstDel pLst, @pLst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub InLstPushFirst(ByVal pLst As InLst Ptr, ByVal value As Integer)
    'Add Value at front of container
    InLstIns pLst, value
End Sub

Sub InLstPushLast(ByVal pLst As InLst Ptr, ByVal value As Integer)
    'Add Value at end of container
    InLstAdd pLst, value
End Sub

Function InLstPeekFirst(ByVal pLst As InLst Ptr) As Integer
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function InLstPeekLast(ByVal pLst As InLst Ptr) As Integer
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function InLstPopFirst(ByVal pLst As InLst Ptr) As Integer
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        InLstDel pLst, @pLst.first
    End If
End Function

Function InLstPopLast(ByVal pLst As InLst Ptr) As Integer
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        InLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function InLstClone(ByVal pLst As InLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As InLst Ptr
    Local node As InLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    Err = 0
    pClone = InLstNew() : If Err Then Exit Function
    node = @pLst.first
    While node
        InLstAdd pClone, @node.value
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function InLstStore(ByVal pLst As InLst Ptr) As String
    'store container to String
    Local s As String
    Local node As InLstNode Ptr
    Local p As Integer Ptr
    ExitF(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    If @pLst.count Then
        s = Nul$(@pLst.count * %SizeIn)
        p = StrPtr(s)
        node = @pLst.first
        While node
            @p = @node.value : Incr p
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub InLstRestore(ByVal pLst As InLst Ptr, ByRef s As String)
    'restore container from string
    Local items As Long
    Local p As Integer Ptr
    ExitS(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    InLstClear pLst
    If Len(s) Then
        p = StrPtr(s)
        items = Len(s) / %SizeIn
        While items
            InLstAdd pLst, @p
            Incr p
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub InLstFileStore(ByVal pLst As InLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    s = InLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub InLstFileRestore(ByVal pLst As InLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>InLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        InLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function InLstNodeAlloc(ByVal pLst As InLst Ptr) Private As Long
    Local node As InLstNode Ptr
    node = MemAlloc(SizeOf(InLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = InLstNodeTag
    Incr @pLst.count
    Function = node
End Function
