#INCLUDE THIS ONCE
#INCLUDE ONCE "..\HLib3\HLib.inc"

MACRO LnLstNodeTag = 1540788735
MACRO LnLstTag = 667850440
TYPE LnLstNode
    tag AS LONG
    next AS LnLstNode PTR
    prev AS LnLstNode PTR
    value AS LONG
END TYPE
TYPE LnLst
    tag AS LONG
    count AS LONG
    first AS LnLstNode PTR
    last AS LnLstNode PTR
END TYPE

'++
    '----------------------------------------------------------------------------------------
    '   Long List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = LnLstNew() 'get handle for new container
    '       h = LnLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

FUNCTION LnLstNew() AS LONG
    'allocate new container - return handle
    LOCAL pLst AS LnLst PTR
    pLst = MemAlloc(SIZEOF(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = LnLstTag
    FUNCTION = pLst
END FUNCTION

FUNCTION LnLstFinal(BYVAL pLst AS LnLst PTR) AS LONG
    'free allocated container - return null
    IF pLst THEN
        ExitF(@pLst.tag<>LnLstTag, LibErrH)
        LnLstClear pLst
        MemFree(pLst)
    END IF
END FUNCTION

FUNCTION LnLstValidate(BYVAL pLst AS LnLst PTR) AS LONG
    'True/False if valid handle for this container
    IF pLst AND @pLst.tag = LnLstTag THEN FUNCTION = @pLst.tag
END FUNCTION

SUB LnLstClear(BYVAL pLst AS LnLst PTR)
    'delete all data
    LOCAL node AS LnLstNode PTR
    ExitS(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    WHILE @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    WEND
    @pLst.last = 0
    @pLst.count = 0
END SUB

FUNCTION LnLstCount(BYVAL pLst AS LnLst PTR) AS LONG
    'get item count (number of characters)
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    FUNCTION = @pLst.count
END FUNCTION

SUB LnLstAdd(BYVAL pLst AS LnLst PTR, BYVAL value AS LONG)
    'append Value to end of List
    LOCAL node AS LnLstNode PTR
    ERR = 0
    ExitS(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.count THEN
        node = LnLstNodeAlloc(pLst) : IF ERR THEN EXIT SUB
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    ELSE
        node = LnLstNodeAlloc(pLst) : IF ERR THEN EXIT SUB
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    END IF
END SUB

SUB LnLstIns(BYVAL pLst AS LnLst PTR, BYVAL value AS LONG)
    'insert Value at front of List
    LOCAL node AS LnLstNode PTR
    ERR = 0
    ExitS(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.count THEN
        node = LnLstNodeAlloc(pLst) : IF ERR THEN EXIT SUB
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    ELSE
        node = LnLstNodeAlloc(pLst) : IF ERR THEN EXIT SUB
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    END IF
END SUB

FUNCTION LnLstFirst(BYVAL pLst AS LnLst PTR) AS LONG
    'get handle to first node in List
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    FUNCTION = @pLst.first
END FUNCTION

FUNCTION LnLstLast(BYVAL pLst AS LnLst PTR) AS LONG
    'get handle to last node in List
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    FUNCTION = @pLst.last
END FUNCTION

FUNCTION LnLstNext(BYVAL pNode AS LnLstNode PTR) AS LONG
    'get handle to next node in List
    ExitF(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    FUNCTION = @pNode.next
END FUNCTION

FUNCTION LnLstPrev(BYVAL pNode AS LnLstNode PTR) AS LONG
    'get handle to previous node in List
    ExitF(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    FUNCTION = @pNode.prev
END FUNCTION

FUNCTION LnLstGet(BYVAL pNode AS LnLstNode PTR) AS LONG
    'get node's Value
    ExitF(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    FUNCTION = @pNode.value
END FUNCTION

SUB LnLstSet(BYVAL pNode AS LnLstNode PTR, BYVAL value AS LONG)
    'set node's Value
    ExitS(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    @pNode.value = value
END SUB

SUB LnLstInsPrev(BYVAL pLst AS LnLst PTR, BYVAL pNode AS LnLstNode PTR, BYVAL value AS LONG)
    'insert Value before node
    LOCAL node AS LnLstNode PTR
    ERR = 0
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>LnLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    IF pNode = @pLst.first THEN
        LnLstIns pLst, value
    ELSE
        ExitS(@pNode.prev=0, LibErrU)
        node = LnLstNodeAlloc(pLst) : IF ERR THEN EXIT SUB
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    END IF
END SUB

SUB LnLstInsNext(BYVAL pLst AS LnLst PTR, BYVAL pNode AS LnLstNode PTR, BYVAL value AS LONG)
    'insert Value after Cursor
    LOCAL node AS LnLstNode PTR
    ERR = 0
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>LnLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    IF pNode = @pLst.last THEN
        LnLstAdd pLst, value
    ELSE
        ExitS(@pNode.next=0, LibErrU)
        node = LnLstNodeAlloc(pLst) : IF ERR THEN EXIT SUB
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    END IF
END SUB

SUB LnLstDel(BYVAL pLst AS LnLst PTR, BYVAL pNode AS LnLstNode PTR) PRIVATE
    'remove node from list
    ExitS(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    IF pNode THEN
        ExitS(@pNode.tag<>LnLstNodeTag, LibErrH)
        IF @pLst.first = pNode THEN @pLst.first = @pNode.next
        IF @pLst.last = pNode THEN @pLst.last = @pNode.prev
        IF @pNode.prev THEN @pNode.@prev.next = @pNode.next
        IF @pNode.next THEN @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        DECR @pLst.count
        MemFree(pNode)
    END IF
END SUB

SUB LnLstDelPrev(BYVAL pLst AS LnLst PTR, BYVAL pNode AS LnLstNode PTR)
    'remove node before this node
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>LnLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstDel pLst, @pNode.prev
END SUB

SUB LnLstDelNext(BYVAL pLst AS LnLst PTR, BYVAL pNode AS LnLstNode PTR)
    'remove node after this node
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>LnLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstDel pLst, @pNode.next
END SUB

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

SUB LnLstStkPush(BYVAL pLst AS LnLst PTR, BYVAL value AS LONG)
    'Push Value on Stack
    LnLstAdd pLst, value
END SUB

FUNCTION LnLstStkPeek(BYVAL pLst AS LnLst PTR) AS LONG
    'get top Value on Stack
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.last THEN FUNCTION = @pLst.@last.value
END FUNCTION

FUNCTION LnLstStkPop(BYVAL pLst AS LnLst PTR) AS LONG
    'get and remove top Value on Stack
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.last THEN
        FUNCTION = @pLst.@last.value
        LnLstDel pLst, @pLst.last
    END IF
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

SUB LnLstQuePush(BYVAL pLst AS LnLst PTR, BYVAL value AS LONG)
    'Add Value to end of Queue
    LnLstAdd pLst, value
END SUB

FUNCTION LnLstQuePeek(BYVAL pLst AS LnLst PTR) AS LONG
    'get first Value in Queue
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.first THEN FUNCTION = @pLst.@first.value
END FUNCTION

FUNCTION LnLstQuePop(BYVAL pLst AS LnLst PTR) AS LONG
    'get and remove first Value in Queue
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.first THEN
        FUNCTION = @pLst.@first.value
        LnLstDel pLst, @pLst.first
    END IF
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

SUB LnLstPushFirst(BYVAL pLst AS LnLst PTR, BYVAL value AS LONG)
    'Add Value at front of container
    LnLstIns pLst, value
END SUB

SUB LnLstPushLast(BYVAL pLst AS LnLst PTR, BYVAL value AS LONG)
    'Add Value at end of container
    LnLstAdd pLst, value
END SUB

FUNCTION LnLstPeekFirst(BYVAL pLst AS LnLst PTR) AS LONG
    'get first Value in container
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.first THEN FUNCTION = @pLst.@first.value
END FUNCTION

FUNCTION LnLstPeekLast(BYVAL pLst AS LnLst PTR) AS LONG
    'get last Value in container
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.last THEN FUNCTION = @pLst.@last.value
END FUNCTION

FUNCTION LnLstPopFirst(BYVAL pLst AS LnLst PTR) AS LONG
    'get and remove first Value in container
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.first THEN
        FUNCTION = @pLst.@first.value
        LnLstDel pLst, @pLst.first
    END IF
END FUNCTION

FUNCTION LnLstPopLast(BYVAL pLst AS LnLst PTR) AS LONG
    'get and remove last Value in container
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.last THEN
        FUNCTION = @pLst.@last.value
        LnLstDel pLst, @pLst.last
    END IF
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

FUNCTION LnLstClone(BYVAL pLst AS LnLst PTR) AS LONG
    'returns handle to duplicate container
    LOCAL pClone AS LnLst PTR
    LOCAL node AS LnLstNode PTR
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    ERR = 0
    pClone = LnLstNew() : IF ERR THEN EXIT FUNCTION
    node = @pLst.first
    WHILE node
        LnLstAdd pClone, @node.value
        node = @node.next
    WEND
    FUNCTION = pClone
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

FUNCTION LnLstStore(BYVAL pLst AS LnLst PTR) AS STRING
    'store container to String
    LOCAL s AS STRING
    LOCAL node AS LnLstNode PTR
    LOCAL p AS LONG PTR
    ExitF(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    IF @pLst.count THEN
        s = NUL$(@pLst.count * %SizeLn)
        p = STRPTR(s)
        node = @pLst.first
        WHILE node
            @p = @node.value : INCR p
            node = @node.next
        WEND
    END IF
    FUNCTION = s
END FUNCTION

SUB LnLstRestore(BYVAL pLst AS LnLst PTR, BYREF s AS STRING)
    'restore container from string
    LOCAL items AS LONG
    LOCAL p AS LONG PTR
    ExitS(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    LnLstClear pLst
    IF LEN(s) THEN
        p = STRPTR(s)
        items = LEN(s) / %SizeLn
        WHILE items
            LnLstAdd pLst, @p
            INCR p
            DECR items
        WEND
    END IF
END SUB

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

SUB LnLstFileStore(BYVAL pLst AS LnLst PTR, BYVAL FILE AS STRING)
    'store container to file
    LOCAL s AS STRING
    LOCAL f AS LONG
    ERR = 0
    ExitS(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    s = LnLstStore(pLst) : IF ERR THEN EXIT SUB
    TRY
        f = FREEFILE
        OPEN FILE FOR BINARY AS f
        SETEOF f
        PUT$ f, s
        CLOSE f
    CATCH
        ExitLogErr(LibErrF)
    FINALLY
        IF f THEN CLOSE f
    END TRY
END SUB

SUB LnLstFileRestore(BYVAL pLst AS LnLst PTR, BYVAL FILE AS STRING)
    'restore container from file
    LOCAL f AS LONG
    LOCAL s AS STRING
    ERR = 0
    ExitS(pLst=0 OR @pLst.tag<>LnLstTag, LibErrH)
    TRY
        f = FREEFILE
        OPEN FILE FOR BINARY AS f
        GET$ f, LOF(f), s
        LnLstRestore pLst, s
    CATCH
        ExitLogErr(LibErrF)
    FINALLY
        IF f THEN CLOSE f
    END TRY
END SUB


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

FUNCTION LnLstNodeAlloc(BYVAL pLst AS LnLst PTR) PRIVATE AS LONG
    LOCAL node AS LnLstNode PTR
    node = MemAlloc(SIZEOF(LnLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = LnLstNodeTag
    INCR @pLst.count
    FUNCTION = node
END FUNCTION
