#Include This Once
#Include Once "C:\HLib3\HLib.inc"

Macro SnLstNodeTag = -264589208
Macro SnLstTag = -212996673
Type SnLstNode
    tag As Long
    next As SnLstNode Ptr
    prev As SnLstNode Ptr
    value As Single
End Type
Type SnLst
    tag As Long
    count As Long
    first As SnLstNode Ptr
    last As SnLstNode Ptr
End Type

'++
    '----------------------------------------------------------------------------------------
    '   Single List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = SnLstNew() 'get handle for new container
    '       h = SnLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

Function SnLstNew() As Long
    'allocate new container - return handle
    Local pLst As SnLst Ptr
    pLst = MemAlloc(SizeOf(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = SnLstTag
    Function = pLst
End Function

Function SnLstFinal(ByVal pLst As SnLst Ptr) As Long
    'free allocated container - return null
    If pLst Then
        ExitF(@pLst.tag<>SnLstTag, LibErrH)
        SnLstClear pLst
        MemFree(pLst)
    End If
End Function

Function SnLstValidate(ByVal pLst As SnLst Ptr) As Long
    'True/False if valid handle for this container
    If pLst And @pLst.tag = SnLstTag Then Function = @pLst.tag
End Function

Sub SnLstClear(ByVal pLst As SnLst Ptr)
    'delete all data
    Local node As SnLstNode Ptr
    ExitS(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    While @pLst.first
        node = @pLst.first
        @pLst.first = @node.next
        MemFree(node)
    Wend
    @pLst.last = 0
    @pLst.count = 0
End Sub

Function SnLstCount(ByVal pLst As SnLst Ptr) As Long
    'get item count (number of characters)
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    Function = @pLst.count
End Function

Sub SnLstAdd(ByVal pLst As SnLst Ptr, ByVal value As Single)
    'append Value to end of List
    Local node As SnLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.count Then
        node = SnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.last=0, LibErrU)
        @pLst.@last.next = node
        @node.prev = @pLst.last
        @pLst.last = node
    Else
        node = SnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Sub SnLstIns(ByVal pLst As SnLst Ptr, ByVal value As Single)
    'insert Value at front of List
    Local node As SnLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.count Then
        node = SnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        ExitS(@pLst.first=0, LibErrU)
        @pLst.@first.prev = node
        @node.next = @pLst.first
        @pLst.first = node
    Else
        node = SnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @pLst.first = node
        @pLst.last = node
        @pLst.count = 1
    End If
End Sub

Function SnLstFirst(ByVal pLst As SnLst Ptr) As Long
    'get handle to first node in List
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    Function = @pLst.first
End Function

Function SnLstLast(ByVal pLst As SnLst Ptr) As Long
    'get handle to last node in List
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    Function = @pLst.last
End Function

Function SnLstNext(ByVal pNode As SnLstNode Ptr) As Long
    'get handle to next node in List
    ExitF(pNode=0 Or @pNode.tag<>SnLstNodeTag, LibErrH)
    Function = @pNode.next
End Function

Function SnLstPrev(ByVal pNode As SnLstNode Ptr) As Long
    'get handle to previous node in List
    ExitF(pNode=0 Or @pNode.tag<>SnLstNodeTag, LibErrH)
    Function = @pNode.prev
End Function

Function SnLstGet(ByVal pNode As SnLstNode Ptr) As Single
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>SnLstNodeTag, LibErrH)
    Function = @pNode.value
End Function

Sub SnLstSet(ByVal pNode As SnLstNode Ptr, ByVal value As Single)
    'set node's Value
    ExitS(pNode=0 Or @pNode.tag<>SnLstNodeTag, LibErrH)
    @pNode.value = value
End Sub

Sub SnLstInsPrev(ByVal pLst As SnLst Ptr, ByVal pNode As SnLstNode Ptr, ByVal value As Single)
    'insert Value before node
    Local node As SnLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>SnLstTag Or @pNode.tag<>SnLstNodeTag, LibErrH)
    If pNode = @pLst.first Then
        SnLstIns pLst, value
    Else
        ExitS(@pNode.prev=0, LibErrU)
        node = SnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = pNode
        @node.prev = @pNode.prev
        @pNode.@prev.next = node
        @pNode.prev = node
    End If
End Sub

Sub SnLstInsNext(ByVal pLst As SnLst Ptr, ByVal pNode As SnLstNode Ptr, ByVal value As Single)
    'insert Value after Cursor
    Local node As SnLstNode Ptr
    Err = 0
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>SnLstTag Or @pNode.tag<>SnLstNodeTag, LibErrH)
    If pNode = @pLst.last Then
        SnLstAdd pLst, value
    Else
        ExitS(@pNode.next=0, LibErrU)
        node = SnLstNodeAlloc(pLst) : If Err Then Exit Sub
        @node.value = value
        @node.next = @pNode.next
        @node.prev = pNode
        @pNode.@next.prev = node
        @pNode.next = node
    End If
End Sub

Sub SnLstDel(ByVal pLst As SnLst Ptr, ByVal pNode As SnLstNode Ptr) Private
    'remove node from list
    ExitS(pNode=0 Or @pNode.tag<>SnLstNodeTag, LibErrH)
    If pNode Then
        ExitS(@pNode.tag<>SnLstNodeTag, LibErrH)
        If @pLst.first = pNode Then @pLst.first = @pNode.next
        If @pLst.last = pNode Then @pLst.last = @pNode.prev
        If @pNode.prev Then @pNode.@prev.next = @pNode.next
        If @pNode.next Then @pNode.@next.prev = @pNode.prev
        ExitS(@pLst.count=0, LibErrU)
        Decr @pLst.count
        MemFree(pNode)
    End If
End Sub

Sub SnLstDelPrev(ByVal pLst As SnLst Ptr, ByVal pNode As SnLstNode Ptr)
    'remove node before this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>SnLstTag Or @pNode.tag<>SnLstNodeTag, LibErrH)
    SnLstDel pLst, @pNode.prev
End Sub

Sub SnLstDelNext(ByVal pLst As SnLst Ptr, ByVal pNode As SnLstNode Ptr)
    'remove node after this node
    ExitS(pLst=0 Or pNode=0 Or @pLst.tag<>SnLstTag Or @pNode.tag<>SnLstNodeTag, LibErrH)
    SnLstDel pLst, @pNode.next
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SnLstStkPush(ByVal pLst As SnLst Ptr, ByVal value As Single)
    'Push Value on Stack
    SnLstAdd pLst, value
End Sub

Function SnLstStkPeek(ByVal pLst As SnLst Ptr) As Single
    'get top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function SnLstStkPop(ByVal pLst As SnLst Ptr) As Single
    'get and remove top Value on Stack
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        SnLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

Sub SnLstQuePush(ByVal pLst As SnLst Ptr, ByVal value As Single)
    'Add Value to end of Queue
    SnLstAdd pLst, value
End Sub

Function SnLstQuePeek(ByVal pLst As SnLst Ptr) As Single
    'get first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function SnLstQuePop(ByVal pLst As SnLst Ptr) As Single
    'get and remove first Value in Queue
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        SnLstDel pLst, @pLst.first
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

Sub SnLstPushFirst(ByVal pLst As SnLst Ptr, ByVal value As Single)
    'Add Value at front of container
    SnLstIns pLst, value
End Sub

Sub SnLstPushLast(ByVal pLst As SnLst Ptr, ByVal value As Single)
    'Add Value at end of container
    SnLstAdd pLst, value
End Sub

Function SnLstPeekFirst(ByVal pLst As SnLst Ptr) As Single
    'get first Value in container
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.first Then Function = @pLst.@first.value
End Function

Function SnLstPeekLast(ByVal pLst As SnLst Ptr) As Single
    'get last Value in container
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.last Then Function = @pLst.@last.value
End Function

Function SnLstPopFirst(ByVal pLst As SnLst Ptr) As Single
    'get and remove first Value in container
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.first Then
        Function = @pLst.@first.value
        SnLstDel pLst, @pLst.first
    End If
End Function

Function SnLstPopLast(ByVal pLst As SnLst Ptr) As Single
    'get and remove last Value in container
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.last Then
        Function = @pLst.@last.value
        SnLstDel pLst, @pLst.last
    End If
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SnLstClone(ByVal pLst As SnLst Ptr) As Long
    'returns handle to duplicate container
    Local pClone As SnLst Ptr
    Local node As SnLstNode Ptr
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    Err = 0
    pClone = SnLstNew() : If Err Then Exit Function
    node = @pLst.first
    While node
        SnLstAdd pClone, @node.value
        node = @node.next
    Wend
    Function = pClone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SnLstStore(ByVal pLst As SnLst Ptr) As String
    'store container to String
    Local s As String
    Local node As SnLstNode Ptr
    Local p As Single Ptr
    ExitF(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    If @pLst.count Then
        s = Nul$(@pLst.count * %SizeSn)
        p = StrPtr(s)
        node = @pLst.first
        While node
            @p = @node.value : Incr p
            node = @node.next
        Wend
    End If
    Function = s
End Function

Sub SnLstRestore(ByVal pLst As SnLst Ptr, ByRef s As String)
    'restore container from string
    Local items As Long
    Local p As Single Ptr
    ExitS(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    SnLstClear pLst
    If Len(s) Then
        p = StrPtr(s)
        items = Len(s) / %SizeSn
        While items
            SnLstAdd pLst, @p
            Incr p
            Decr items
        Wend
    End If
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SnLstFileStore(ByVal pLst As SnLst Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    s = SnLstStore(pLst) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SnLstFileRestore(ByVal pLst As SnLst Ptr, ByVal file As String)
    'restore container from file
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pLst=0 Or @pLst.tag<>SnLstTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SnLstRestore pLst, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub


    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Function SnLstNodeAlloc(ByVal pLst As SnLst Ptr) Private As Long
    Local node As SnLstNode Ptr
    node = MemAlloc(SizeOf(SnLstNode))
    ExitF(node=0, LibErrM)
    @node.tag = SnLstNodeTag
    Incr @pLst.count
    Function = node
End Function
