#INCLUDE THIS ONCE
#INCLUDE ONCE "..\HLib3\List\LnLst.inc"
#INCLUDE ONCE "..\HLib3\String\SsStr.inc"

MACRO SsLstTag = 501260071
TYPE SsLst
    tag AS LONG
    lst AS LnLst PTR
END TYPE

'++
    '----------------------------------------------------------------------------------------
    '   String List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = SsLstNew() 'get handle for new container
    '       h = SsLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

FUNCTION SsLstNew() AS LONG
    'allocate new container - return handle
    LOCAL pLst AS SsLst PTR
    ERR = 0
    pLst = MemAlloc(SIZEOF(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = SsLstTag
    @pLst.lst = LnLstNew() : IF ERR THEN EXIT FUNCTION
    FUNCTION = pLst
END FUNCTION

FUNCTION SsLstFinal(BYVAL pLst AS SsLst PTR) AS LONG
    'free allocated container - return null
    IF pLst THEN
        ExitF(@pLst.tag<>SsLstTag, LibErrH)
        SsLstClear pLst
        @pLst.lst = LnLstFinal(@pLst.lst)
        MemFree(pLst)
    END IF
END FUNCTION

FUNCTION SsLstValidate(BYVAL pLst AS SsLst PTR) AS LONG
    'True/False if valid handle for this container
    IF pLst AND @pLst.tag = SsLstTag THEN FUNCTION = @pLst.tag
END FUNCTION

SUB SsLstClear(BYVAL pLst AS SsLst PTR)
    'delete all data
    LOCAL node AS LnLstNode PTR
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    node = @pLst.@lst.first
    WHILE node
        @node.value = SsFinal(@node.value)
        node = @node.next
    WEND
    LnLstClear @pLst.lst
END SUB

FUNCTION SsLstCount(BYVAL pLst AS SsLst PTR) AS LONG
    'get item count (number of characters)
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    FUNCTION = @pLst.@lst.count
END FUNCTION

SUB SsLstAdd(BYVAL pLst AS SsLst PTR, BYREF value AS STRING)
    'append Value to end of List
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    LnLstAdd @pLst.lst, SsSetNew(value)
END SUB

SUB SsLstIns(BYVAL pLst AS SsLst PTR, BYREF value AS STRING)
    'insert Value at front of List
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    LnLstIns @pLst.lst, SsSetNew(value)
END SUB

FUNCTION SsLstFirst(BYVAL pLst AS SsLst PTR) AS LONG
    'get handle to first node in List
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    FUNCTION = @pLst.@lst.first
END FUNCTION

FUNCTION SsLstLast(BYVAL pLst AS SsLst PTR) AS LONG
    'get handle to last node in List
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    FUNCTION = @pLst.@lst.last
END FUNCTION

FUNCTION SsLstNext(BYVAL pNode AS LnLstNode PTR) AS LONG
    'get handle to next node in List
    ExitF(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    FUNCTION = @pNode.next
END FUNCTION

FUNCTION SsLstPrev(BYVAL pNode AS LnLstNode PTR) AS LONG
    'get handle to previous node in List
    ExitF(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    FUNCTION = @pNode.prev
END FUNCTION

FUNCTION SsLstGet(BYVAL pNode AS LnLstNode PTR) AS STRING
    'get node's Value
    ExitF(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    FUNCTION = SsGet(@pNode.value)
END FUNCTION

SUB SsLstSet(BYVAL pNode AS LnLstNode PTR, BYREF value AS STRING)
    'set node's Value
    ExitS(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    SsSet @pNode.value, value
END SUB

SUB SsLstInsPrev(BYVAL pLst AS SsLst PTR, BYVAL pNode AS LnLstNode PTR, BYREF value AS STRING)
    'insert Value before node
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>SsLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstInsPrev @pLst.lst, pNode, SsSetNew(value)
END SUB

SUB SsLstInsNext(BYVAL pLst AS SsLst PTR, BYVAL pNode AS LnLstNode PTR, BYREF value AS STRING)
    'insert Value after Cursor
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>SsLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstInsNext @pLst.lst, pNode, SsSetNew(value)
END SUB

SUB SsLstDel(BYVAL pLst AS SsLst PTR, BYVAL pNode AS LnLstNode PTR) PRIVATE
    'remove node from list
    ExitS(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    IF pNode THEN
        ExitS(@pNode.tag<>LnLstNodeTag, LibErrH)
        SsFinal @pNode.value
        LnLstDel @pLst.lst, pNode
    END IF
END SUB

SUB SsLstDelPrev(BYVAL pLst AS SsLst PTR, BYVAL pNode AS LnLstNode PTR)
    'remove node before this node
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>SsLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    SsLstDel pLst, @pNode.prev
END SUB

SUB SsLstDelNext(BYVAL pLst AS SsLst PTR, BYVAL pNode AS LnLstNode PTR)
    'remove node after this node
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>SsLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    SsLstDel pLst, @pNode.next
END SUB

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

SUB SsLstStkPush(BYVAL pLst AS SsLst PTR, BYREF value AS STRING)
    'Push Value on Stack
    SsLstAdd pLst, value
END SUB

FUNCTION SsLstStkPeek(BYVAL pLst AS SsLst PTR) AS STRING
    'get top Value on Stack
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    IF @pLst.@lst.last THEN FUNCTION = SsGet(@pLst.@lst.@last.value)
END FUNCTION

FUNCTION SsLstStkPop(BYVAL pLst AS SsLst PTR) AS STRING
    'get and remove top Value on Stack
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    IF @pLst.@lst.last THEN
        FUNCTION = SsGet(@pLst.@lst.@last.value)
        SsLstDel pLst, @pLst.@lst.last
    END IF
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

SUB SsLstQuePush(BYVAL pLst AS SsLst PTR, BYREF value AS STRING)
    'Add Value to end of Queue
    SsLstAdd pLst, value
END SUB

FUNCTION SsLstQuePeek(BYVAL pLst AS SsLst PTR) AS STRING
    'get first Value in Queue
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    IF @pLst.@lst.first THEN FUNCTION = SsGet(@pLst.@lst.@first.value)
END FUNCTION

FUNCTION SsLstQuePop(BYVAL pLst AS SsLst PTR) AS STRING
    'get and remove first Value in Queue
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    IF @pLst.@lst.first THEN
        FUNCTION = SsGet(@pLst.@lst.@first.value)
        SsLstDel pLst, @pLst.@lst.first
    END IF
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

SUB SsLstPushFirst(BYVAL pLst AS SsLst PTR, BYREF value AS STRING)
    'Add Value at front of container
    SsLstIns pLst, value
END SUB

SUB SsLstPushLast(BYVAL pLst AS SsLst PTR, BYREF value AS STRING)
    'Add Value at end of container
    SsLstAdd pLst, value
END SUB

FUNCTION SsLstPeekFirst(BYVAL pLst AS SsLst PTR) AS STRING
    'get first Value in container
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    IF @pLst.@lst.first THEN FUNCTION = SsGet(@pLst.@lst.@first.value)
END FUNCTION

FUNCTION SsLstPeekLast(BYVAL pLst AS SsLst PTR) AS STRING
    'get last Value in container
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    IF @pLst.@lst.last THEN FUNCTION = SsGet(@pLst.@lst.@last.value)
END FUNCTION

FUNCTION SsLstPopFirst(BYVAL pLst AS SsLst PTR) AS STRING
    'get and remove first Value in container
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    IF @pLst.@lst.first THEN
        FUNCTION = SsGet(@pLst.@lst.@first.value)
        SsLstDel pLst, @pLst.@lst.first
    END IF
END FUNCTION

FUNCTION SsLstPopLast(BYVAL pLst AS SsLst PTR) AS STRING
    'get and remove last Value in container
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    IF @pLst.@lst.last THEN
        FUNCTION = SsGet(@pLst.@lst.@last.value)
        SsLstDel pLst, @pLst.@lst.last
    END IF
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

FUNCTION SsLstClone(BYVAL pLst AS SsLst PTR) AS LONG
    'returns handle to duplicate container
    LOCAL pClone AS SsLst PTR
    LOCAL node AS LnLstNode PTR
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    ERR = 0
    pClone = SsLstNew() : IF ERR THEN EXIT FUNCTION
    node = @pLst.@lst.first
    WHILE node
        SsLstAdd pClone, SsGet(@node.value)
        node = @node.next
    WEND
    FUNCTION = pClone
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

FUNCTION SsLstStore(BYVAL pLst AS SsLst PTR) AS STRING
    'store container to String
    LOCAL tot, bytes AS LONG
    LOCAL node AS LnLstNode PTR
    LOCAL ps AS SsStr PTR
    LOCAL s AS STRING
    LOCAL pl AS LONG PTR
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    IF @pLst.@lst.count THEN
        tot = 4
        node = @pLst.@lst.first
        WHILE node
            ps = @node.value
            ExitF(ps=0, LibErrN)
            tot += 4 + @ps.count
            node = @node.next
        WEND
        s = NUL$(tot)
        pl = STRPTR(s)
        @pl = @pLst.@lst.count : INCR pl
        node = @pLst.@lst.first
        WHILE node
            ps = @node.value
            bytes = @ps.count
            @pl = bytes : INCR pl
            IF bytes THEN
                MEMORY COPY @ps.mem, pl, bytes : pl += bytes
            END IF
            node = @node.next
        WEND
    END IF
    FUNCTION = s
END FUNCTION

SUB SsLstRestore(BYVAL pLst AS SsLst PTR, BYREF s AS STRING)
    'restore container from string
    LOCAL items, bytes AS LONG
    LOCAL pl AS LONG PTR
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    SsLstClear pLst
    IF LEN(s) THEN
        pl = STRPTR(s)
        items = @pl : INCR pl
        WHILE items
            bytes = @pl : INCR pl
            IF bytes THEN
                SsLstAdd pLst, PEEK$(pl, bytes) : pl += bytes
            ELSE
                SsLstAdd pLst, ""
            END IF
            DECR items
        WEND
    END IF
END SUB

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

SUB SsLstFileStore(BYVAL pLst AS SsLst PTR, BYVAL FILE AS STRING)
    'store container to file
    LOCAL s AS STRING
    LOCAL f AS LONG
    ERR = 0
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    s = SsLstStore(pLst) : IF ERR THEN EXIT SUB
    TRY
        f = FREEFILE
        OPEN FILE FOR BINARY AS f
        SETEOF f
        PUT$ f, s
        CLOSE f
    CATCH
        ExitLogErr(LibErrF)
    FINALLY
        IF f THEN CLOSE f
    END TRY
END SUB

SUB SsLstFileRestore(BYVAL pLst AS SsLst PTR, BYVAL FILE AS STRING)
    'restore container from file
    LOCAL f AS LONG
    LOCAL s AS STRING
    ERR = 0
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    TRY
        f = FREEFILE
        OPEN FILE FOR BINARY AS f
        GET$ f, LOF(f), s
        SsLstRestore pLst, s
    CATCH
        ExitLogErr(LibErrF)
    FINALLY
        IF f THEN CLOSE f
    END TRY
END SUB

'++
    '----------------------------------------------------------------------------------------
    '   String Builder Procedures
    '       use container's other procedures to Add/Insert/Modify/Remove string segments
    '----------------------------------------------------------------------------------------
'--

FUNCTION SsLstBuildStr(BYVAL pLst AS SsLst PTR) AS STRING
    'get complete string containing all string segments in container
    LOCAL tot, hMem AS LONG
    LOCAL node AS LnLstNode PTR
    LOCAL ps AS SsStr PTR
    LOCAL s AS STRING
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    node = @pLst.@lst.first
    WHILE node
        ps = @node.value
        tot += @ps.count
        node = @node.next
    WEND
    IF tot THEN
        s = NUL$(tot)
        hMem = STRPTR(s)
        node = @pLst.@lst.first
        WHILE node
            ps = @node.value
            IF @ps.count THEN
                MEMORY COPY @ps.mem, hMem, @ps.count : hMem += @ps.count
            END IF
            node = @node.next
        WEND
    END IF
    FUNCTION = s
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Text Document Procedures
    '----------------------------------------------------------------------------------------
'--

SUB SsLstTextLoad(BYVAL pLst AS SsLst PTR, BYVAL FILE AS STRING)
    'load text file into container clearing current contents
    LOCAL f  AS LONG
    LOCAL s AS STRING
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    SsLstClear pLst
    TRY
        f = FREEFILE
        OPEN FILE FOR INPUT AS f
        WHILE ISFALSE EOF(f)
            LINE INPUT# f, s
            SsLstAdd pLst, s
        WEND
    CATCH
         ExitLogErr(LibErrF)
    FINALLY
        IF f THEN CLOSE f
    END TRY
END SUB

SUB SsLstTextSave(BYVAL pLst AS SsLst PTR, BYVAL FILE AS STRING)
    'save container to text file overwriting file's current contents
    LOCAL ok, f  AS LONG
    LOCAL node AS LnLstNode PTR
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    TRY
        f = FREEFILE
        OPEN FILE FOR OUTPUT AS f
        node = @pLst.@lst.first
        WHILE node
            PRINT# f, SsGet(@node.value)
            node = @node.next
        WEND
    CATCH
        ExitLogErr(LibErrF)
    FINALLY
        IF f THEN CLOSE f
    END TRY
END SUB

FUNCTION SsLstTextGet(BYVAL pLst AS SsLst PTR) AS STRING
    'get container's contents as a Text Document, appending CrLf to each item
    LOCAL tot AS LONG
    LOCAL node AS LnLstNode PTR
    LOCAL ps AS SsStr PTR
    LOCAL pb AS BYTE PTR
    LOCAL s AS STRING
    ExitF(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    node = @pLst.@lst.first
    WHILE node
        ps = @node.value
        tot += @ps.count + 2
        node = @node.next
    WEND
    IF tot THEN
        s = NUL$(tot)
        pb = STRPTR(s)
        node = @pLst.@lst.first
        WHILE node
            ps = @node.value
            IF @ps.count THEN
                MEMORY COPY @ps.mem, pb, @ps.count : pb += @ps.count
            END IF
            @pb = 13 : INCR pb
            @pb = 10 : INCR pb
            node = @node.next
        WEND
    END IF
    FUNCTION = s
END FUNCTION

SUB SsLstTextSet(BYVAL pLst AS SsLst PTR, BYVAL s AS STRING)
    'replace container's contents with a Text Document - each line = item in container (removing CrLf)
    LOCAL start, x, bytes AS LONG
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    SsLstClear pLst
    IF LEN(s) THEN
        start = 1
        x = INSTR(start, s, $CRLF)
        WHILE x
            bytes = x - start
            SsLstAdd pLst, MID$(s, start, bytes)
            start = x + 2
            x = INSTR(start, s, $CRLF)
        WEND
        IF start < LEN(s) THEN SsLstAdd pLst, MID$(s, start)
    END IF
END SUB

'++
    '----------------------------------------------------------------------------------------
    '   File List Procedures
    '----------------------------------------------------------------------------------------
'--

SUB SsLstGetFiles(BYVAL pLst AS SsLst PTR, BYVAL folder AS STRING, BYVAL mask AS STRING)
    'load container with all files in folder matching the mask
    'loads file names without path
    LOCAL FILE, s AS STRING
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    SsLstClear pLst
    ExitS(ISFALSE ISFOLDER(folder), LibErrA)
    s = RTRIM$(folder, "\") + "\" + mask
    FILE = DIR$(s)
    WHILE LEN(FILE)
        SsLstAdd pLst, FILE
        FILE = DIR$(NEXT)
    WEND
END SUB

SUB SsLstGetPaths(BYVAL pLst AS SsLst PTR, BYVAL folder AS STRING, BYVAL mask AS STRING)
    'load container with all files in folder matching the mask
    'loads full path
    LOCAL FILE, s AS STRING
    ExitS(pLst=0 OR @pLst.tag<>SsLstTag, LibErrH)
    SsLstClear pLst
    ExitS(ISFALSE ISFOLDER(folder), LibErrA)
    folder = RTRIM$(folder, "\") + "\"
    s = folder + mask
    FILE = DIR$(s)
    WHILE LEN(FILE)
        SsLstAdd pLst, folder + FILE
        FILE = DIR$(NEXT)
    WEND
END SUB
