#INCLUDE THIS ONCE
#INCLUDE ONCE "..\HLib3\List\LnLst.inc"
#INCLUDE ONCE "..\HLib3\String\WsStr.inc"

MACRO WsLstTag = -212996673
TYPE WsLst
    tag AS LONG
    lst AS LnLst PTR
END TYPE

'++
    '----------------------------------------------------------------------------------------
    '   WString List Container Procedures
    '       container accessed with handle
    '       handle protected by hash tag
    '       h = WsLstNew() 'get handle for new container
    '       h = WsLstFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--

FUNCTION WsLstNew() AS LONG
    'allocate new container - return handle
    LOCAL pLst AS WsLst PTR
    ERR = 0
    pLst = MemAlloc(SIZEOF(@pLst))
    ExitF(pLst=0, LibErrM)
    @pLst.tag = WsLstTag
    @pLst.lst = LnLstNew() : IF ERR THEN EXIT FUNCTION
    FUNCTION = pLst
END FUNCTION

FUNCTION WsLstFinal(BYVAL pLst AS WsLst PTR) AS LONG
    'free allocated container - return null
    IF pLst THEN
        ExitF(@pLst.tag<>WsLstTag, LibErrH)
        WsLstClear pLst
        @pLst.lst = LnLstFinal(@pLst.lst)
        MemFree(pLst)
    END IF
END FUNCTION

FUNCTION WsLstValidate(BYVAL pLst AS WsLst PTR) AS LONG
    'True/False if valid handle for this container
    IF pLst AND @pLst.tag = WsLstTag THEN FUNCTION = @pLst.tag
END FUNCTION

SUB WsLstClear(BYVAL pLst AS WsLst PTR)
    'delete all data
    LOCAL node AS LnLstNode PTR
    ExitS(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    node = @pLst.@lst.first
    WHILE node
        @node.value = WsFinal(@node.value)
        node = @node.next
    WEND
    LnLstClear @pLst.lst
END SUB

FUNCTION WsLstCount(BYVAL pLst AS WsLst PTR) AS LONG
    'get item count (number of characters)
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    FUNCTION = @pLst.@lst.count
END FUNCTION

SUB WsLstAdd(BYVAL pLst AS WsLst PTR, BYREF value AS WSTRING)
    'append Value to end of List
    ExitS(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    LnLstAdd @pLst.lst, WsSetNew(value)
END SUB

SUB WsLstIns(BYVAL pLst AS WsLst PTR, BYREF value AS WSTRING)
    'insert Value at front of List
    ExitS(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    LnLstIns @pLst.lst, WsSetNew(value)
END SUB

FUNCTION WsLstFirst(BYVAL pLst AS WsLst PTR) AS LONG
    'get handle to first node in List
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    FUNCTION = @pLst.@lst.first
END FUNCTION

FUNCTION WsLstLast(BYVAL pLst AS WsLst PTR) AS LONG
    'get handle to last node in List
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    FUNCTION = @pLst.@lst.last
END FUNCTION

FUNCTION WsLstNext(BYVAL pNode AS LnLstNode PTR) AS LONG
    'get handle to next node in List
    ExitF(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    FUNCTION = @pNode.next
END FUNCTION

FUNCTION WsLstPrev(BYVAL pNode AS LnLstNode PTR) AS LONG
    'get handle to previous node in List
    ExitF(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    FUNCTION = @pNode.prev
END FUNCTION

FUNCTION WsLstGet(BYVAL pNode AS LnLstNode PTR) AS WSTRING
    'get node's Value
    ExitF(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    FUNCTION = WsGet(@pNode.value)
END FUNCTION

SUB WsLstSet(BYVAL pNode AS LnLstNode PTR, BYREF value AS WSTRING)
    'set node's Value
    ExitS(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    WsSet @pNode.value, value
END SUB

SUB WsLstInsPrev(BYVAL pLst AS WsLst PTR, BYVAL pNode AS LnLstNode PTR, BYREF value AS WSTRING)
    'insert Value before node
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>WsLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstInsPrev @pLst.lst, pNode, WsSetNew(value)
END SUB

SUB WsLstInsNext(BYVAL pLst AS WsLst PTR, BYVAL pNode AS LnLstNode PTR, BYREF value AS WSTRING)
    'insert Value after Cursor
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>WsLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    LnLstInsNext @pLst.lst, pNode, WsSetNew(value)
END SUB

SUB WsLstDel(BYVAL pLst AS WsLst PTR, BYVAL pNode AS LnLstNode PTR) PRIVATE
    'remove node from list
    ExitS(pNode=0 OR @pNode.tag<>LnLstNodeTag, LibErrH)
    IF pNode THEN
        ExitS(@pNode.tag<>LnLstNodeTag, LibErrH)
        WsFinal @pNode.value
        LnLstDel @pLst.lst, pNode
    END IF
END SUB

SUB WsLstDelPrev(BYVAL pLst AS WsLst PTR, BYVAL pNode AS LnLstNode PTR)
    'remove node before this node
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>WsLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    WsLstDel pLst, @pNode.prev
END SUB

SUB WsLstDelNext(BYVAL pLst AS WsLst PTR, BYVAL pNode AS LnLstNode PTR)
    'remove node after this node
    ExitS(pLst=0 OR pNode=0 OR @pLst.tag<>WsLstTag OR @pNode.tag<>LnLstNodeTag, LibErrH)
    WsLstDel pLst, @pNode.next
END SUB

'++
    '----------------------------------------------------------------------------------------
    '   Stack Procedures
    '----------------------------------------------------------------------------------------
'--

SUB WsLstStkPush(BYVAL pLst AS WsLst PTR, BYREF value AS WSTRING)
    'Push Value on Stack
    WsLstAdd pLst, value
END SUB

FUNCTION WsLstStkPeek(BYVAL pLst AS WsLst PTR) AS WSTRING
    'get top Value on Stack
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    IF @pLst.@lst.last THEN FUNCTION = WsGet(@pLst.@lst.@last.value)
END FUNCTION

FUNCTION WsLstStkPop(BYVAL pLst AS WsLst PTR) AS WSTRING
    'get and remove top Value on Stack
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    IF @pLst.@lst.last THEN
        FUNCTION = WsGet(@pLst.@lst.@last.value)
        WsLstDel pLst, @pLst.@lst.last
    END IF
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Queue Procedures
    '----------------------------------------------------------------------------------------
'--

SUB WsLstQuePush(BYVAL pLst AS WsLst PTR, BYREF value AS WSTRING)
    'Add Value to end of Queue
    WsLstAdd pLst, value
END SUB

FUNCTION WsLstQuePeek(BYVAL pLst AS WsLst PTR) AS WSTRING
    'get first Value in Queue
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    IF @pLst.@lst.first THEN FUNCTION = WsGet(@pLst.@lst.@first.value)
END FUNCTION

FUNCTION WsLstQuePop(BYVAL pLst AS WsLst PTR) AS WSTRING
    'get and remove first Value in Queue
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    IF @pLst.@lst.first THEN
        FUNCTION = WsGet(@pLst.@lst.@first.value)
        WsLstDel pLst, @pLst.@lst.first
    END IF
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Deque Procedures (double-ended Queue)
    '----------------------------------------------------------------------------------------
'--

SUB WsLstPushFirst(BYVAL pLst AS WsLst PTR, BYREF value AS WSTRING)
    'Add Value at front of container
    WsLstIns pLst, value
END SUB

SUB WsLstPushLast(BYVAL pLst AS WsLst PTR, BYREF value AS WSTRING)
    'Add Value at end of container
    WsLstAdd pLst, value
END SUB

FUNCTION WsLstPeekFirst(BYVAL pLst AS WsLst PTR) AS WSTRING
    'get first Value in container
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    IF @pLst.@lst.first THEN FUNCTION = WsGet(@pLst.@lst.@first.value)
END FUNCTION

FUNCTION WsLstPeekLast(BYVAL pLst AS WsLst PTR) AS WSTRING
    'get last Value in container
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    IF @pLst.@lst.last THEN FUNCTION = WsGet(@pLst.@lst.@last.value)
END FUNCTION

FUNCTION WsLstPopFirst(BYVAL pLst AS WsLst PTR) AS WSTRING
    'get and remove first Value in container
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    IF @pLst.@lst.first THEN
        FUNCTION = WsGet(@pLst.@lst.@first.value)
        WsLstDel pLst, @pLst.@lst.first
    END IF
END FUNCTION

FUNCTION WsLstPopLast(BYVAL pLst AS WsLst PTR) AS WSTRING
    'get and remove last Value in container
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    IF @pLst.@lst.last THEN
        FUNCTION = WsGet(@pLst.@lst.@last.value)
        WsLstDel pLst, @pLst.@lst.last
    END IF
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

FUNCTION WsLstClone(BYVAL pLst AS WsLst PTR) AS LONG
    'returns handle to duplicate container
    LOCAL pClone AS WsLst PTR
    LOCAL node AS LnLstNode PTR
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    ERR = 0
    pClone = WsLstNew() : IF ERR THEN EXIT FUNCTION
    node = @pLst.@lst.first
    WHILE node
        WsLstAdd pClone, WsGet(@node.value)
        node = @node.next
    WEND
    FUNCTION = pClone
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

FUNCTION WsLstStore(BYVAL pLst AS WsLst PTR) AS STRING
    'store container to String
    LOCAL tot, bytes AS LONG
    LOCAL node AS LnLstNode PTR
    LOCAL ps AS WsStr PTR
    LOCAL s AS STRING
    LOCAL pl AS LONG PTR
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    IF @pLst.@lst.count THEN
        tot = 4
        node = @pLst.@lst.first
        WHILE node
            ps = @node.value
            ExitF(ps=0, LibErrN)
            tot += 4 + (@ps.count * 2)
            node = @node.next
        WEND
        s = NUL$(tot)
        pl = STRPTR(s)
        @pl = @pLst.@lst.count : INCR pl
        node = @pLst.@lst.first
        WHILE node
            ps = @node.value
            bytes = @ps.count * 2
            @pl = bytes : INCR pl
            IF bytes THEN
                MEMORY COPY @ps.mem, pl, bytes : pl += bytes
            END IF
            node = @node.next
        WEND
    END IF
    FUNCTION = s
END FUNCTION

SUB WsLstRestore(BYVAL pLst AS WsLst PTR, BYREF s AS STRING)
    'restore container from string
    LOCAL items, bytes AS LONG
    LOCAL pl AS LONG PTR
    ExitS(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    WsLstClear pLst
    IF LEN(s) THEN
        pl = STRPTR(s)
        items = @pl : INCR pl
        WHILE items
            bytes = @pl : INCR pl
            IF bytes THEN
                WsLstAdd pLst, PEEK$$(pl, bytes / 2) : pl += bytes
            ELSE
                WsLstAdd pLst, ""
            END IF
            DECR items
        WEND
    END IF
END SUB

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

SUB WsLstFileStore(BYVAL pLst AS WsLst PTR, BYVAL FILE AS STRING)
    'store container to file
    LOCAL s AS STRING
    LOCAL f AS LONG
    ERR = 0
    ExitS(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    s = WsLstStore(pLst) : IF ERR THEN EXIT SUB
    TRY
        f = FREEFILE
        OPEN FILE FOR BINARY AS f
        SETEOF f
        PUT$ f, s
        CLOSE f
    CATCH
        ExitLogErr(LibErrF)
    FINALLY
        IF f THEN CLOSE f
    END TRY
END SUB

SUB WsLstFileRestore(BYVAL pLst AS WsLst PTR, BYVAL FILE AS STRING)
    'restore container from file
    LOCAL f AS LONG
    LOCAL s AS STRING
    ERR = 0
    ExitS(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    TRY
        f = FREEFILE
        OPEN FILE FOR BINARY AS f
        GET$ f, LOF(f), s
        WsLstRestore pLst, s
    CATCH
        ExitLogErr(LibErrF)
    FINALLY
        IF f THEN CLOSE f
    END TRY
END SUB

'++
    '----------------------------------------------------------------------------------------
    '   WString Builder Procedures
    '       use container's other procedures to Add/Insert/Modify/Remove string segments
    '----------------------------------------------------------------------------------------
'--

FUNCTION WsLstBuildWStr(BYVAL pLst AS WsLst PTR) AS WSTRING
    'get complete string containing all string segments in container
    LOCAL tot, hMem AS LONG
    LOCAL node AS LnLstNode PTR
    LOCAL ps AS WsStr PTR
    LOCAL s AS WSTRING
    ExitF(pLst=0 OR @pLst.tag<>WsLstTag, LibErrH)
    node = @pLst.@lst.first
    WHILE node
        ps = @node.value
        tot += @ps.count * 2
        node = @node.next
    WEND
    IF tot THEN
        s = NUL$(tot / 2)
        hMem = STRPTR(s)
        node = @pLst.@lst.first
        WHILE node
            ps = @node.value
            IF @ps.count THEN
                MEMORY COPY @ps.mem, hMem, @ps.count * 2: hMem += (@ps.count * 2)
            END IF
            node = @node.next
        WEND
    END IF
    FUNCTION = s
END FUNCTION
