#INCLUDE THIS ONCE
#INCLUDE ONCE "..\HLib3\HLib.inc"
#INCLUDE ONCE "..\HLib3\List\SsLst.inc"
#INCLUDE ONCE "..\HLib3\List\WsLst.inc"
#INCLUDE ONCE "..\HLib3\List\LnLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'WString/Long ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - no limit on Key length
    '   - use WsLnTreComparison() to change how Keys compared
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = WsLnTreNew() 'get handle for new container
    '   h = WsLnTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


MACRO WsLnTreNodeTag = 1320870834
MACRO WsLnTreTag = -212996673
TYPE WsLnTreNode
    tag AS LONG
    P AS WsLnTreNode PTR
    L AS WsLnTreNode PTR
    R AS WsLnTreNode PTR
    HL AS WORD
    HR AS WORD
    K AS LONG
    V AS LONG
END TYPE
TYPE WsLnTre
    tag AS LONG
    count AS LONG
    root AS WsLnTreNode PTR
    compareCB AS LONG
    collation AS WsStr PTR
END TYPE

FUNCTION WsLnTreNew() AS LONG
    'allocate new container - return handle
    LOCAL p AS WsLnTre PTR
    ERR = 0
    p = MemAlloc(SIZEOF(@p))
    ExitF(p=0, LibErrM)
    @p.tag = WsLnTreTag
    @p.compareCB = CODEPTR(WsCompare)
    @p.collation = WsNew() : IF ERR THEN EXIT FUNCTION
    FUNCTION = p
END FUNCTION

FUNCTION WsLnTreFinal(BYVAL pTree AS WsLnTre PTR) AS LONG
    'free allocated container - return null
    IF pTree THEN
        ExitF(@pTree.tag<>WsLnTreTag, LibErrH)
        @pTree.collation = WsFinal(@pTree.collation)
        WsLnTreClear pTree
        MemFree(pTree)
    END IF
END FUNCTION

FUNCTION WsLnTreValidate(BYVAL pTree AS WsLnTre PTR) AS LONG
    'True/False if valid handle for this container
    IF pTree AND @pTree.tag = WsLnTreTag THEN FUNCTION = @pTree.tag
END FUNCTION

SUB WsLnTreComparison(BYVAL pTree AS WsLnTre PTR, BYVAL compareUCase AS LONG, BYVAL collationSequence AS WSTRING)
    'set how WStrings compared
    'default = case ignored
    'if collationSequence WString provided then
    '   WStrings are compared using the order of the collation sequence WString
    '   collation WString must be 65536 characters
    'else if compareUCase = True then
    '   WStrings compared UCase
    ExitS(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    @pTree.compareCB = CODEPTR(WsCompare)
    WsClear @pTree.collation
    IF LEN(collationSequence) THEN
        ExitS(LEN(collationSequence)<>65536, LibErrS)
        WsSet @pTree.collation, collationSequence : IF ERR THEN EXIT SUB
        @pTree.compareCB = CODEPTR(WsCompareCollate)
    ELSEIF compareUCase THEN
        @pTree.compareCB = CODEPTR(WsCompareUCase)
    END IF
END SUB

SUB WsLnTreClear(BYVAL pTree AS WsLnTre PTR)
    'delete all data
    LOCAL i AS LONG
    LOCAL pNode AS WsLnTreNode PTR
    LOCAL nodes() AS LONG
    ExitS(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    IF @pTree.count THEN
        REDIM nodes(1 TO @pTree.count)
        i = 0
        pNode = WsLnTreFirst(pTree)
        WHILE pNode
            INCR i
            nodes(i) = pNode
            pNode = WsLnTreNext(pNode)
        WEND
        FOR i = 1 TO @pTree.count
            WsLnTreFreeNode(pTree, nodes(i))
        NEXT i
    END IF
    @pTree.count = 0
    @pTree.root = 0
END SUB

FUNCTION WsLnTreCount(BYVAL pTree AS WsLnTre PTR) AS LONG
    'get item count
    IF pTree THEN FUNCTION = @pTree.count
END FUNCTION

SUB WsLnTreSet(BYVAL pTree AS WsLnTre PTR, BYREF key AS WSTRING, BYVAL value AS LONG, OPT BYVAL DontReplace AS LONG)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    LOCAL compare, temp AS LONG
    LOCAL n AS WsLnTreNode PTR
    ERR = 0
    ExitS(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    temp = WsSetNew(key) : IF ERR THEN EXIT SUB
    IF @pTree.root THEN
        n = @pTree.root
        WHILE 1
            CALL DWORD @pTree.compareCB USING WsCompareCB(temp, @n.K, @pTree.collation) TO compare
            IF compare > 0 THEN
                IF @n.R THEN
                    n = @n.R
                ELSE
                    @n.R = WsLnTreAllocNode(pTree) : IF ERR THEN EXIT SUB
                    @n.@R.P = n
                    WsSet @n.@R.K, key
                    @n.@R.V = value
                    WsLnTreBalanceBranch pTree, n
                    EXIT LOOP
                END IF
            ELSEIF compare < 0 THEN
                IF @n.L THEN
                    n = @n.L
                ELSE
                    @n.L = WsLnTreAllocNode(pTree) : IF ERR THEN EXIT SUB
                    @n.@L.P = n
                    WsSet @n.@L.K, key
                    @n.@L.V = value
                    WsLnTreBalanceBranch pTree, n
                    EXIT LOOP
                END IF
            ELSE
                IF ISFALSE DontReplace THEN @n.V = value
                EXIT LOOP
            END IF
        WEND
    ELSE
        @pTree.root = WsLnTreAllocNode(pTree) : IF ERR THEN EXIT SUB
        WsSet @pTree.@root.K, key
        @pTree.@root.V = value
        @pTree.count = 1
    END IF
    temp = WsFinal(temp)
END SUB

FUNCTION WsLnTreGet(BYVAL pTree AS WsLnTre PTR, BYREF key AS WSTRING) AS LONG
    'get Key's associated Value
    LOCAL compare, temp AS LONG
    LOCAL n AS WsLnTreNode PTR
    ERR = 0
    ExitF(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    temp = WsSetNew(key) : IF ERR THEN EXIT FUNCTION
    n = @pTree.root
    WHILE n
        CALL DWORD @pTree.compareCB USING WsCompareCB(temp, @n.K, @pTree.collation) TO compare
        IF compare < 0 THEN
            n = @n.L
        ELSEIF compare > 0 THEN
            n = @n.R
        ELSE
            FUNCTION = @n.V
            EXIT LOOP
        END IF
    WEND
    temp = WsFinal(temp)
END FUNCTION

FUNCTION WsLnTreGot(BYVAL pTree AS WsLnTre PTR, BYREF key AS WSTRING) AS LONG
    'True/False if Key exist
    LOCAL compare, temp AS LONG
    LOCAL n AS WsLnTreNode PTR
    ERR = 0
    ExitF(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    temp = WsSetNew(key) : IF ERR THEN EXIT FUNCTION
    n = @pTree.root
    WHILE n
        CALL DWORD @pTree.compareCB USING WsCompareCB(temp, @n.K, @pTree.collation) TO compare
        IF compare < 0 THEN
            n = @n.L
        ELSEIF compare > 0 THEN
            n = @n.R
        ELSE
            FUNCTION = n
            EXIT LOOP
        END IF
    WEND
    temp = WsFinal(temp)
END FUNCTION

SUB WsLnTreDel(BYVAL pTree AS WsLnTre PTR, BYREF key AS WSTRING)
    'remove Key and associated Value
    LOCAL pNode AS WsLnTreNode PTR
    ExitS(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    pNode = WsLnTreGot(pTree, key)
    IF pNode THEN
        WsLnTreRemoveNode(pTree, pNode)
    END IF
END SUB

FUNCTION WsLnTreFirst(BYVAL pTree AS WsLnTre PTR) AS LONG
    'get handle to first node in tree
    LOCAL n AS WsLnTreNode PTR
    ExitF(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    n = @pTree.root
    IF n THEN
        WHILE @n.L
            n = @n.L
        WEND
    END IF
    FUNCTION = n
END FUNCTION

FUNCTION WsLnTreLast(BYVAL pTree AS WsLnTre PTR) AS LONG
    'get handle to last node in tree
    LOCAL n AS WsLnTreNode PTR
    ExitF(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    n = @pTree.root
    IF n THEN
        WHILE @n.R
            n = @n.R
        WEND
    END IF
    FUNCTION = n
END FUNCTION

FUNCTION WsLnTreNext(BYVAL pNode AS WsLnTreNode PTR) AS LONG
    'get handle to next node in tree
    LOCAL minR AS WsLnTreNode PTR
    IF pNode THEN
        ExitF(@pNode.tag<>WsLnTreNodeTag, LibErrH)
        minR = WsLnTreMinRight(pNode)
        IF pNode <> minR THEN
            FUNCTION = minR
        ELSE
            FUNCTION = WsLnTreParentGreater(pNode)
        END IF
    END IF
END FUNCTION

FUNCTION WsLnTrePrev(BYVAL pNode AS WsLnTreNode PTR) AS LONG
    'get handle to previous node in tree
    LOCAL maxL AS WsLnTreNode PTR
    IF pNode THEN
        ExitF(@pNode.tag<>WsLnTreNodeTag, LibErrH)
        maxL = WsLnTreMaxLeft(pNode)
        IF pNode <> maxL THEN
            FUNCTION = maxL
        ELSE
            FUNCTION = WsLnTreParentLesser(pNode)
        END IF
    END IF
END FUNCTION

FUNCTION WsLnTreGetKey(BYVAL pNode AS WsLnTreNode PTR) AS WSTRING
    'get node's Key
    ExitF(pNode=0 OR @pNode.tag<>WsLnTreNodeTag, LibErrH)
    FUNCTION = WsGet(@pNode.K)
END FUNCTION

FUNCTION WsLnTreGetVal(BYVAL pNode AS WsLnTreNode PTR) AS LONG
    'get node's Value
    ExitF(pNode=0 OR @pNode.tag<>WsLnTreNodeTag, LibErrH)
    FUNCTION = @pNode.V
END FUNCTION

SUB WsLnTreSetVal(BYVAL pNode AS WsLnTreNode PTR, BYVAL value AS LONG)
    'get node's Value
    ExitS(pNode=0 OR @pNode.tag<>WsLnTreNodeTag, LibErrH)
    @pNode.V = value
END SUB

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

FUNCTION WsLnTreClone(BYVAL pTree AS WsLnTre PTR) AS LONG
    'create duplicate container
    LOCAL h, clone AS LONG
    ERR = 0
    ExitF(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    clone = WsLnTreNew() : IF ERR THEN EXIT FUNCTION
    h = WsLnTreFirst(pTree)
    WHILE h
        WsLnTreSet clone, WsLnTreGetKey(h), WsLnTreGetVal(h)
        h = WsLnTreNext(h)
    WEND
    FUNCTION = clone
END FUNCTION

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

FUNCTION WsLnTreStore(BYVAL pTree AS WsLnTre PTR) AS STRING
    'store container to string
    LOCAL h, keys, vals, stor AS LONG
    ERR = 0
    ExitF(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    keys = WsLstNew() : IF ERR THEN EXIT FUNCTION
    vals = LnLstNew() : IF ERR THEN EXIT FUNCTION
    stor = SsLstNew() : IF ERR THEN EXIT FUNCTION
    IF @pTree.count THEN
        h = WsLnTreFirst(pTree)
        WHILE h
            WsLstAdd keys, WsLnTreGetKey(h)
            LnLstAdd vals, WsLnTreGetVal(h)
            h = WsLnTreNext(h)
        WEND
        SsLstAdd stor, WsLstStore(keys)
        SsLstAdd stor, LnLstStore(vals)
        FUNCTION = SsLstStore(stor)
    END IF
    keys = WsLstFinal(keys)
    vals = LnLstFinal(vals)
    stor = SsLstFinal(stor)
END FUNCTION

SUB WsLnTreRestore(BYVAL pTree AS WsLnTre PTR, BYVAL s AS STRING)
    'restore container from string
    LOCAL keys, vals, stor AS LONG
    ERR = 0
    ExitS(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    WsLnTreClear pTree
    keys = WsLstNew() : IF ERR THEN EXIT SUB
    vals = LnLstNew() : IF ERR THEN EXIT SUB
    stor = SsLstNew() : IF ERR THEN EXIT SUB
    IF LEN(s) THEN
        SsLstRestore stor, s : IF ERR THEN EXIT SUB
        ExitS(SsLstCount(stor)<>2, LibErrU)
        WsLstRestore keys, SsLstPopFirst(stor)
        LnLstRestore vals, SsLstPopFirst(stor)
        ExitS(WsLstCount(keys)<>LnLstCount(vals), LibErrU)
        WHILE WsLstCount(keys)
            WsLnTreSet pTree, WsLstPopFirst(keys), LnLstPopFirst(vals)
        WEND
    END IF
    keys = WsLstFinal(keys)
    vals = LnLstFinal(vals)
    stor = SsLstFinal(stor)
END SUB


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

SUB WsLnTreFileStore(BYVAL pTree AS WsLnTre PTR, BYVAL FILE AS STRING)
    'store container to file
    LOCAL s AS STRING
    LOCAL f AS LONG
    ERR = 0
    ExitS(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    s = WsLnTreStore(pTree) : IF ERR THEN EXIT SUB
    TRY
        f = FREEFILE
        OPEN FILE FOR BINARY AS f
        SETEOF f
        PUT$ f, s
        CLOSE f
    CATCH
        ExitLogErr(LibErrF)
    FINALLY
        IF f THEN CLOSE f
    END TRY
END SUB

SUB WsLnTreFileRestore(BYVAL pTree AS WsLnTre PTR, BYVAL FILE AS STRING)
    'restore container from file - Modifies Container Data
    LOCAL f AS LONG
    LOCAL s AS STRING
    ERR = 0
    ExitS(pTree=0 OR @pTree.tag<>WsLnTreTag, LibErrH)
    TRY
        f = FREEFILE
        OPEN FILE FOR BINARY AS f
        GET$ f, LOF(f), s
        WsLnTreRestore pTree, s
    CATCH
        ExitLogErr(LibErrF)
    FINALLY
        IF f THEN CLOSE f
    END TRY
END SUB

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

'========================================================
' SUB: WsLnTreRemoveNode (Corrected Data-Swapping Approach)
'========================================================
' Note: Use this with the corrected WsLnTreBalanceBranch (Attempt 4)
SUB WsLnTreRemoveNode(BYVAL p AS WsLnTre PTR, BYVAL n AS WsLnTreNode PTR) PRIVATE
    ExitS(n=0, LibErrP) ' Exit if node handle is NULL
    LOCAL nodeToDelete AS WsLnTreNode PTR ' The node whose original position needs deletion after data swap
    LOCAL nP AS WsLnTreNode PTR         ' Parent of nodeToDelete
    LOCAL replacementNode AS WsLnTreNode PTR ' Child node that replaces nodeToDelete (can be NULL)
    LOCAL nodeToBalanceFrom AS WsLnTreNode PTR ' Where to start balancing

    nodeToDelete = n ' Start with the node passed in

    ' --- Step 1: If node has TWO children, find successor, swap data, target successor for deletion ---
    IF @nodeToDelete.L AND @nodeToDelete.R THEN
        ' Node has two children. Find inorder successor (minimum node in right subtree).
        LOCAL successor AS WsLnTreNode PTR
        successor = WsLnTreMinRight(nodeToDelete) ' Find the successor node

        ' Swap Key handles (K) and Values (V) between nodeToDelete and successor
        SWAP @nodeToDelete.K, @successor.K
        SWAP @nodeToDelete.V, @successor.V

        ' Now, the node we *actually* need to remove physically is the successor node
        ' from its original position. It's guaranteed to have 0 or 1 right child.
        nodeToDelete = successor
    END IF

    ' --- Step 2: Now 'nodeToDelete' has 0 or 1 child. Remove it physically. ---

    ' Determine the single child (if any) that will replace the nodeToDelete
    IF @nodeToDelete.L THEN
        replacementNode = @nodeToDelete.L
    ELSE
        replacementNode = @nodeToDelete.R ' Can be NULL if no children
    END IF

    ' Get the parent of the node we are about to physically remove
    nP = @nodeToDelete.P
    nodeToBalanceFrom = nP ' Balance starts from here

    ' --- Link replacement node to parent (nP) ---
    IF nP = %NULL THEN
        ' Deleting the root node
        @p.root = replacementNode ' New root is the replacement (or NULL if tree empty)
        IF replacementNode THEN @replacementNode.P = %NULL ' Update replacement's parent
    ELSE
        ' Deleting a non-root node
        IF @nP.L = nodeToDelete THEN
            @nP.L = replacementNode ' Link parent's left to replacement
        ELSE
            @nP.R = replacementNode ' Link parent's right to replacement
        END IF
        IF replacementNode THEN @replacementNode.P = nP ' Update replacement's parent
    END IF

    ' --- Free the node that was physically removed ---
    nodeToDelete = WsLnTreFreeNode(p, nodeToDelete) ' Frees memory, decrements count

    ' --- Rebalance starting from the parent of the removed node's position ---
    IF nodeToBalanceFrom THEN ' Only balance if there was a parent
        WsLnTreBalanceBranch p, nodeToBalanceFrom
    END IF

END SUB


FUNCTION WsLnTreFreeNode(BYVAL p AS WsLnTre PTR, BYVAL n AS WsLnTreNode PTR) AS LONG
    IF n THEN
        @n.K = WsFinal(@n.K)
        MemFree(n)
        IF @p.count = 0 THEN
            ERR = LibErrU
            FUNCTION = %FALSE
            EXIT FUNCTION
        END IF
        DECR @p.count
        FUNCTION = %TRUE
    END IF
END FUNCTION




FUNCTION WsLnTreAllocNode(BYVAL p AS WsLnTre PTR) PRIVATE AS LONG
    LOCAL n AS WsLnTreNode PTR
    n = MemAlloc(SIZEOF(WsLnTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = WsLnTreNodeTag
    @n.K = WsNew() : IF ERR THEN EXIT FUNCTION
    INCR @p.count
    FUNCTION = n
END FUNCTION


'========================================================
' SUB: WsLnTreBalanceBranch (Attempt 4 - PB Syntax Fixes)
'========================================================
SUB WsLnTreBalanceBranch(BYVAL p AS WsLnTre PTR, BYVAL n AS WsLnTreNode PTR) PRIVATE
    LOCAL rotatedNodeHandle AS LONG       ' Handle of the new root after rotation
    LOCAL nodeToProcessCurrent AS WsLnTreNode PTR ' Node being processed in this iteration
    LOCAL nodeToProcessNext AS WsLnTreNode PTR  ' Node to process in the *next* iteration
    LOCAL tempRotatedNodePtr AS WsLnTreNode PTR ' Temporary pointer for casting handle

    nodeToProcessNext = n ' Start balancing from the specified node

    WHILE nodeToProcessNext ' Loop upwards as long as we have a valid node handle
        nodeToProcessCurrent = nodeToProcessNext ' Assign node for this iteration

        ' --- Store Parent for the NEXT iteration EARLY ---
        ' This is crucial because rotations might change the current node's parent pointer.
        IF @nodeToProcessCurrent.P THEN
            nodeToProcessNext = @nodeToProcessCurrent.P ' Store its parent handle
        ELSE
            nodeToProcessNext = %NULL ' Reached the root's parent (NULL)
        END IF

        ' --- Recalculate heights for the current node ---
        @nodeToProcessCurrent.HL = IIF&(@nodeToProcessCurrent.L, MAX&(@nodeToProcessCurrent.@L.HL, @nodeToProcessCurrent.@L.HR) + 1, 1)
        @nodeToProcessCurrent.HR = IIF&(@nodeToProcessCurrent.R, MAX&(@nodeToProcessCurrent.@R.HL, @nodeToProcessCurrent.@R.HR) + 1, 1)

        ' --- Check for imbalance and perform rotation if needed ---
        IF @nodeToProcessCurrent.HL > @nodeToProcessCurrent.HR + 1 THEN
            ' Rotate Right needed
            rotatedNodeHandle = WsLnTreRotateRight(p, nodeToProcessCurrent)
            ' After rotation, the original 'nodeToProcessCurrent' might have moved.
            ' The 'rotatedNodeHandle' points to the NEW root of this balanced subtree.
            ' The loop should continue from the PARENT of whatever node is now at the top
            ' of the subtree where the rotation occurred.
            IF rotatedNodeHandle <> 0 THEN ' Check if rotation actually returned a valid handle
                 tempRotatedNodePtr = rotatedNodeHandle ' Assign LONG handle to PTR
                 IF @tempRotatedNodePtr.P THEN          ' Does the new root have a parent?
                     nodeToProcessNext = @tempRotatedNodePtr.P ' Continue from new root's parent
                 ELSE
                     nodeToProcessNext = %NULL          ' New root became the tree root
                 END IF
            ELSE
                 ' Rotation function failed or returned NULL unexpectedly
                 nodeToProcessNext = %NULL ' Stop balancing
            END IF
            ' *** NO ITERATE needed here *** - Let WHILE condition check nodeToProcessNext

        ELSEIF @nodeToProcessCurrent.HR > @nodeToProcessCurrent.HL + 1 THEN
            ' Rotate Left needed
            rotatedNodeHandle = WsLnTreRotateLeft(p, nodeToProcessCurrent)
             ' Similar logic as above
            IF rotatedNodeHandle <> 0 THEN
                 tempRotatedNodePtr = rotatedNodeHandle ' Assign LONG handle to PTR
                 IF @tempRotatedNodePtr.P THEN
                     nodeToProcessNext = @tempRotatedNodePtr.P ' Continue from new root's parent
                 ELSE
                     nodeToProcessNext = %NULL
                 END IF
            ELSE
                 nodeToProcessNext = %NULL ' Stop balancing
            END IF
             ' *** NO ITERATE needed here ***

        ELSE
            ' Node is balanced at this level.
            ' The loop will continue with the parent stored in 'nodeToProcessNext'
            ' which was captured at the beginning of this iteration.
            ' *** NO ITERATE needed here ***
        END IF

    WEND ' Loop continues with nodeToProcessNext
END SUB

FUNCTION WsLnTreMaxLeft(BYVAL n AS WsLnTreNode PTR) PRIVATE AS LONG
    IF n THEN
        IF @n.L THEN
            n = @n.L
            WHILE @n.R
                n = @n.R
            WEND
        END IF
    END IF
    FUNCTION = n
END FUNCTION

FUNCTION WsLnTreMinRight(BYVAL n AS WsLnTreNode PTR) PRIVATE AS LONG
    IF n THEN
        IF @n.R THEN
            n = @n.R
            WHILE @n.L
                n = @n.L
            WEND
        END IF
    END IF
    FUNCTION = n
END FUNCTION

FUNCTION WsLnTreParentGreater(BYVAL n AS WsLnTreNode PTR) PRIVATE AS LONG
    IF n THEN
        WHILE @n.P
            IF @n.@P.L = n THEN
                FUNCTION = @n.P
                EXIT FUNCTION
            ELSE
                n = @n.P
            END IF
        WEND
    END IF
END FUNCTION

FUNCTION WsLnTreParentLesser(BYVAL n AS WsLnTreNode PTR) PRIVATE AS LONG
    IF n THEN
        WHILE @n.P
            IF @n.@P.R = n THEN
                FUNCTION = @n.P
                EXIT FUNCTION
            ELSE
                n = @n.P
            END IF
        WEND
    END IF
END FUNCTION
             '========================================================
' FUNCTION: WsLnTreRotateLeft (Corrected Return Value)
'========================================================
FUNCTION WsLnTreRotateLeft(BYVAL p AS WsLnTre PTR, BYVAL n AS WsLnTreNode PTR) PRIVATE AS LONG
    LOCAL nR AS WsLnTreNode PTR     ' Right child of n
    LOCAL nRL AS WsLnTreNode PTR    ' Right child's Left child (for double rotation)
    LOCAL newSubtreeRoot AS LONG  ' Handle to the node that becomes root of this subtree

    nR = @n.R ' Get right child

    ' Check for Right-Left Case (Double Rotation needed)
    IF @nR.HL > @nR.HR THEN
        ' Perform Rotation Right on nR first
        nRL = @nR.L
        @n.R = nRL      ' n's right child becomes nRL
        @nRL.P = n      ' nRL's parent becomes n
        @nR.L = @nRL.R  ' nR's left child becomes nR's old right grandchild
        IF @nR.L THEN @nR.@L.P = nR ' Update parent of that grandchild
        @nRL.R = nR      ' nRL's right child becomes nR
        @nR.P = nRL      ' nR's parent becomes nRL

        ' Recalculate heights for nR after its rotation
        @nR.HL = IIF&(@nR.L, MAX&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIF&(@nR.R, MAX&(@nR.@R.HL, @nR.@R.HR) + 1, 1)

        ' nR has been updated locally, update pointer for main rotation
        nR = @n.R ' nR now points to nRL (the new right child of n)
    END IF

    ' Perform Rotation Left on n
    newSubtreeRoot = nR  ' nR (or nRL if double) will be the new root

    ' Update tree root if n was the root
    IF @p.root = n THEN @p.root = nR

    ' Re-parent nR
    @nR.P = @n.P ' nR takes n's old parent
    IF @nR.P THEN ' Make old parent point to nR
        IF @nR.@P.L = n THEN @nR.@P.L = nR ELSE @nR.@P.R = nR
    END IF

    ' Adjust child pointers
    @n.R = @nR.L        ' n adopts nR's left child as its right child
    IF @n.R THEN @n.@R.P = n ' Update parent of adopted child

    @nR.L = n           ' n becomes the left child of nR
    @n.P = nR           ' n's parent becomes nR

    ' *** Recalculate heights AFTER structure changes
    ' Height of n (which moved down)
    @n.HL = IIF&(@n.L, MAX&(@n.@L.HL, @n.@L.HR) + 1, 1)
    @n.HR = IIF&(@n.R, MAX&(@n.@R.HL, @n.@R.HR) + 1, 1)
    ' Height of nR (the new root of this subtree)
    @nR.HL = IIF&(@nR.L, MAX&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
    @nR.HR = IIF&(@nR.R, MAX&(@nR.@R.HL, @nR.@R.HR) + 1, 1)


    FUNCTION = newSubtreeRoot ' *** Corrected: Return the NEW root of this subtree ***
END FUNCTION

'========================================================
' FUNCTION: WsLnTreRotateRight (Corrected Return Value)
'========================================================
FUNCTION WsLnTreRotateRight(BYVAL p AS WsLnTre PTR, BYVAL n AS WsLnTreNode PTR) PRIVATE AS LONG
    LOCAL nL AS WsLnTreNode PTR     ' Left child of n
    LOCAL nLR AS WsLnTreNode PTR    ' Left child's Right child (for double rotation)
    LOCAL newSubtreeRoot AS LONG  ' Handle to the node that becomes root of this subtree

    nL = @n.L ' Get left child

    ' Check for Left-Right Case (Double Rotation needed)
    IF @nL.HR > @nL.HL THEN
        ' Perform Rotation Left on nL first
        nLR = @nL.R
        @n.L = nLR       ' n's left child becomes nLR
        @nLR.P = n       ' nLR's parent becomes n
        @nL.R = @nLR.L   ' nL's right child becomes nLR's old left grandchild
        IF @nL.R THEN @nL.@R.P = nL ' Update parent of that grandchild
        @nLR.L = nL       ' nLR's left child becomes nL
        @nL.P = nLR       ' nL's parent becomes nLR

        ' Recalculate heights for nL after its rotation
        @nL.HL = IIF&(@nL.L, MAX&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIF&(@nL.R, MAX&(@nL.@R.HL, @nL.@R.HR) + 1, 1)

        ' nL has been updated locally, update pointer for main rotation
        nL = @n.L ' nL now points to nLR (the new left child of n)
    END IF

    ' Perform Rotation Right on n
    newSubtreeRoot = nL ' nL (or nLR if double) will be the new root

    ' Update tree root if n was the root
    IF @p.root = n THEN @p.root = nL

    ' Re-parent nL
    @nL.P = @n.P ' nL takes n's old parent
    IF @nL.P THEN ' Make old parent point to nL
        IF @nL.@P.L = n THEN @nL.@P.L = nL ELSE @nL.@P.R = nL
    END IF

     ' Adjust child pointers
    @n.L = @nL.R        ' n adopts nL's right child as its left child
    IF @n.L THEN @n.@L.P = n ' Update parent of adopted child

    @nL.R = n           ' n becomes the right child of nL
    @n.P = nL           ' n's parent becomes nL


    ' *** Recalculate heights AFTER structure changes
     ' Height of n (which moved down)
    @n.HL = IIF&(@n.L, MAX&(@n.@L.HL, @n.@L.HR) + 1, 1)
    @n.HR = IIF&(@n.R, MAX&(@n.@R.HL, @n.@R.HR) + 1, 1)
    ' Height of nL (the new root of this subtree)
    @nL.HL = IIF&(@nL.L, MAX&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
    @nL.HR = IIF&(@nL.R, MAX&(@nL.@R.HL, @nL.@R.HR) + 1, 1)


    FUNCTION = newSubtreeRoot ' *** Corrected: Return the NEW root of this subtree ***
END FUNCTION
