#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\List\ExLst.inc"
#Include Once "C:\HLib3\List\SsLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'Long/Extended ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = LnExTreNew() 'get handle for new container
    '   h = LnExTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro LnExTreNodeTag = 513674861
Macro LnExTreTag = 719713421
Type LnExTreNode
    tag As Long
    P As LnExTreNode Ptr
    L As LnExTreNode Ptr
    R As LnExTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As Extended
End Type
Type LnExTre
    tag As Long
    count As Long
    root As LnExTreNode Ptr
End Type

Function LnExTreNew() As Long
    'allocate new container - return handle
    Local p As LnExTre Ptr
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = LnExTreTag
    Function = p
End Function

Function LnExTreFinal(ByVal pTree As LnExTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>LnExTreTag, LibErrH)
        LnExTreClear pTree
        MemFree(pTree)
    End If
End Function

Function LnExTreValidate(ByVal pTree As LnExTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = LnExTreTag Then Function = @pTree.tag
End Function

Sub LnExTreClear(ByVal pTree As LnExTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As LnExTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = LnExTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = LnExTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            LnExTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function LnExTreCount(ByVal pTree As LnExTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub LnExTreSet(ByVal pTree As LnExTre Ptr, ByVal key As Long, ByVal value As Extended, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local n As LnExTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    If @pTree.root Then
        n = @pTree.root
        While 1
            If key > @n.K Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = LnExTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    @n.@R.K = key
                    @n.@R.V = value
                    LnExTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf key < @n.K Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = LnExTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    @n.@L.K = key
                    @n.@L.V = value
                    LnExTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then @n.V = value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = LnExTreAllocNode(pTree) : If Err Then Exit Sub
        @pTree.@root.K = key
        @pTree.@root.V = value
        @pTree.count = 1
    End If
End Sub

Function LnExTreGet(ByVal pTree As LnExTre Ptr, ByVal key As Long) As Extended
    'get Key's associated Value
    Register k As Long : k = key
    Local n As LnExTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = @n.V
            Exit Loop
        End If
    Wend
End Function

Function LnExTreGot(ByVal pTree As LnExTre Ptr, ByVal key As Long) As Long
    'True/False if Key exist
    Register k As Long : k = key
    Local n As LnExTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
End Function

Sub LnExTreDel(ByVal pTree As LnExTre Ptr, ByVal key As Long)
    'remove Key and associated Value
    Local pNode As LnExTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    pNode = LnExTreGot(pTree, key)
    If pNode Then
        LnExTreRemoveNode(pTree, pNode)
    End If
End Sub

Function LnExTreFirst(ByVal pTree As LnExTre Ptr) As Long
    'get handle to first node in tree
    Local n As LnExTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function LnExTreLast(ByVal pTree As LnExTre Ptr) As Long
    'get handle to last node in tree
    Local n As LnExTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function LnExTreNext(ByVal pNode As LnExTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As LnExTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnExTreNodeTag, LibErrH)
        minR = LnExTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = LnExTreParentGreater(pNode)
        End If
    End If
End Function

Function LnExTrePrev(ByVal pNode As LnExTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As LnExTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnExTreNodeTag, LibErrH)
        maxL = LnExTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = LnExTreParentLesser(pNode)
        End If
    End If
End Function

Function LnExTreGetKey(ByVal pNode As LnExTreNode Ptr) As Long
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>LnExTreNodeTag, LibErrH)
    Function = @pNode.K
End Function

Function LnExTreGetVal(ByVal pNode As LnExTreNode Ptr) As Extended
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>LnExTreNodeTag, LibErrH)
    Function = @pNode.V
End Function

Sub LnExTreSetVal(ByVal pNode As LnExTreNode Ptr, ByVal value As Extended)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>LnExTreNodeTag, LibErrH)
    @pNode.V = value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnExTreClone(ByVal pTree As LnExTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    clone = LnExTreNew() : If Err Then Exit Function
    h = LnExTreFirst(pTree)
    While h
        LnExTreSet clone, LnExTreGetKey(h), LnExTreGetVal(h)
        h = LnExTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnExTreStore(ByVal pTree As LnExTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    keys = LnLstNew() : If Err Then Exit Function
    vals = ExLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = LnExTreFirst(pTree)
        While h
            LnLstAdd keys, LnExTreGetKey(h)
            ExLstAdd vals, LnExTreGetVal(h)
            h = LnExTreNext(h)
        Wend
        SsLstAdd stor, LnLstStore(keys)
        SsLstAdd stor, ExLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = LnLstFinal(keys)
    vals = ExLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub LnExTreRestore(ByVal pTree As LnExTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    LnExTreClear pTree
    keys = LnLstNew() : If Err Then Exit Sub
    vals = ExLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        LnLstRestore keys, SsLstPopFirst(stor)
        ExLstRestore vals, SsLstPopFirst(stor)
        ExitS(LnLstCount(keys)<>ExLstCount(vals), LibErrU)
        While LnLstCount(keys)
            LnExTreSet pTree, LnLstPopFirst(keys), ExLstPopFirst(vals)
        Wend
    End If
    keys = LnLstFinal(keys)
    vals = ExLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnExTreFileStore(ByVal pTree As LnExTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    s = LnExTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnExTreFileRestore(ByVal pTree As LnExTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnExTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnExTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub LnExTreRemoveNode(ByVal p As LnExTre Ptr, ByVal n As LnExTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As LnExTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, LnExTreMaxLeft(n), LnExTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        LnExTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = LnExTreFreeNode(p, n)
        LnExTreBalanceBranch p, nP
    End If
End Sub

Function LnExTreAllocNode(ByVal p As LnExTre Ptr) Private As Long
    Local n As LnExTreNode Ptr
    n = MemAlloc(SizeOf(LnExTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = LnExTreNodeTag
    Incr @p.count
    Function = n
End Function

Function LnExTreFreeNode(ByVal p As LnExTre Ptr, ByVal n As LnExTreNode Ptr) As Long
    If n Then
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub LnExTreBalanceBranch(ByVal p As LnExTre Ptr, ByVal n As LnExTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = LnExTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = LnExTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function LnExTreMaxLeft(ByVal n As LnExTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function LnExTreMinRight(ByVal n As LnExTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function LnExTreParentGreater(ByVal n As LnExTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnExTreParentLesser(ByVal n As LnExTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnExTreRotateLeft(ByVal p As LnExTre Ptr, ByVal n As LnExTreNode Ptr) Private As Long
    Local nR, nRL As LnExTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function LnExTreRotateRight(ByVal p As LnExTre Ptr, ByVal n As LnExTreNode Ptr) Private As Long
    Local nL, nLR As LnExTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
